﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Microsoft.VisualStudio.TestTools.UnitTesting;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Settings;
using System;
using System.IO;
using System.Linq;
using System.Reflection;

namespace NintendoWare.Spy.Tests
{
    // NOTE:
    // SettingsService は SpyFoundation で定義されていますが、
    // Spy.exe アセンブリのクラスにアクセスする必要があるので
    // SpyFoundation.Tests ではなく Spy.Tests プロジェクトでテストします。
    [TestClass()]
    public class SettingsServiceTests
    {
        /// <summary>
        /// 古い設定ファイルを読めるかのテストです。
        /// </summary>
        [TestMethod()]
        public void LoadOldTest()
        {
            var settingsService = new SettingsService();

            using (var stream = new FileStream("Resources/raw/Spy.settings.old", FileMode.Open))
            {
                settingsService.Load(stream);
                Assert.IsFalse(settingsService.Items.IsEmpty());
            }
        }

        /// <summary>
        /// 保存と読込が正常に機能しているかのテストです。
        /// 適当な設定を保存し、読み込んでみて、内容が一致しているかを調べます。
        /// </summary>
        [TestMethod()]
        public void SaveAndLoadTest()
        {
            var memoryStream = new MemoryStream(0);

            var settingsService1 = new SettingsService();
            settingsService1.GetApplicationSettings().OutputDirectoryPath = "a";
            settingsService1.GetApplicationSettings().PlaybackDeviceName = "b";
            settingsService1.GetApplicationSettings().TimeUnit = SpyTimeUnit.AudioFrame;
            settingsService1.GetApplicationSettings().UpdateInterval = TimeSpan.FromSeconds(10);
            settingsService1.GetCafeAudioSettings().FinalOutDrcWaveVisibility = WaveVisibility.Surround4ch;
            settingsService1.GetCafeAudioSettings().FinalOutTvWaveVisibility = WaveVisibility.Surround6ch;
            settingsService1.GetCafeAudioSettings().LoudnessDrcWaveVisibility = WaveVisibility.Surround4ch;
            settingsService1.GetCafeAudioSettings().LoudnessTvWaveVisibility = WaveVisibility.Surround6ch;
            settingsService1.GetConnectionSettings().IsConnecting = true;
            settingsService1.GetConnectionSettings().TargetPlatformName = "c";
            settingsService1.GetWindowsSettings().MainWindowSettings.PanelLayouts = "d";
            settingsService1.GetWindowsSettings().MainWindowSettings.Position = new WindowPosition() { Top = 3, Left = 4, Width = 5, Height = 6 };

            settingsService1.Save(memoryStream);

            var settingsService2 = new SettingsService();

            settingsService2.Load(new MemoryStream(memoryStream.ToArray()));

            var srcValues = settingsService1.Items.Select(it => it.Value.Value);
            var copyValues = settingsService2.Items.Select(it => it.Value.Value);

            Assert.IsTrue(
                srcValues
                .Zip(copyValues, (src, copy) => DeepCompare(src, copy))
                .All(it => it == true));
        }

        /// <summary>
        /// オブジェクトのプロパティが一致しているか再帰的に確認します。
        /// </summary>
        private bool DeepCompare(object a, object b, int nest = 0)
        {
            Enumerable.Range(0, nest).ForEach(_ => Console.Write("  "));
            Console.WriteLine($"{a} {b}");

            if (nest > 10)
            {
                return true;
            }

            if (object.Equals(a, b))
            {
                return true;
            }

            if (a == null || b == null)
            {
                return false;
            }

            if (a.GetType() != b.GetType())
            {
                return false;
            }

            var properties = a.GetType().GetProperties(BindingFlags.Public | BindingFlags.Instance | BindingFlags.GetProperty);
            foreach (var prop in properties)
            {
                var aValue = prop.GetValue(a);
                var bValue = prop.GetValue(b);

                if (!DeepCompare(aValue, bValue, nest + 1))
                {
                    return false;
                }
            }

            return true;
        }
    }
}
