﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 */

#include <iostream>
#include <iomanip>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <sstream>
#include <nn/repair/repair_Api.h>
#include <nn/repair/repair_LabelButton.h>
#include <nn/repair/repair_ShutdownButton.h>
#include <nn/repair/repair_StreamTextView.h>
#include <nn/repair/repair_Sdcard.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/repair/repair_NetworkSettingsParser.h>
// #include <nn/manu/manu_Api.h>

// ペアレンタルコントロールサービス利用
#include <nn/pctl.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiWatcher.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>

// デバイス情報取得
#include <nn/util/util_BitFlagSet.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/time/time_SteadyClockTimePoint.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_Api.h>

// 署名・暗号化
#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Signer.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Verifier.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_ProtectedFileEncryptor.h>
#include <nn/repair/repair_Authentication.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/spl/spl_Api.h>
// #include "repair_DevKeyPair1.h"
// #include "repair_DevKeyPair3.h"
// #include "repair_ProdKeyPair1.h"
// #include "repair_ProdKeyPair3.h"
// #include "repair_DevPubKey2.h"
// #include "repair_DevPubKey4.h"
// #include "repair_ProdPubKey2.h"
// #include "repair_ProdPubKey4.h"
#endif


#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#if defined( NN_SDK_BUILD_LIBRARY )
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#else
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#endif

//#define GLV_ENABLE_LOG_DEBUG

//!--------------------------------------------------------------------------------------
//! @brief SDKライブラリビルドでのアサート/ログマクロ制限.
//!--------------------------------------------------------------------------------------
#if defined( NN_SDK_BUILD_LIBRARY )
#define NN_ASSERT NN_SDK_ASSERT
#define GLV_LOG(...) NN_SDK_LOG( "[GLV] " __VA_ARGS__ )
#else
#define GLV_LOG(...) NN_LOG( "[GLV] " __VA_ARGS__ )
#endif // defined( NN_SDK_BUILD_LIBRARY )


namespace {
    const char* HostMountDir         = "d:/port/repair";
    const char* ToolName             = "NX Pre-repair check tool";
    const int   ToolMajorVersion     = 1;
    const int   ToolMinorVersion     = 1;

    const char* DispMsgPCtlEnable    = "Parental controls state\t\tEnabled";
    const char* DispMsgPCtlDisable   = "Parental controls state\t\tDisabled";

    const char* DispMsgButtonPut0    = "B Req"; // generate Request to Server for Backup
    const char* DispMsgButtonGet0    = "B Get"; // Recieve from server response of Backup
    const char* DispMsgButtonPut1    = "R Req"; // generate Request to Server for Restore
    const char* DispMsgButtonGet1    = "R Get"; // Recieve from server response of Restore

    static nn::repair::StreamTextView*    s_pTextView;
    static glv::Label*        s_isPctlValidLabel[2];
    static std::shared_ptr<nn::repair::IAuthenticationKeySource> m_pKeySource = nullptr;
    static nn::repair::Id128  m_sessionId;
    static nn::repair::AuthenticationArchiveContent m_authenticatedContent; // 欲しいもの

    static const size_t AppletHeapSize = 256 * 1024 * 1024;                 //!< アプリケーション予約ヒープメモリサイズ
    static const size_t AppletAllocatableSize = 256 * 1024 * 1024;          //!< アプリケーション稼働ヒープ上限メモリサイズ
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024; //!< NVNグラフィクス稼働予約メモリ領域サイズ
#endif

#if defined(NN_BUILD_TARGET_PLATFORM_NX)

    void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return aligned_alloc(alignment, size);
    }

    void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        free(addr);
    }

    void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }
#endif

    //!--------------------------------------------------------------------------------------
    //! @brief GLV用ユーザ定義アロケータ.
    //!--------------------------------------------------------------------------------------
    void* glvMemoryAllocator( const size_t size, const size_t beginAlignment ) NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        return ::_aligned_malloc( size, beginAlignment );
#else
        return ::aligned_alloc( beginAlignment, size );
#endif
    }

    //!--------------------------------------------------------------------------------------
    //! @brief GLV用ユーザ定義デアロケータ.
    //!--------------------------------------------------------------------------------------
    void glvMemoryDeallocator( void* address ) NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        ::_aligned_free( address );
#else
        ::free( address );
#endif
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ペリフェラルセットアップ
    //!--------------------------------------------------------------------------------------
    static void SetupPeripherals() NN_NOEXCEPT
    {
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        // this memory allocation will be used from the nvn graphics systems at runtime.
        nv::SetGraphicsAllocator( NvAllocate, NvFree, NvReallocate, nullptr );
        nv::InitializeGraphics( ::malloc( GraphicsSystemReservedMemorySize ), GraphicsSystemReservedMemorySize );

#if NN_GFX_IS_TARGET_NVN
        // this memory allocation interface will be used when compiling of shader code at runtime.
        glslcSetAllocator( NvAllocate, NvFree, NvReallocate, nullptr );
#endif
#endif
    }

    //!--------------------------------------------------------------------------------------
    //! @brief HID設定初期値
    //!--------------------------------------------------------------------------------------
    static const glv::HidInitialConfiguration LocalHidConfiguration = glv::HidInitialConfiguration( glv::HidInitialConfiguration::PadAssetAssignRule_BasicPadPrimary );

} // namespace

//!--------------------------------------------------------------------------------------
//! nninitStartup() is invoked before calling nnMain().
//! 256MB確保
//!--------------------------------------------------------------------------------------
NN_OS_EXTERN_C void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::SetMemoryHeapSize( AppletHeapSize ) );
    uintptr_t address;
    const size_t MallocMemorySize = AppletAllocatableSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::AllocateMemoryBlock( &address, MallocMemorySize ) );
    nn::init::InitializeAllocator( reinterpret_cast< void* >( address ), MallocMemorySize );

    // GLV内部で利用されるルートメモリアロケータを上書きします.
    // 上書きしない場合は C++11 の std::aligned_alloc 及び std::free が利用されます.
    glv::detail::ApplicationMemoryAllocator::AttachUserAllocator( glvMemoryAllocator, glvMemoryDeallocator );
}

// メイン処理(send)
void GenerateRequest(bool isBackup) NN_NOEXCEPT
{
    ::nn::Result result;

    if(isBackup)
    {
        s_pTextView->AppendValue("=== Generate Backup Request Data ===\n");
    }
    else
    {
        s_pTextView->AppendValue("=== Generate Restore Request Data ===\n");
    }

    // ファイル準備
    char filepath[128];
    nn::util::TSNPrintf(filepath, sizeof(filepath), "host:/%s", nn::os::GetHostArgv()[1]);

    result = nn::fs::CreateFile(filepath, sizeof(nn::repair::BackupRequestMessage));
    if( nn::fs::ResultPathNotFound::Includes(result) )
    {
        // パスに含まれるディレクトリが存在しません。
        NN_LOG("Parent path is not found.\n");
        return;
    }
    else if( nn::fs::ResultPathAlreadyExists::Includes(result)
             || nn::fs::ResultTargetLocked::Includes(result) )
    {
        // 対象ファイルが既に存在しています。
    }
    else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
    {
        // ストレージに空き領域が不足しています。
        // ホスト側ストレージの空き領域を増やしてください。
        NN_ASSERT("Usable space not enough.\n");
        return;
    }

    NN_LOG("=== Parameter Check ===\n");

    // コマンドライン引数の確認
    // 一つ目：サーバへのリクエストデータ（出力用）
    // 二つ目：サーバからのレスポンスデータ（入力用）
    if (nn::os::GetHostArgc() != 3)
    {
        NN_LOG("\t Invalid parameter\n");
        return;
    }

    // バックアップデータ暗号化鍵の生成(spl::initialize 必要)
    // inputKey = spl 暗号化されたデータ暗号化鍵(バックアップリストア間でガチで守る対象)
    nn::repair::Key128 inputKey;
    {
        std::shared_ptr<nn::repair::IProtectedFileEncryptor> encryptor;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::repair::CreateProtectedFileEncryptor(&encryptor, "Spl"));
        NN_ABORT_UNLESS_RESULT_SUCCESS(encryptor->GenerateEncryptedKey(&inputKey));
    }

    // 認証

    //     result = nn::repair::CreateSplAuthenticationKeySource(&m_keySource);
    //     if ( !result.IsSuccess() )
    //     {
    //         s_pTextView->AppendValue("\t CreateSplAuthenticationKeySource failed\n");
    //         return;
    //     }

    nn::repair::RepairAuthentication auth;
    auth.Initialize(m_pKeySource);

    // made session id
    result = auth.MakeSessionId(&m_sessionId);
    if ( !result.IsSuccess() )
    {
        NN_LOG("\t MakeSessionId failed\n");
        return;
    }

    std::shared_ptr<nn::repair::BackupRequestMessage> backup_request(new nn::repair::BackupRequestMessage());
    nn::repair::RestoreRequestMessage restore_request;
    void *pData = nullptr;
    size_t datasize = 0;

    if( isBackup )
    {
        // struct BackupRequestMessage
        // {
        //     Sign2048 sign;
        //     RsaEncryptedBlock sessionId;
        //     RsaEncryptedBlock sessionKey;
        // };
        pData = backup_request.get();
        datasize = sizeof(nn::repair::BackupRequestMessage);

        std::memset(pData, 0, datasize);

        // 1. session id の暗号化
        // 2. session Key の暗号化
        // 3. 署名の付与
        result = auth.MakeBackupRequestMessage(backup_request.get(), m_sessionId, inputKey);

        if ( !result.IsSuccess() )
        {
            s_pTextView->AppendValue("\t MakeBackupRequestMessage failed\n");
            return;
        }
    }
    else
    {
        // struct RestoreRequestMessage
        // {
        //     Sign2048 sign;
        //     InitialVector iv;
        //     RsaEncryptedBlock sessionId;
        //     RsaEncryptedBlock sessionKey;
        // };


        result = auth.MakeRestoreRequestMessage(&restore_request, m_sessionId, m_authenticatedContent);
        if ( !result.IsSuccess() )
        {
            s_pTextView->AppendValue("\t MakeRestoreRequestMessage failed\n");
            return;
        }
        pData = (void *)&restore_request;
        datasize = sizeof(nn::repair::RestoreRequestMessage);
    }

    NN_LOG("=== write request message ===\n");
    // result = nn::manu::WriteToHost((const void *)(&(*request)), sizeof(*request), nn::os::GetHostArgv()[1], 0, sizeof(*request));
    {
        nn::fs::FileHandle fileHandle;
        result = nn::fs::OpenFile(&fileHandle, filepath, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            // 対象ファイルが存在しません。
            NN_ASSERT("Target file is not found.\n");
        }
        else if( nn::fs::ResultTargetLocked::Includes(result) )
        {
            // 対象ファイルが既にオープンされています。
            NN_ASSERT("Target path is already opened.\n");
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::WriteFile(fileHandle, 0, pData, datasize, nn::fs::WriteOption()));

        // device id を付加
        int DEVICE_ID_HEX_SIZE = 16;
        char dev_id_str[nn::repair::DeviceIdLength];
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::repair::GetDeviceIdHex(dev_id_str));

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::WriteFile(fileHandle, datasize, dev_id_str, DEVICE_ID_HEX_SIZE, nn::fs::WriteOption()));

        // フラッシュに失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::FlushFile(fileHandle));

        nn::fs::CloseFile(fileHandle);
    }

    s_pTextView->AppendValue("\t done.\n");
    // サーバ送信用 request data 作成完了
}

void MyMouseDownHandlerPut0() NN_NOEXCEPT
{
    GenerateRequest(true);
}

void MyMouseDownHandlerPut1() NN_NOEXCEPT
{
    GenerateRequest(false);
}

// メイン処理(recv)
void checkResponse(bool isBackup) NN_NOEXCEPT
{
    ::nn::Result result;


    nn::repair::AuthenticationArchiveContent * pOut = &m_authenticatedContent;

    void *pData = nullptr;
    size_t datasize = 0;
    std::shared_ptr<nn::repair::BackupResponseMessage> backup_response(new nn::repair::BackupResponseMessage());
    nn::repair::RestoreResponseMessage response = {};

    if(isBackup)
    {
        s_pTextView->AppendValue("=== Load backup response message. ===\n");
        pData = backup_response.get();
        datasize = sizeof(nn::repair::BackupResponseMessage);

        std::memset(pData, 0, datasize);
    }
    else
    {
        s_pTextView->AppendValue("=== Load restore response message. ===\n");
        pData = &response;
        datasize = sizeof(nn::repair::RestoreResponseMessage);

    }

    // 指定サイズになってなければ return
    {
        char filepath[128];
        nn::fs::FileHandle fileHandle;
        nn::util::TSNPrintf(filepath, sizeof(filepath), "host:/%s", nn::os::GetHostArgv()[2]);

        result = nn::fs::OpenFile(&fileHandle, filepath, nn::fs::OpenMode_Read);
        if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            // 対象ファイルが存在しません。
            NN_ASSERT("Target file is not found.\n");
        }
        else if( nn::fs::ResultTargetLocked::Includes(result) )
        {
            // 対象ファイルが既にオープンされています。
            NN_ASSERT("Target path is already opened.\n");
        }

        int64_t fileSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::GetFileSize(&fileSize, fileHandle));

        // auto result = nn::manu::GetFileSize(&fileSize, nn::os::GetHostArgv()[2]);
        // result = nn::manu::ReadFromHost(response.get(), sizeof(*response.get()), nn::os::GetHostArgv()[2], 0, sizeof(*response.get()));
        if (fileSize != datasize)
        {
            NN_LOG(" fileSize = %d / %d byte \n", fileSize , datasize );
            nn::fs::CloseFile(fileHandle);
            return;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::ReadFile(fileHandle, 0, pData , datasize ));

        nn::fs::CloseFile(fileHandle);
    }

    nn::repair::RepairAuthentication auth;
    auth.Initialize(m_pKeySource);

    if(isBackup)
    {
        // pOut のデータは restore request で使用する
        result = auth.LoadBackupResponseMessage(pOut, *backup_response, m_sessionId);

        if ( !result.IsSuccess() )
        {
            s_pTextView->AppendValue("\t LoadBackupResponseMessage failed\n");
            return;
        }
    }
    else
    {
        nn::repair::Key128 key;

        // key は このテストでは特に使わない
        result = auth.LoadRestoreResponseMessage(&key, response, m_sessionId);

        if ( !result.IsSuccess() )
        {
            s_pTextView->AppendValue("\t LoadRestoreResponseMessage failed\n");
            return;
        }

    }

    s_pTextView->AppendValue("[[ authrization succeeded! ]]\n");
}

void MyMouseDownHandlerGet0() NN_NOEXCEPT
{
    checkResponse(true);
}

void MyMouseDownHandlerGet1() NN_NOEXCEPT
{
    checkResponse(false);
}

//!--------------------------------------------------------------------------------------
//! @brief ルートサーフェイスコンテキスト
//!--------------------------------------------------------------------------------------
class RootSurfaceContext : public glv::Window, public glv::GLV, public glv::ApplicationLoopCallback
{
private:

public:
    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ.
    //! @details 指定の幅、高さの領域に対して、ページセレクトヘッダバーとページサーフェイスを追加します.
    //!--------------------------------------------------------------------------------------
    RootSurfaceContext( const unsigned width, const unsigned height ) NN_NOEXCEPT
         : glv::Window( width, height, "Main Window" ), glv::GLV()
    {
        // this->setGLV( *this );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief デストラクタ.
    //! @attention
    //! DropDownオブジェクトは内部で ListViewを利用していますが、親による自動解放が正しく動かないバグがある模様のため、
    //! 明示的に親( m_pSurfaceTable )が解放される前にデストラクションしています.
    //! m_pSurfaceTableもついでに.
    //! ※尚、GLVのViewは glv::SmartObjectクラスにより動的確保したオブジェクトは、親から外される際に自動的に delete されます.
    //!   明示的解放の場合には、親子関係の解除も含めるようにしてください.
    //!--------------------------------------------------------------------------------------
    virtual ~RootSurfaceContext() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ランタイムエンジンにアタッチされた際に呼ばれます.
    //! @see glv::ApplicationLoopCallback::OnLoopAttached( ApplicationLoopContext& )
    //!--------------------------------------------------------------------------------------
    virtual void OnLoopAttached( glv::ApplicationLoopContext& context ) NN_NOEXCEPT NN_OVERRIDE
    {
        ApplicationLoopCallback::OnLoopAttached( context );
        GLV_LOG( "OnLoopAttached\n" );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ランタイムエンジンからデタッチされた際に呼ばれます.
    //! @see glv::ApplicationLoopCallback::OnLoopDetached( ApplicationLoopContext& )
    //!--------------------------------------------------------------------------------------
    virtual void OnLoopDetached( glv::ApplicationLoopContext& context ) NN_NOEXCEPT NN_OVERRIDE
    {
        GLV_LOG( "OnLoopDetached\n" );
        ApplicationLoopCallback::OnLoopDetached( context );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ループ中の呼び出し.@n
    //! @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます.@n
    //! この時点ではまだ glvコンテキストのレンダリングは開始していません.@n
    //! また、このメソッドが呼び出されるフレームは OnLoopAfterSceneRendererと同じです.@n
    //! @return reserved.@n
    //! 現在はRequiredRestoration::RequireRestrationNothing を返却してください.
    //! @see glv::ApplicationLoopCallback::OnLoopBeforeSceneRenderer( ApplicationLoopContext&, const HidEvents& )
    //!--------------------------------------------------------------------------------------
    virtual const glv::RequiredRestoration OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( events );
        NN_UNUSED( context );
        if ( events.GetAvailableBasicPadCount() > 0 )
        {
            auto& bpad = events.GetBasicPad( 0 );
            if ( bpad.IsButtonDown<glv::BasicPadEventType::Button::R>() || bpad.IsButtonRepeat<glv::BasicPadEventType::Button::R>() )
            {
                GLV_LOG( "OnLoopBeforeSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }
        }
        return glv::RequiredRestoration::RequireRestrationNothing;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ループ中の呼び出し.@n
    //! @details glvシーンレンダラのレンダリングが終わった後に呼び出されます.@n
    //! また、このメソッドが呼び出されるフレームは OnLoopBeforeSceneRendererと同じです.@n
    //! @return reserved.@n
    //! 現在はRequiredRestoration::RequireRestrationNothing を返却してください.
    //! @see glv::ApplicationLoopCallback::OnLoopAfterSceneRenderer( ApplicationLoopContext&, const HidEvents& )
    //!--------------------------------------------------------------------------------------
    virtual const glv::RequiredRestoration OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( events );
        NN_UNUSED( context );
        const glv::DebugPadEventType& dpad = events.GetDebugPad();
        if ( true == dpad.HasAnyEvent() )
        {
            // SELECT + START 同時押し( 後押し許可版 )
            const glv::DebugPadEventType::ButtonSetType exit( nn::hid::DebugPadButton::Start::Mask | nn::hid::DebugPadButton::Select::Mask );
            if ( exit == ( dpad.GetButtons() & exit )  )
            {
                // アプリケーションループ退場要求
                glv::ApplicationFrameworkExit();
            }
            if ( dpad.IsButtonDown<nn::hid::DebugPadButton::L>() )
            {
                GLV_LOG( "OnLoopAfterSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }
        }
        else if ( events.GetAvailableBasicPadCount() > 0 )
        {
            const glv::BasicPadEventType& bpad = events.GetBasicPad( 0 );
            // SELECT + START 同時押し( 後押し許可版 )
            const glv::BasicPadEventType::ButtonSetType exit( glv::BasicPadEventType::Button::Start::Mask | glv::BasicPadEventType::Button::Select::Mask );
            if ( exit == ( bpad.GetButtons() & exit )  )
            {
                // アプリケーションループ退場要求
                glv::ApplicationFrameworkExit();
            }

            if ( bpad.IsButtonDown<glv::BasicPadEventType::Button::L>() )
            {
                GLV_LOG( "OnLoopAfterSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }

            /*
            // X : ペアコンセット（デバッグ専用）
            if ( bpad.IsButtonDown<glv::BasicPadEventType::Button::X>() )
            {
                nn::pctl::SetPinCode("0000");
                s_pTextView->AppendValue("[[ parental settings generated ]]\n");
            }
            */

        }

        {
            bool isPctl = nn::pctl::IsRestrictionEnabled();

            if(isPctl)
            {
                s_isPctlValidLabel[0] -> enable(glv::Property::Visible);
                s_isPctlValidLabel[1] -> disable(glv::Property::Visible);
            }
            else
            {
                s_isPctlValidLabel[0] -> disable(glv::Property::Visible);
                s_isPctlValidLabel[1] -> enable(glv::Property::Visible);
            }
        }

        return glv::RequiredRestoration::RequireRestrationNothing;
    }

private:
};

void ShowToolInformation(RootSurfaceContext* context)
{
    // タイトルラベル表示
    auto toolInformationLabel = new glv::Label;
    nn::repair::GetToolInformationLabel(toolInformationLabel, ToolName, ToolMajorVersion, ToolMinorVersion);
    *context << toolInformationLabel;
}

void ShowFirmwareVersion(nn::repair::StreamTextView* view)
{
    glv::Label firmwareVersionLabel;
    nn::repair::GetFirmwareVersionLabel(&firmwareVersionLabel);
    view->AppendValue(firmwareVersionLabel.getValue() + "\n");
}

void ShowSerialNumber(nn::repair::StreamTextView* view)
{
    glv::Label serialNumberLabel;
    nn::repair::GetSerialNumberLabel(&serialNumberLabel);
    view->AppendValue(serialNumberLabel.getValue() + "\n");
}

void ShowDeviceId(nn::repair::StreamTextView* view)
{
    ::nn::Result result;

    // device id の定義 ： デバイス名 = prefix(2byte) + device id(16 byte 16進数文字表記) + "-" + ["1" or "0" (0がPROD)]
    glv::Label deviceIdLabel;
    nn::repair::GetDeviceIdLabel(&deviceIdLabel);

    view->AppendValue(deviceIdLabel.getValue() + "\n");
}

//!--------------------------------------------------------------------------------------
//! @brief メイン
//!--------------------------------------------------------------------------------------

void MyMain() NN_NOEXCEPT
{
    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    RootSurfaceContext* context = new RootSurfaceContext( width, height );

    // タイトルラベル表示
    ShowToolInformation(context);

    // ログビュー表示
    s_pTextView = new nn::repair::StreamTextView(glv::Rect(1024, 440), 28.f);
    s_pTextView->pos(120, 164);
    *context << s_pTextView;

    // serial number (set:cal 権限必要)
    ShowSerialNumber(s_pTextView);

    // device id を得る
    ShowDeviceId(s_pTextView);

    // firmware version
    ShowFirmwareVersion(s_pTextView);

    // 有効ラベル表示
    s_isPctlValidLabel[0] = new glv::Label(DispMsgPCtlEnable, false);
    s_isPctlValidLabel[0]->pos(120, 100);
    s_isPctlValidLabel[0]->size(28.f);
    *context << s_isPctlValidLabel[0];
    s_isPctlValidLabel[1] = new glv::Label(DispMsgPCtlDisable, false);
    s_isPctlValidLabel[1]->pos(120, 100);
    s_isPctlValidLabel[1]->size(28.f);
    *context << s_isPctlValidLabel[1];

    // ホストPCをマウント
    NN_LOG("Mount host\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::MountHost("host", HostMountDir) );

    // nn::manu::InitializeUfio();

    // spl 初期化
    nn::spl::InitializeForCrypto();

    // nn::repair Authentication準備
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::repair::CreateSplAuthenticationKeySource(&m_pKeySource));

    // サーバリクエストデータ作成(Backup)
    glv::Button* put0Button = new nn::repair::LabelButton(DispMsgButtonPut0, MyMouseDownHandlerPut0);
    put0Button->pos(200, 580);
    *context << put0Button;

    // サーバレスポンスデータ受け取り(Backup)
    glv::Button* get0Button = new nn::repair::LabelButton(DispMsgButtonGet0, MyMouseDownHandlerGet0);
    get0Button->pos(500, 580);
    *context << get0Button;

    // サーバリクエストデータ作成(Restore)
    glv::Button* put1Button = new nn::repair::LabelButton(DispMsgButtonPut1, MyMouseDownHandlerPut1);
    put1Button->pos(200, 640);
    *context << put1Button;

    // サーバレスポンスデータ受け取り(Restore)
    glv::Button* get1Button = new nn::repair::LabelButton(DispMsgButtonGet1, MyMouseDownHandlerGet1);
    get1Button->pos(500, 640);
    *context << get1Button;

    // シャットダウンボタン(FW 3.0.0 以降で有効)
    glv::Button* shutdownButton = new nn::repair::ShutdownButton();
    shutdownButton->pos(800,640);
    *context << shutdownButton;

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    context -> setGLV(*context);

    // メインループコールバックを登録.
    ApplicationFrameworkRegisterLoopCallback( context );

    glv::Application::run();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());

    if ( nullptr != context )
    {
        delete context;
    }
}

//!--------------------------------------------------------------------------------------
//! @brief プロセスエントリポイント
//!--------------------------------------------------------------------------------------
NN_OS_EXTERN_C void nnMain() NN_NOEXCEPT
{
    SetupPeripherals();

    glv::ApplicationFrameworkInitialize(LocalHidConfiguration);

    MyMain();
}
