﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <string>
#include <algorithm>
#include "testRepair_Utility.h"
#include <nn/repair/repair_IFile.h>
#include <memory>
#include <nn/fs.h>
#include <nn/repair/repair_FileSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/crypto.h>
#include <random>
#include <functional>
#include <nn/repair/repair_ProtectedFile.h>
#include <nn/manu/manu_Api.h>

#include <nn/nn_Common.h>

using namespace nnt::repair;

namespace {
    void* AllocateForFileSystem(size_t size) NN_NOEXCEPT
    {
        return ::std::malloc(size);
    }

    void DeallocateForFileSystem(void* addr, size_t) NN_NOEXCEPT
    {
        ::std::free(addr);
    }
}

using namespace nn::repair;

class ProtectedFileTest : public ::testing::Test
{
protected:
    virtual void SetUp()
    {
        if (GetTestEnv().IsManuEnabled())
        {
            m_FileSystem = GetTestEnv().GetManuFileSystem();
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::repair::CreateProtectedFileEncryptor(&m_Encryptor, "Spl"));
        }
        else
        {
            m_FileSystem = GetTestEnv().GetNnfsFileSystem();
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::repair::CreateProtectedFileEncryptor(&m_Encryptor, "FixedKey"));
        }

        filename = GetTestEnv().GetWorkingDirectory() + "/test.bin";
        nn::fs::DeleteFile(filename.c_str());
    }

    virtual void TearDown()
    {

    }

    std::shared_ptr<FileSystem> m_FileSystem;
    std::shared_ptr<nn::repair::IProtectedFileEncryptor> m_Encryptor;

    std::string filename;
};

TEST_F(ProtectedFileTest, SingleBlock)
{
    Key128 key;
    NNT_ASSERT_RESULT_SUCCESS(
        m_Encryptor->GenerateEncryptedKey(&key));

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            CreateProtectedFile(&file, m_FileSystem, filename.c_str(), m_Encryptor, key, AuthenticationArchiveContent::MakeZero()));

        const char *data = "abc\n";
        NNT_ASSERT_RESULT_SUCCESS(file->Write(0, data, std::strlen(data), true));
        NNT_ASSERT_RESULT_SUCCESS(file->Write(4, data, std::strlen(data), true));

        EXPECT_EQ(1, file->GetBlockCount());
        EXPECT_EQ(8, file->GetDataSize());
    }

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            OpenProtectedFile(&file, m_FileSystem, filename.c_str(), m_Encryptor, key));

        char data[32] = {};
        size_t readSize;
        NNT_ASSERT_RESULT_SUCCESS(
            file->Read(&readSize, 0, data, 8));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, file->Read(&readSize, 0, data, 9));

        EXPECT_EQ(1, file->GetBlockCount());
        EXPECT_EQ(8, file->GetDataSize());
        EXPECT_STREQ("abc\nabc\n", data);
    }

    nn::fs::FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::OpenFile(&handle, filename.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{
        nn::fs::CloseFile(handle);
    };

    int64_t fileSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, handle));

    const int64_t BlockSize = 1 * 1024 * 1024;
    EXPECT_EQ(fileSize, sizeof(ProtectedFileHeader) + sizeof(BlockHeader) + BlockSize);
}

TEST_F(ProtectedFileTest, MultiBlock)
{
    Key128 key;
    NNT_ASSERT_RESULT_SUCCESS(
        m_Encryptor->GenerateEncryptedKey(&key));

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            CreateProtectedFile(&file, m_FileSystem, filename.c_str(), m_Encryptor, key, AuthenticationArchiveContent::MakeZero()));

        const char *data = "abc\n";
        NNT_ASSERT_RESULT_SUCCESS(file->Write(0, data, std::strlen(data) + 1, true));
        NNT_ASSERT_RESULT_SUCCESS(file->Write(1024 * 1024 - 2, data, std::strlen(data) + 1, true));

        EXPECT_EQ(2, file->GetBlockCount());
        EXPECT_EQ(1024 * 1024 + 3, file->GetDataSize());
    }

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            OpenProtectedFile(&file, m_FileSystem, filename.c_str(), m_Encryptor, key));

        char data[32] = {};
        size_t readSize;
        NNT_ASSERT_RESULT_SUCCESS(
            file->Read(&readSize, 1024 * 1024 - 2, data, 5));

        EXPECT_EQ(2, file->GetBlockCount());
        EXPECT_EQ(1024 * 1024 + 3, file->GetDataSize());
        EXPECT_STREQ("abc\n", data);
    }
}

TEST_F(ProtectedFileTest, Corrupt)
{
    Key128 key;
    NNT_ASSERT_RESULT_SUCCESS(
        m_Encryptor->GenerateEncryptedKey(&key));

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            CreateProtectedFile(&file, m_FileSystem, filename.c_str(), m_Encryptor, key, AuthenticationArchiveContent::MakeZero()));

        const char *data = "abc\n";
        NNT_ASSERT_RESULT_SUCCESS(file->Write(0, data, std::strlen(data) + 1, true));

        EXPECT_EQ(1, file->GetBlockCount());
        EXPECT_EQ(5, file->GetDataSize());
    }

    {
        nn::fs::FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::OpenFile(&handle, filename.c_str(), nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT{
            nn::fs::CloseFile(handle);
        };

        uint32_t corruptData = 0xffffffff;

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::WriteFile(handle, 120, &corruptData, sizeof(corruptData), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            OpenProtectedFile(&file, m_FileSystem, filename.c_str(), m_Encryptor, key));

        char data[32] = {};
        size_t readSize;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultDataCorrupted,
            file->Read(&readSize, 0, data, 5));
    }
}

#if defined NN_BUILD_CONFIG_OS_HORIZON
TEST_F(ProtectedFileTest, SplEncryptor)
{
    {
        std::shared_ptr<nn::repair::IProtectedFileEncryptor> encryptor;
        nn::repair::CreateProtectedFileEncryptor(&encryptor, "Spl");

        Key128 key1, key2;
        NNT_ASSERT_RESULT_SUCCESS(
            encryptor->GenerateEncryptedKey(&key1));
        NNT_ASSERT_RESULT_SUCCESS(
            encryptor->GenerateEncryptedKey(&key2));

        EXPECT_NE(key1, key2);
    }
}

TEST_F(ProtectedFileTest, SplMultiBlock)
{
    std::shared_ptr<nn::repair::IProtectedFileEncryptor> encryptor;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::repair::CreateProtectedFileEncryptor(&encryptor, "Spl"));

    Key128 key;
    NNT_ASSERT_RESULT_SUCCESS(
        m_Encryptor->GenerateEncryptedKey(&key));

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            CreateProtectedFile(&file, m_FileSystem, filename.c_str(), encryptor, key, AuthenticationArchiveContent::MakeZero()));

        const char *data = "abc\n";
        NNT_ASSERT_RESULT_SUCCESS(file->Write(0, data, std::strlen(data) + 1, true));
        NNT_ASSERT_RESULT_SUCCESS(file->Write(1024 * 1024 - 2, data, std::strlen(data) + 1, true));

        EXPECT_EQ(2, file->GetBlockCount());
        EXPECT_EQ(1024 * 1024 + 3, file->GetDataSize());
    }

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            OpenProtectedFile(&file, m_FileSystem, filename.c_str(), encryptor, key));

        char data[32] = {};
        size_t readSize;
        NNT_ASSERT_RESULT_SUCCESS(
            file->Read(&readSize, 1024 * 1024 - 2, data, 5));

        EXPECT_EQ(2, file->GetBlockCount());
        EXPECT_EQ(1024 * 1024 + 3, file->GetDataSize());
        EXPECT_STREQ("abc\n", data);
    }
}

TEST_F(ProtectedFileTest, SplHeaderCorrupt)
{
    std::shared_ptr<nn::repair::IProtectedFileEncryptor> encryptor;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::repair::CreateProtectedFileEncryptor(&encryptor, "Spl"));

    Key128 key;
    NNT_ASSERT_RESULT_SUCCESS(
        encryptor->GenerateEncryptedKey(&key));

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            CreateProtectedFile(&file, m_FileSystem, filename.c_str(), encryptor, key, AuthenticationArchiveContent::MakeZero()));

        const char *data = "abc\n";
        NNT_ASSERT_RESULT_SUCCESS(file->Write(0, data, std::strlen(data) + 1, true));

        EXPECT_EQ(1, file->GetBlockCount());
        EXPECT_EQ(5, file->GetDataSize());
    }

    {
        nn::fs::FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::OpenFile(&handle, filename.c_str(), nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT{
            nn::fs::CloseFile(handle);
        };

        uint32_t corruptData = 0xffffffff;

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::WriteFile(handle, 20, &corruptData, sizeof(corruptData), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultDataCorrupted,
            OpenProtectedFile(&file, m_FileSystem, filename.c_str(), encryptor, key));
    }
}

TEST_F(ProtectedFileTest, SplCorrupt)
{
    std::shared_ptr<nn::repair::IProtectedFileEncryptor> encryptor;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::repair::CreateProtectedFileEncryptor(&encryptor, "Spl"));

    Key128 key;
    NNT_ASSERT_RESULT_SUCCESS(
        encryptor->GenerateEncryptedKey(&key));

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            CreateProtectedFile(&file, m_FileSystem, filename.c_str(), encryptor, key, AuthenticationArchiveContent::MakeZero()));

        const char *data = "abc\n";
        NNT_ASSERT_RESULT_SUCCESS(file->Write(0, data, std::strlen(data) + 1, true));

        EXPECT_EQ(1, file->GetBlockCount());
        EXPECT_EQ(5, file->GetDataSize());
    }

    {
        nn::fs::FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::OpenFile(&handle, filename.c_str(), nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT{
            nn::fs::CloseFile(handle);
        };

        uint32_t corruptData = 0xffffffff;

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::WriteFile(handle, 120, &corruptData, sizeof(corruptData), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    {
        std::shared_ptr<ProtectedFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            OpenProtectedFile(&file, m_FileSystem, filename.c_str(), encryptor, key));

        char data[32] = {};
        size_t readSize;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultDataCorrupted,
            file->Read(&readSize, 0, data, 5));
    }
}
#endif

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(AllocateForFileSystem, ::DeallocateForFileSystem);

    nn::fs::MountHost("hostfs", GetTestEnv().GetWorkingDirectory().c_str());
    nn::fs::MountHostRoot();

    int result = RUN_ALL_TESTS();
    ::nnt::Exit(result);
}
