﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

param
(
    [string]
    # The path to test_detail.xml
    $TestDetailXmlPath,

    [string]
    # The path to the Outputs directory
    $OutputsPath,

    [string]
    # The build type
    $BuildType
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
$scriptDirectoryPath = "${scriptDirectoryPath}\..\..\..\..\..\Integrate\Scripts"

Import-Module "${scriptDirectoryPath}\Modules\Error"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

$TestName = "testCalWriterBinary"

$TestBinaryName = "CalWriter.exe"

if (-not $TestDetailXmlPath)
{
    Write-ErrorMessage "TestDetailXmlPath option not specified"
    exit 1
}

if (-not $OutputsPath)
{
    Write-ErrorMessage "OutputsPath option not specified"
    exit 1
}

if (-not (Test-Path $OutputsPath -Type Container))
{
    Write-ErrorMessage "Outputs directory not found"
    exit 1
}

if (-not $BuildType)
{
    Write-ErrorMessage "BuildType option not specified"
    exit 1
}

$exeFiles = New-Object "System.Collections.Generic.List[string]"

Get-ChildItem -Recurse -Path $OutputsPath -Filter $TestBinaryName | % {
    $fileName = $_.FullName

    if ($fileName -cmatch "\\${TestName}\\${BuildType}\\")
    {
        $exeFiles.Add($fileName)
    }
}

if (0 -eq $exeFiles.Count)
{
    Write-ErrorMessage "${TestBinaryName} not found"
    exit 1
}

$CalWriterPath = $exeFiles[0]

$dataDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
$dataDirectoryPath = "${dataDirectoryPath}\data"

$tests = New-Object "System.Collections.Specialized.OrderedDictionary"

# コマンドの指定が無い場合も正しく処理されるか
$tests["NoCommandTest1"] =
{
    "[ERROR] No command specified." -ceq (& $CalWriterPath)
}

# ゼロ埋めに成功するか
$tests["ZeroInitializeTest1"] =
{
    "[SUCCESS] Done." -ceq (& $CalWriterPath ZeroInitialize)
}

# 仕上げ処理に成功するか
$tests["FinishUpTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $result = (& $CalWriterPath FinishUp)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    if (-not ($result.IndexOf("[OK] MagicNumber : 304C4143`n") -ge 0))
    {
        return $false
    }

    if (-not ($result.IndexOf("[OK] Version     : 13`n") -ge 0))
    {
        return $false
    }

    if (-not ($result.IndexOf("[OK] BodySize    : 32704`n") -ge 0))
    {
        return $false
    }

    if (-not ($result.IndexOf("[OK] Model       : 1`n") -ge 0))
    {
        return $false
    }

    if (-not ($result.IndexOf("[OK] UpdateCount : 1`n") -ge 0))
    {
        return $false
    }

    if (-not ($result.IndexOf("[OK] CRC-16      : FC0A`n") -ge 0))
    {
        return $false
    }

    if (-not ($result.IndexOf("[OK] BodyHash    :`n") -ge 0))
    {
        return $false
    }

    $result = (& $CalWriterPath FinishUp)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    if (-not ($result.IndexOf("[OK] UpdateCount : 2`n") -ge 0))
    {
        return $false
    }

    return $true
}

# 空の構成識別子 1 の書き込みに成功するか
$tests["WriteConfigurationId1Test1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $result = (& $CalWriterPath WriteConfigurationId1)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    return ($result.IndexOf("[OK] ConfigurationId1 :`n[--] `n") -ge 0)
}

# 長い構成識別子 1 の書き込みに成功するか
$tests["WriteConfigurationId1Test2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $value = "a" * 28 + "b"

    $result = (& $CalWriterPath WriteConfigurationId1 $value)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    return ($result.IndexOf("[OK] ConfigurationId1 :`n[--] ${value}`n") -ge 0)
}

# 0 個の無線 LAN 用カントリーコードの書き込みに成功するか
$tests["WriteWlanCountryCodesTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $result = (& $CalWriterPath WriteWlanCountryCodes)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    return ($result.IndexOf("[OK] WirelessLanCountryCodes :`n[--] `n") -ge 0)
}

# 128 個の無線 LAN 用カントリーコードの書き込みに成功するか
$tests["WriteWlanCountryCodesTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $codes = @()

    for ($i = 0; $i -lt 128; ++$i)
    {
        $codes += [char](($i % 26) + 65) + [char]((($i + 10) % 26) + 65)
    }

    $result = (& $CalWriterPath WriteWlanCountryCodes @codes)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] WirelessLanCountryCodes :`n"

    for ($i = 0; $i -lt 8; ++$i)
    {
        $answer += "[--]"

        for ($j = 0; $j -lt 16; ++$j)
        {
            $answer += " " + $codes[$j + ($i * 16)] + ","
        }

        $answer += "`n"
    }

    $answer = $answer.SubString(0, $answer.Length - 2)

    return ($result.IndexOf("${answer}`n") -ge 0)
}

# 無線 LAN 用 MAC アドレスの書き込みに成功するか
$tests["WriteWlanMacAddressTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $values = ("12", "34", "56", "AB", "CD", "EF")

    $result = (& $CalWriterPath WriteWlanMacAddress @values)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] WirelessLanMacAddress :`n[--] "
    $answer += ($values -join "::")

    return ($result.IndexOf("${answer}`n") -ge 0)
}

# 不正な無線 LAN 用 MAC アドレスの書き込みが正しく処理されるか
$tests["WriteWlanMacAddressTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath WriteWlanMacAddress)))
    {
        return $false
    }

    $values = @()
    $values += "12"

    $message = "[ERROR] Invalid arguments count: 1"

    if (-not ($message -ceq (& $CalWriterPath WriteWlanMacAddress @values)))
    {
        return $false
    }
    $values += ("34", "56", "AB", "CD", "EF")

    $message = "[ERROR] Invalid octet: GH"

    for ($i = 0; $i -lt 6; ++$i)
    {
        $errs = $values.Clone()

        $errs[$i] = "GH"

        if (-not ($message -ceq (& $CalWriterPath WriteWlanMacAddress @errs)))
        {
            return $false
        }
    }

    return $true
}

# BD アドレスの書き込みに成功するか
$tests["WriteBdAddressTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $values = ("12", "34", "56", "AB", "CD", "EF")

    $result = (& $CalWriterPath WriteBdAddress @values)


    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] BluetoothBdAddress :`n[--] "
    $answer += ($values -join "::")

    return ($result.IndexOf("${answer}`n") -ge 0)
}

# 不正な BD アドレスの書き込みが正しく処理されるか
$tests["WriteBdAddressTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath WriteBdAddress)))
    {
        return $false
    }

    $values = @()
    $values += "12"

    $message = "[ERROR] Invalid arguments count: 1"

    if (-not ($message -ceq (& $CalWriterPath WriteBdAddress @values)))
    {
        return $false
    }

    $values += ("34", "56", "AB", "CD", "EF")

    $message = "[ERROR] Invalid octet: GH"

    for ($i = 0; $i -lt 6; ++$i)
    {
        $errs = $values.Clone()

        $errs[$i] = "GH"

        if (-not ($message -ceq (& $CalWriterPath WriteBdAddress @errs)))
        {
            return $false
        }
    }

    return $true
}

# 内蔵加速度センサのオフセット値の書き込みに成功するか
$tests["WriteAccelerometerOffsetTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $values = ("256", "4096", "32767")

    $result = (& $CalWriterPath WriteAccelerometerOffset @values)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] AccelerometerOffset :`n[--] (   256,   4096,  32767)"

    return ($result.IndexOf("${answer}`n") -ge 0)
}

# 不正な内蔵加速度センサのオフセット値の書き込みが正しく処理されるか
$tests["WriteAccelerometerOffsetTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteAccelerometerOffset"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $values = @()
    $values += "12"

    $message = "[ERROR] Invalid arguments count: 1"

    if (-not ($message -ceq (& $CalWriterPath $command @values)))
    {
        return $false
    }

    $values += ("34", "56")

    $message = "[ERROR] Invalid number: GH"

    for ($i = 0; $i -lt 3; ++$i)
    {
        $errs = $values.Clone()

        $errs[$i] = "GH"

        if (-not ($message -ceq (& $CalWriterPath $command @errs)))
        {
            return $false
        }
    }

    return $true
}

# 内蔵加速度センサのスケール値の書き込みに成功するか
$tests["WriteAccelerometerScaleTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $values = ("256", "4096", "32767")

    $result = (& $CalWriterPath WriteAccelerometerScale @values)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] AccelerometerScale :`n[--] (   256,   4096,  32767)"

    return ($result.IndexOf("${answer}`n") -ge 0)
}

# 不正な内蔵加速度センサのスケール値の書き込みが正しく処理されるか
$tests["WriteAccelerometerScaleTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteAccelerometerScale"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $values = @()
    $values += "12"

    $message = "[ERROR] Invalid arguments count: 1"

    if (-not ($message -ceq (& $CalWriterPath $command @values)))
    {
        return $false
    }

    $values += ("34", "56")

    $message = "[ERROR] Invalid number: GH"

    for ($i = 0; $i -lt 3; ++$i)
    {
        $errs = $values.Clone()

        $errs[$i] = "GH"

        if (-not ($message -ceq (& $CalWriterPath $command @errs)))
        {
            return $false
        }
    }

    return $true
}

# 内蔵ジャイロスコープのオフセット値の書き込みに成功するか
$tests["WriteGyroscopeOffsetTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $values = ("256", "4096", "32767")

    $result = (& $CalWriterPath WriteGyroscopeOffset @values)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] GyroscopeOffset :`n[--] (   256,   4096,  32767)"

    return ($result.IndexOf("${answer}`n") -ge 0)
}

# 不正な内蔵ジャイロスコープのオフセット値の書き込みが正しく処理されるか
$tests["WriteGyroscopeOffsetTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteGyroscopeOffset"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $values = @()
    $values += "12"

    $message = "[ERROR] Invalid arguments count: 1"

    if (-not ($message -ceq (& $CalWriterPath $command @values)))
    {
        return $false
    }

    $values += ("34", "56")

    $message = "[ERROR] Invalid number: GH"

    for ($i = 0; $i -lt 3; ++$i)
    {
        $errs = $values.Clone()

        $errs[$i] = "GH"

        if (-not ($message -ceq (& $CalWriterPath $command @errs)))
        {
            return $false
        }
    }

    return $true
}

# 内蔵ジャイロスコープのスケール値の書き込みに成功するか
$tests["WriteGyroscopeScaleTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $values = ("256", "4096", "32767")

    $result = (& $CalWriterPath WriteGyroscopeScale @values)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] GyroscopeScale :`n[--] (   256,   4096,  32767)"

    return ($result.IndexOf("${answer}`n") -ge 0)
}

# 不正な内蔵ジャイロスコープのスケール値の書き込みが正しく処理されるか
$tests["WriteGyroscopeScaleTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteGyroscopeScale"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $values = @()
    $values += "12"

    $message = "[ERROR] Invalid arguments count: 1"

    if (-not ($message -ceq (& $CalWriterPath $command @values)))
    {
        return $false
    }

    $values += ("34", "56")

    $message = "[ERROR] Invalid number: GH"

    for ($i = 0; $i -lt 3; ++$i)
    {
        $errs = $values.Clone()

        $errs[$i] = "GH"

        if (-not ($message -ceq (& $CalWriterPath $command @errs)))
        {
            return $false
        }
    }

    return $true
}

# 空の製品シリアル番号の書き込みに成功するか
$tests["WriteSerialNumberTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $result = (& $CalWriterPath WriteSerialNumber)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    return ($result.IndexOf("[OK] SerialNumber :`n[--] `n") -ge 0)
}

# 長い製品シリアル番号の書き込みに成功するか
$tests["WriteSerialNumberTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $value = "a" * 22 + "b"

    $result = (& $CalWriterPath WriteSerialNumber $value)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    return ($result.IndexOf("[OK] SerialNumber :`n[--] ${value}`n") -ge 0)
}

# デバイス登録用秘密鍵 （ECC-P256 版） の書き込みに成功するか
$tests["WriteEccP256DeviceKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\devicekey.txt"

    $result = (& $CalWriterPath WriteEccP256DeviceKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] EccP256DeviceKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正なデバイス登録用秘密鍵 （ECC-P256 版） の書き込みが正しく処理されるか
$tests["WriteEccP256DeviceKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteEccP256DeviceKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 48 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# デバイス登録用証明書 （ECC-P256 版） の書き込みに成功するか
$tests["WriteEccP256DeviceCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\devicecertificate.txt"

    $result = (& $CalWriterPath WriteEccP256DeviceCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] EccP256DeviceCertificate :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正なデバイス登録用証明書 （ECC-P256 版） の書き込みが正しく処理されるか
$tests["WriteEccP256DeviceCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteEccP256DeviceCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 384 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# デバイス登録用秘密鍵 （ECC-B233 版） の書き込みに成功するか
$tests["WriteEccB233DeviceKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\devicekey.txt"

    $result = (& $CalWriterPath WriteEccB233DeviceKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] EccB233DeviceKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正なデバイス登録用秘密鍵 （ECC-B233 版） の書き込みが正しく処理されるか
$tests["WriteEccB233DeviceKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteEccB233DeviceKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 48 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# デバイス登録用証明書 （ECC-B233 版） の書き込みに成功するか
$tests["WriteEccB233DeviceCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\devicecertificate.txt"

    $result = (& $CalWriterPath WriteEccB233DeviceCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] EccB233DeviceCertificate :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正なデバイス登録用証明書 （ECC-B233 版） の書き込みが正しく処理されるか
$tests["WriteEccB233DeviceCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteEccB233DeviceCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 384 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# eTicket 発行用秘密鍵 （ECC-P256 版） の書き込みに成功するか
$tests["WriteEccP256ETicketKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\devicekey.txt"

    $result = (& $CalWriterPath WriteEccP256ETicketKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] EccP256ETicketKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な eTicket 発行用秘密鍵 （ECC-P256 版） の書き込みが正しく処理されるか
$tests["WriteEccP256ETicketKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteEccP256ETicketKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 48 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# eTicket 発行用証明書 （ECC-P256 版） の書き込みに成功するか
$tests["WriteEccP256ETicketCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\devicecertificate.txt"

    $result = (& $CalWriterPath WriteEccP256ETicketCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] EccP256ETicketCertificate :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な eTicket 発行用証明書 （ECC-P256 版） の書き込みが正しく処理されるか
$tests["WriteEccP256ETicketCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteEccP256ETicketCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 384 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# eTicket 発行用秘密鍵 （ECC-B233 版） の書き込みに成功するか
$tests["WriteEccB233ETicketKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\devicekey.txt"

    $result = (& $CalWriterPath WriteEccB233ETicketKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] EccB233ETicketKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な eTicket 発行用秘密鍵 （ECC-B233 版） の書き込みが正しく処理されるか
$tests["WriteEccB233ETicketKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteEccB233ETicketKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 48 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# eTicket 発行用証明書 （ECC-B233 版） の書き込みに成功するか
$tests["WriteEccB233ETicketCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\devicecertificate.txt"

    $result = (& $CalWriterPath WriteEccB233ETicketCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] EccB233ETicketCertificate :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な eTicket 発行用証明書 （ECC-B233 版） の書き込みが正しく処理されるか
$tests["WriteEccB233ETicketCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteEccB233ETicketCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 384 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# SSL クライアント秘密鍵の書き込みに成功するか
$tests["WriteSslKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\sslkey.txt"

    $result = (& $CalWriterPath WriteSslKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] SslKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な SSL クライアント秘密鍵の書き込みが正しく処理されるか
$tests["WriteSslKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteSslKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 272 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# 空のSSL クライアント証明書の書き込みに成功するか
$tests["WriteSslCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\empty.txt"

    $result = (& $CalWriterPath WriteSslCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer  = "[OK] SslCertificateSize :`n"
    $answer += "[--] " + (Get-ChildItem $path).Length + "`n"
    $answer += "[OK] SslCertificateHash :`n"
    $sha = [System.Security.Cryptography.SHA256]::Create()
    $hash = $sha.ComputeHash([System.IO.File]::ReadAllBytes($path))
    $octets = New-Object "System.Collections.ArrayList"
    $hash | % { [void]$octets.Add($_.ToString("X2")) }
    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 短い SSL クライアント証明書の書き込みに成功するか
$tests["WriteSslCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\one.txt"

    $result = (& $CalWriterPath WriteSslCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer  = "[OK] SslCertificateSize :`n"
    $answer += "[--] " + (Get-ChildItem $path).Length + "`n"
    $answer += "[OK] SslCertificateHash :`n"
    $sha = [System.Security.Cryptography.SHA256]::Create()
    $hash = $sha.ComputeHash([System.IO.File]::ReadAllBytes($path))
    $octets = New-Object "System.Collections.ArrayList"
    $hash | % { [void]$octets.Add($_.ToString("X2")) }
    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 長い SSL クライアント証明書の書き込みに成功するか
$tests["WriteSslCertificateTest3"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\sslcertificate.txt"

    $result = (& $CalWriterPath WriteSslCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer  = "[OK] SslCertificateSize :`n"
    $answer += "[--] " + (Get-ChildItem $path).Length + "`n"
    $answer += "[OK] SslCertificateHash :`n"
    $sha = [System.Security.Cryptography.SHA256]::Create()
    $hash = $sha.ComputeHash([System.IO.File]::ReadAllBytes($path))
    $octets = New-Object "System.Collections.ArrayList"
    $hash | % { [void]$octets.Add($_.ToString("X2")) }
    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な SSL クライアント証明書の書き込みが正しく処理されるか
$tests["WriteSslCertificateTest4"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteSslCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    return $true
}

# 乱数の書き込みに成功するか
$tests["WriteRandomNumberTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\random.txt"

    $result = (& $CalWriterPath WriteRandomNumber $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] RandomNumberHash :`n"
    $sha = [System.Security.Cryptography.SHA256]::Create()
    $hash = $sha.ComputeHash([System.IO.File]::ReadAllBytes($path))
    $octets = New-Object "System.Collections.ArrayList"
    $hash | % { [void]$octets.Add($_.ToString("X2")) }
    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な乱数の書き込みが正しく処理されるか
$tests["WriteRandomNumberTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteRandomNumber"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 4096 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# ゲームカード相互認証秘密鍵の書き込みに成功するか
$tests["WriteGameCardKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\gamecardkey.txt"

    $result = (& $CalWriterPath WriteGameCardKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] GameCardKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正なゲームカード相互認証秘密鍵の書き込みが正しく処理されるか
$tests["WriteGameCardKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteGameCardKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 272 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# ゲームカード相互認証証明書の書き込みに成功するか
$tests["WriteGameCardCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\gamecardcertificate.txt"

    $result = (& $CalWriterPath WriteGameCardCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] GameCardCertificateHash :`n"
    $sha = [System.Security.Cryptography.SHA256]::Create()
    $hash = $sha.ComputeHash([System.IO.File]::ReadAllBytes($path))
    $octets = New-Object "System.Collections.ArrayList"
    $hash | % { [void]$octets.Add($_.ToString("X2")) }
    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正なゲームカード相互認証証明書の書き込みが正しく処理されるか
$tests["WriteGameCardCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteGameCardCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 1024 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# 空の電池 LOT の書き込みに成功するか
$tests["WriteBatteryLot1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $result = ((& $CalWriterPath WriteBatteryLot) -join "`n")

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] BatteryLot :`n"
    $answer += "[--] " + "0" * 32 + "`n"
    $answer += "[--] " + "0" * 16  + "`n"

    return ($result.IndexOf("${answer}") -ge 0)
}

# 電池 LOT の書き込みに成功するか
$tests["WriteBatteryLot2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $value = "01234567890"
    $result = ((& $CalWriterPath WriteBatteryLot $value) -join "`n")

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] BatteryLot :`n"

    return ($result.IndexOf("${answer}") -ge 0)
}

# eTicket 発行用証明書 （RSA2048 版） の書き込みに成功するか
$tests["WriteRsa2048ETicketCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\rsa2048certificate.txt"

    $result = (& $CalWriterPath WriteRsa2048ETicketCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] Rsa2048ETicketCertificate :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な eTicket 発行用証明書 （RSA2048 版） の書き込みが正しく処理されるか
$tests["WriteRsa2048ETicketCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteRsa2048ETicketCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 576 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# eTicket 発行用秘密鍵 （RSA2048 版） の書き込みに成功するか
$tests["WriteRsa2048ETicketKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\rsa2048key.txt"

    $result = (& $CalWriterPath WriteRsa2048ETicketKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] Rsa2048ETicketKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な eTicket 発行用秘密鍵 （RSA2048 版） の書き込みが正しく処理されるか
$tests["WriteRsa2048ETicketKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteRsa2048ETicketKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 544 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# スピーカー補正値の書き込みに成功するか
$tests["WriteSpeakerCalibrationValueTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\speakercalibration.txt"

    $result = (& $CalWriterPath WriteSpeakerCalibrationValue $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] SpeakerCalibrationValue :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $size = 16
        if($octets.Count -lt $size)
        {
            $size = $octets.Count
        }

        $answer += "[--] " + ($octets.GetRange(0, $size) -join "") + "`n"
        $octets.RemoveRange(0, $size)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# リージョンコードの書き込みに成功するか
$tests["WriteRegionCodeTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $result = (& $CalWriterPath WriteRegionCode Taiwan)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] RegionCode :`n"
	$answer += "[--] Taiwan"

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正なリージョンコードの書き込みが正しく処理されるか
$tests["WriteRegionCodeTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)
	$result = (& $CalWriterPath WriteRegionCode)

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq $result))
    {
        return $false
    }

    [void](& $CalWriterPath ZeroInitialize)

	$input = "1A2"
    $result = (& $CalWriterPath WriteRegionCode $input)

	$message = "[ERROR] Invalid region code: " + $input

    if (-not ($message -ceq $result))
    {
        return $false
    }

	return $true
}

# Amiibo 用秘密鍵の書き込みに成功するか
$tests["WriteAmiiboKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\amiibokey.txt"

    $result = (& $CalWriterPath WriteAmiiboKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] AmiiboKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な Amiibo 用秘密鍵の書き込みが正しく処理されるか
$tests["WriteAmiiboKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteAmiiboKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 80 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# amiibo用 ECQV 証明書の書き込みに成功するか
$tests["WriteAmiiboEcqvCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\amiiboecqvcertificate.txt"

    $result = (& $CalWriterPath WriteAmiiboEcqvCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] AmiiboEcqvCertificate :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $size = 16
        if($octets.Count -lt $size)
        {
            $size = $octets.Count
        }

        $answer += "[--] " + ($octets.GetRange(0, $size) -join "") + "`n"
        $octets.RemoveRange(0, $size)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な amiibo用 ECQV 証明書の書き込みが正しく処理されるか
$tests["WriteAmiiboEcqvCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteAmiiboEcqvCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 20 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# amiibo用 ECDSA 証明書の書き込みに成功するか
$tests["WriteAmiiboEcdsaCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\amiiboecdsacertificate.txt"

    $result = (& $CalWriterPath WriteAmiiboEcdsaCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] AmiiboEcdsaCertificate :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な amiibo用 ECDSA 証明書の書き込みが正しく処理されるか
$tests["WriteAmiiboEcdsaCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteAmiiboEcdsaCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 112 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# amiibo 用 ECQV-BLS 本体秘密鍵を書き込みに成功するか
$tests["WriteAmiiboEcqvBlsKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\amiiboecqvblskey.txt"

    $result = (& $CalWriterPath WriteAmiiboEcqvBlsKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] AmiiboEcqvBlsKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な amiibo 用 ECQV-BLS 本体秘密鍵を書き込みが正しく処理されるか
$tests["WriteAmiiboEcqvBlsKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteAmiiboEcqvBlsKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 64 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# amiibo 用 ECQV-BLS 証明書を書き込みに成功するか
$tests["WriteAmiiboEcqvBlsCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\amiiboecqvblscertificate.txt"

    $result = (& $CalWriterPath WriteAmiiboEcqvBlsCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] AmiiboEcqvBlsCertificate :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な amiibo 用 ECQV-BLS 証明書を書き込みが正しく処理されるか
$tests["WriteAmiiboEcqvBlsCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteAmiiboEcqvBlsCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 32 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# amiibo 用 ルート証明書 (ECQV-BLS) を書き込みに成功するか
$tests["WriteAmiiboEcqvBlsRootCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\amiiboecqvblsrootcertificate.txt"

    $result = (& $CalWriterPath WriteAmiiboEcqvBlsRootCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] AmiiboEcqvBlsRootCertificate :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な amiibo 用 ルート証明書 (ECQV-BLS) を書き込みが正しく処理されるか
$tests["WriteAmiiboEcqvBlsRootCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteAmiiboEcqvBlsRootCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 144 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# 製品モデルタイプの書き込みに成功するか
$tests["WriteProductModelTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $value = "Nx"

    $result = (& $CalWriterPath WriteProductModel $value)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] ProductModel :`n"
    $answer += "[--] " + "Nx"

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な製品モデルタイプの書き込みが正しく処理されるか
$tests["WriteProductModelTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteProductModel"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $value = "hoge"
    $message = "[ERROR] Invalid product model: ${value}"

    if (-not ($message -ceq (& $CalWriterPath $command $value)))
    {
        return $false
    }

    return $true
}

# 本体カラーバリエーションの書き込みに成功するか
$tests["WriteColorVariationTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\colorvariation.txt"

    $result = (& $CalWriterPath WriteColorVariation $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] ColorVariation :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 6) -join "") + "`n"
        $octets.RemoveRange(0, 6)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な本体カラーバリエーションの書き込みが正しく処理されるか
$tests["WriteColorVariationTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteColorVariation"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 6 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# LCD バックライト出力の写像の係数の書き込みの書き込みに成功するか
$tests["WriteLcdBacklightBrightnessMappingTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $result = (& $CalWriterPath WriteLcdBacklightBrightnessMapping 0.1 0.2 3.5)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] LcdBacklightBrightnessMapping :`n"

    $answers += "[--] " + "(0.100, 0.200, 3.500)"

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正なLCD バックライト出力の写像の係数の書き込みが正しく処理されるか
$tests["WriteLcdBacklightBrightnessMappingTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteLcdBacklightBrightnessMapping"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $message = "[ERROR] Invalid arguments count: 2"

    if (-not ($message -ceq (& $CalWriterPath $command 1.0 1.0)))
    {
        return $false
    }

    return $true
}

# 領域拡張されたデバイス登録用秘密鍵 （ECC-B233 版） の書き込みに成功するか
$tests["WriteExtendedEccB233DeviceKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\extendeddevicekey.txt"

    $result = (& $CalWriterPath WriteExtendedEccB233DeviceKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] ExtendedEccB233DeviceKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な領域拡張されたデバイス登録用秘密鍵 （ECC-B233 版） の書き込みが正しく処理されるか
$tests["WriteExtendedEccB233DeviceKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteExtendedEccB233DeviceKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 80 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# 領域拡張された eTicket 発行用秘密鍵 （ECC-P256 版） の書き込みに成功するか
$tests["WriteExtendedEccP256ETicketKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\extendeddevicekey.txt"

    $result = (& $CalWriterPath WriteExtendedEccP256ETicketKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] ExtendedEccP256ETicketKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な領域拡張された eTicket 発行用秘密鍵 （ECC-P256 版） の書き込みが正しく処理されるか
$tests["WriteExtendedEccP256ETicketKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteExtendedEccP256ETicketKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 80 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# 領域拡張された eTicket 発行用秘密鍵 （ECC-P256 版） の書き込みに成功するか
$tests["WriteExtendedEccP256ETicketKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\extendeddevicekey.txt"

    $result = (& $CalWriterPath WriteExtendedEccP256ETicketKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] ExtendedEccP256ETicketKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な eTicket 発行用秘密鍵 （ECC-P256 版） の書き込みが正しく処理されるか
$tests["WriteExtendedEccP256ETicketKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteExtendedEccP256ETicketKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 80 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# 領域拡張された eTicket 発行用秘密鍵 （ECC-B233 版） の書き込みに成功するか
$tests["WriteExtendedEccB233ETicketKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\extendeddevicekey.txt"

    $result = (& $CalWriterPath WriteExtendedEccB233ETicketKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] ExtendedEccB233ETicketKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な領域拡張された eTicket 発行用秘密鍵 （ECC-B233 版） の書き込みが正しく処理されるか
$tests["WriteExtendedEccB233ETicketKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteExtendedEccB233ETicketKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 80 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}


# 領域拡張された eTicket 発行用秘密鍵 （RSA2048 版） の書き込みに成功するか
$tests["WriteExtendedRsa2048ETicketKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\extendedrsa2048key.txt"

    $result = (& $CalWriterPath WriteExtendedRsa2048ETicketKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] ExtendedRsa2048ETicketKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な領域拡張された eTicket 発行用秘密鍵 （RSA2048 版） の書き込みが正しく処理されるか
$tests["WriteExtendedRsa2048ETicketKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteExtendedRsa2048ETicketKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 576 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# 領域拡張された SSL クライアント秘密鍵の書き込みに成功するか
$tests["WriteExtendedSslKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\extendedsslkey.txt"

    $result = (& $CalWriterPath WriteExtendedSslKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] ExtendedSslKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な領域拡張された SSL クライアント秘密鍵の書き込みが正しく処理されるか
$tests["WriteExtendedSslKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteExtendedSslKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 304 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# 領域拡張されたゲームカード相互認証秘密鍵の書き込みに成功するか
$tests["WriteExtendedGameCardKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\extendedgamecardkey.txt"

    $result = (& $CalWriterPath WriteExtendedGameCardKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] ExtendedGameCardKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な領域拡張されたゲームカード相互認証秘密鍵の書き込みが正しく処理されるか
$tests["WriteExtendedGameCardKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteExtendedGameCardKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 304 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# LCD ベンダ ID の書き込みに成功するか
$tests["WriteLcdVendorIdTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $arg = "1a2"
    $result = (& $CalWriterPath WriteLcdVendorId $arg)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] LcdVendorId :`n"
    $answer += "[--] " + $arg + "`n"

    return ($result.IndexOf("${answer}") -ge 0)
}

# LCD ベンダ ID の書き込みが正しく処理されるか
$tests["WriteLcdVendorIdTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteLcdVendorId"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $arg = "VENDOR"

    $message = "[ERROR] Invalid hexadecimal number: $arg"

    if (-not ($message -ceq (& $CalWriterPath $command $arg)))
    {
        return $false
    }

    return $true
}

# 領域拡張されたデバイス登録用秘密鍵 （RSA2048 版） の書き込みに成功するか
$tests["WriteExtendedRsa2048DeviceKeyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\extendedrsa2048key.txt"

    $result = (& $CalWriterPath WriteExtendedRsa2048DeviceKey $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] ExtendedRsa2048DeviceKey :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な領域拡張されたデバイス登録用秘密鍵 （RSA2048 版） の書き込みが正しく処理されるか
$tests["WriteExtendedRsa2048DeviceKeyTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteExtendedRsa2048DeviceKey"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 576 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# デバイス登録用証明書 （RSA2048 版） の書き込みに成功するか
$tests["WriteRsa2048DeviceCertificateTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $path = "${dataDirectoryPath}\rsa2048certificate.txt"

    $result = (& $CalWriterPath WriteRsa2048DeviceCertificate $path)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] Rsa2048DeviceCertificate :`n"

    $octets = New-Object "System.Collections.ArrayList"
    [System.IO.File]::ReadAllBytes($path) | % {
        [void]$octets.Add($_.ToString("X2"))
    }

    while ($octets.Count -gt 0)
    {
        $answer += "[--] " + ($octets.GetRange(0, 16) -join "") + "`n"
        $octets.RemoveRange(0, 16)
    }

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正なデバイス登録用証明書 （RSA2048 版） の書き込みが正しく処理されるか
$tests["WriteRsa2048DeviceCertificateTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteRsa2048DeviceCertificate"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\nothing.txt"

    $message = "[ERROR] Failed to open the file: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "${dataDirectoryPath}\one.txt"

    $message = "[ERROR] Expected size is 576 byte(s) but "

    if (-not (& $CalWriterPath $command $path).StartsWith($message))
    {
        return $false
    }

    return $true
}

# USB Type-C Power Source 回路バージョンの書き込みに成功するか
$tests["WriteUsbTypeCPowerSourceCircuitVersionTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $arg = "1"
    $result = (& $CalWriterPath WriteUsbTypeCPowerSourceCircuitVersion $arg)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] UsbTypeCPowerSourceCircuit :`n"
    $answer += "[--] " + $arg + "`n"

    return ($result.IndexOf("${answer}") -ge 0)
}

# USB Type-C Power Source 回路バージョンの書き込みが正しく処理されるか
$tests["WriteUsbTypeCPowerSourceCircuitVersionTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteUsbTypeCPowerSourceCircuitVersion"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $arg = "USB"

    $message = "[ERROR] Invalid hexadecimal number: $arg"

    if (-not ($message -ceq (& $CalWriterPath $command $arg)))
    {
        return $false
    }

    return $true
}

# 本体サブカラーの書き込みに成功するか
$tests["WriteHousingSubColorTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $values = ("255", "255", "255", "255")

    $result = (& $CalWriterPath WriteHousingSubColor @values)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] HousingSubColor :`n[--] FFFFFFFF"

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な本体サブカラーの書き込みが正しく処理されるか
$tests["WriteHousingSubColorTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteHousingSubColor"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $values = @()
    $values += "255"

    $message = "[ERROR] Invalid arguments count: 1"

    if (-not ($message -ceq (& $CalWriterPath $command @values)))
    {
        return $false
    }

    $values += ("255", "255", "255")

    $message = "[ERROR] Invalid number: GH"

    for ($i = 0; $i -lt 4; ++$i)
    {
        $errs = $values.Clone()

        $errs[$i] = "GH"

        if (-not ($message -ceq (& $CalWriterPath $command @errs)))
        {
            return $false
        }
    }

    return $true
}

# 本体液晶縁カラーの書き込みに成功するか
$tests["WriteHousingBezelColorTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $values = ("255", "255", "255", "255")

    $result = (& $CalWriterPath WriteHousingBezelColor @values)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] HousingBezelColor :`n[--] FFFFFFFF"

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な本体液晶縁カラーの書き込みが正しく処理されるか
$tests["WriteHousingBezelColorTest2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteHousingBezelColor"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $values = @()
    $values += "255"

    $message = "[ERROR] Invalid arguments count: 1"

    if (-not ($message -ceq (& $CalWriterPath $command @values)))
    {
        return $false
    }

    $values += ("255", "255", "255")

    $message = "[ERROR] Invalid number: GH"

    for ($i = 0; $i -lt 4; ++$i)
    {
        $errs = $values.Clone()

        $errs[$i] = "GH"

        if (-not ($message -ceq (& $CalWriterPath $command @errs)))
        {
            return $false
        }
    }

    return $true
}

# 本体カラー 1 の書き込みに成功するか
$tests["WriteHousingMainColor1Test1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $values = ("255", "255", "255", "255")

    $result = (& $CalWriterPath WriteHousingMainColor1 @values)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] HousingMainColor1 :`n[--] FFFFFFFF"

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な本体カラー 1 の書き込みが正しく処理されるか
$tests["WriteHousingMainColor1Test2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteHousingMainColor1"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $values = @()
    $values += "255"

    $message = "[ERROR] Invalid arguments count: 1"

    if (-not ($message -ceq (& $CalWriterPath $command @values)))
    {
        return $false
    }

    $values += ("255", "255", "255")

    $message = "[ERROR] Invalid number: GH"

    for ($i = 0; $i -lt 4; ++$i)
    {
        $errs = $values.Clone()

        $errs[$i] = "GH"

        if (-not ($message -ceq (& $CalWriterPath $command @errs)))
        {
            return $false
        }
    }

    return $true
}

# 本体カラー 2 の書き込みに成功するか
$tests["WriteHousingMainColor2Test1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $values = ("255", "255", "255", "255")

    $result = (& $CalWriterPath WriteHousingMainColor2 @values)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] HousingMainColor2 :`n[--] FFFFFFFF"

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な本体カラー 2 の書き込みが正しく処理されるか
$tests["WriteHousingMainColor2Test2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteHousingMainColor2"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $values = @()
    $values += "255"

    $message = "[ERROR] Invalid arguments count: 1"

    if (-not ($message -ceq (& $CalWriterPath $command @values)))
    {
        return $false
    }

    $values += ("255", "255", "255")

    $message = "[ERROR] Invalid number: GH"

    for ($i = 0; $i -lt 4; ++$i)
    {
        $errs = $values.Clone()

        $errs[$i] = "GH"

        if (-not ($message -ceq (& $CalWriterPath $command @errs)))
        {
            return $false
        }
    }

    return $true
}



# 本体カラー 3 の書き込みに成功するか
$tests["WriteHousingMainColor3Test1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $values = ("255", "255", "255", "255")

    $result = (& $CalWriterPath WriteHousingMainColor3 @values)

    if (-not ("[SUCCESS] Done." -ceq $result))
    {
        return $false
    }

    $result = ((& $CalWriterPath Verify) -join "`n")

    $answer = "[OK] HousingMainColor3 :`n[--] FFFFFFFF"

    return ($result.IndexOf("${answer}") -ge 0)
}

# 不正な本体カラー 3 の書き込みが正しく処理されるか
$tests["WriteHousingMainColor3Test2"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $command = "WriteHousingMainColor3"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $values = @()
    $values += "255"

    $message = "[ERROR] Invalid arguments count: 1"

    if (-not ($message -ceq (& $CalWriterPath $command @values)))
    {
        return $false
    }

    $values += ("255", "255", "255")

    $message = "[ERROR] Invalid number: GH"

    for ($i = 0; $i -lt 4; ++$i)
    {
        $errs = $values.Clone()

        $errs[$i] = "GH"

        if (-not ($message -ceq (& $CalWriterPath $command @errs)))
        {
            return $false
        }
    }

    return $true
}

# ベリフィケーションが正しく処理されるか
$tests["VerifyTest1"] =
{
    [void](& $CalWriterPath ZeroInitialize)

    $result = ((& $CalWriterPath Verify) -join "`n")

    if (-not ($result.IndexOf("[ERROR] Verification failed.") -ge 0))
    {
        return $false
    }

    $deviceKeyPath    = "${dataDirectoryPath}\devicekey.txt"
    $deviceCertPath   = "${dataDirectoryPath}\devicecertificate.txt"
    $sslKeyPath       = "${dataDirectoryPath}\sslkey.txt"
    $sslCertPath      = "${dataDirectoryPath}\sslcertificate.txt"
    $randomPath       = "${dataDirectoryPath}\random.txt"
    $gameCardKeyPath  = "${dataDirectoryPath}\gamecardkey.txt"
    $gameCardCertPath = "${dataDirectoryPath}\gamecardcertificate.txt"
    $speakerCalibrationPath = "${dataDirectoryPath}\speakercalibration.txt"
    $rsa2048DeviceKeyPath   = "${dataDirectoryPath}\rsa2048key.txt"
    $rsa2048CertPath        = "${dataDirectoryPath}\rsa2048certificate.txt"
    $amiiboKeyPath          = "${dataDirectoryPath}\amiibokey.txt"
    $amiiboEcqvCertPath     = "${dataDirectoryPath}\amiiboecqvcertificate.txt"
    $amiiboEcdsaCertPath    = "${dataDirectoryPath}\amiiboecdsacertificate.txt"
    $amiiboEcqvBlsKeyPath   = "${dataDirectoryPath}\amiiboecqvblskey.txt"
    $amiiboEcqvBlsCertPath   = "${dataDirectoryPath}\amiiboecqvblscertificate.txt"
    $amiiboEcqvBlsRootCertPath   = "${dataDirectoryPath}\amiiboecqvblsrootcertificate.txt"
    $colorVariationPath   = "${dataDirectoryPath}\colorvariation.txt"
    $extendedDeviceKeyPath = "${dataDirectoryPath}\extendeddevicekey.txt"
    $extendedRsa2048KeyPath = "${dataDirectoryPath}\extendedrsa2048key.txt"
    $extendedSslKeyPath = "${dataDirectoryPath}\extendedsslkey.txt"
    $extendedGameCardKeyPath = "${dataDirectoryPath}\extendedgamecardkey.txt"

    [void](& $CalWriterPath WriteConfigurationId1 BoardType)
    [void](& $CalWriterPath WriteWlanCountryCodes CA US)
    [void](& $CalWriterPath WriteWlanMacAddress 32 61 3C 4E B6 05)
    [void](& $CalWriterPath WriteBdAddress 04 A3 43 5F 43 23)
    [void](& $CalWriterPath WriteAccelerometerOffset 0 0 0)
    [void](& $CalWriterPath WriteAccelerometerScale 4096 4096 4096)
    [void](& $CalWriterPath WriteGyroscopeOffset 0 0 0)
    [void](& $CalWriterPath WriteGyroscopeScale 6686 6686 6686)
    [void](& $CalWriterPath WriteSerialNumber ABCD0123456789)
    [void](& $CalWriterPath WriteEccP256DeviceKey $deviceKeyPath)
    [void](& $CalWriterPath WriteEccP256DeviceCertificate $deviceCertPath)
    [void](& $CalWriterPath WriteEccB233DeviceKey $deviceKeyPath)
    [void](& $CalWriterPath WriteEccB233DeviceCertificate $deviceCertPath)
    [void](& $CalWriterPath WriteEccP256ETicketKey $deviceKeyPath)
    [void](& $CalWriterPath WriteEccP256ETicketCertificate $deviceCertPath)
    [void](& $CalWriterPath WriteEccB233ETicketKey $deviceKeyPath)
    [void](& $CalWriterPath WriteEccB233ETicketCertificate $deviceCertPath)
    [void](& $CalWriterPath WriteSslKey $sslKeyPath)
    [void](& $CalWriterPath WriteSslCertificate $sslCertPath)
    [void](& $CalWriterPath WriteRandomNumber $randomPath)
    [void](& $CalWriterPath WriteGameCardKey $gameCardKeyPath)
    [void](& $CalWriterPath WriteGameCardCertificate $gameCardCertPath)
    [void](& $CalWriterPath WriteRsa2048ETicketKey $rsa2048DeviceKeyPath)
    [void](& $CalWriterPath WriteRsa2048ETicketCertificate $rsa2048CertPath)
    [void](& $CalWriterPath WriteBatteryLot 1234567890)
    [void](& $CalWriterPath WriteSpeakerCalibrationValue $speakerCalibrationPath)
	[void](& $CalWriterPath WriteRegionCode Japan)
    [void](& $CalWriterPath WriteAmiiboKey $amiiboKeyPath)
    [void](& $CalWriterPath WriteAmiiboEcqvCertificate $amiiboEcqvCertPath)
    [void](& $CalWriterPath WriteAmiiboEcdsaCertificate $amiiboEcdsaCertPath)
    [void](& $CalWriterPath WriteAmiiboEcqvBlsKey $amiiboEcqvBlsKeyPath)
    [void](& $CalWriterPath WriteAmiiboEcqvBlsCertificate $amiiboEcqvBlsCertPath)
    [void](& $CalWriterPath WriteAmiiboEcqvBlsRootCertificate $amiiboEcqvBlsRootCertPath)
    [void](& $CalWriterPath WriteProductModel Nx)
    [void](& $CalWriterPath WriteColorVariation $colorVariationPath)
    [void](& $CalWriterPath WriteLcdBacklightBrightnessMapping 1.0 0.0 0.07)
    [void](& $CalWriterPath WriteExtendedEccB233DeviceKey $extendedDeviceKeyPath)
    [void](& $CalWriterPath WriteExtendedEccP256ETicketKey $extendedDeviceKeyPath)
    [void](& $CalWriterPath WriteExtendedEccB233ETicketKey $extendedDeviceKeyPath)
    [void](& $CalWriterPath WriteExtendedRsa2048ETicketKey $extendedRsa2048KeyPath)
    [void](& $CalWriterPath WriteExtendedSslKey $extendedSslKeyPath)
    [void](& $CalWriterPath WriteExtendedGameCardKey $extendedGameCardKeyPath)
    [void](& $CalWriterPath WriteLcdVendorId 1a4)
    [void](& $CalWriterPath WriteExtendedRsa2048DeviceKey $extendedRsa2048KeyPath)
    [void](& $CalWriterPath WriteRsa2048DeviceCertificate $rsa2048CertPath)
    [void](& $CalWriterPath WriteUsbTypeCPowerSourceCircuitVersion 1)
    [void](& $CalWriterPath WriteHousingSubColor 255 255 255 255)
    [void](& $CalWriterPath WriteHousingBezelColor 255 255 255 255)
    [void](& $CalWriterPath WriteHousingMainColor1 255 255 255 255)
    [void](& $CalWriterPath WriteHousingMainColor2 255 255 255 255)
    [void](& $CalWriterPath WriteHousingMainColor3 255 255 255 255)
    [void](& $CalWriterPath FinishUp)

    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not ($result.IndexOf("[SUCCESS] Done.") -ge 0))
    {
        return $false
    }

    $calDatPath  = [System.IO.Path]::GetDirectoryName($CalWriterPath)
    $calDatPath += "\calibration.dat"

    $calBytes = [System.IO.File]::ReadAllBytes($calDatPath)

    $bytes = $calBytes.Clone()
    $bytes[0] = ($bytes[0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] CRC-16 ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000040] = ($bytes[0x000040] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] ConfigurationId1 ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000080] = ($bytes[0x000080] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] WirelessLanCountryCodes ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000210] = ($bytes[0x000210] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] WirelessLanMacAddress ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000220] = ($bytes[0x000220] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] BluetoothBdAddress ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000230] = ($bytes[0x000230] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] AccelerometerOffset ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000238] = ($bytes[0x000238] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] AccelerometerScale ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000240] = ($bytes[0x000240] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] GyroscopeOffset ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000248] = ($bytes[0x000248] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] GyroscopeScale ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000250] = ($bytes[0x000250] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] SerialNumber ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000270] = ($bytes[0x000270] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] EccP256DeviceKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x0002B0] = ($bytes[0x0002B0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] EccP256DeviceCertificate ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000440] = ($bytes[0x000440] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] EccB233DeviceKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000480] = ($bytes[0x000480] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] EccB233DeviceCertificate ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000610] = ($bytes[0x000610] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] EccP256ETicketKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000650] = ($bytes[0x000650] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] EccP256ETicketCertificate ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x0007E0] = ($bytes[0x0007E0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] EccB233ETicketKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000820] = ($bytes[0x000820] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] EccB233ETicketCertificate ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x0009B0] = ($bytes[0x0009B0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] SslKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000AD0] = ($bytes[0x000AD0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] SslCertificateSize ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x000AE0] = ($bytes[0x000AE0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] SslCertificateHash ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x001300] = ($bytes[0x001300] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] RandomNumberHash ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x002320] = ($bytes[0x002320] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] GameCardKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x002440] = ($bytes[0x002440] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] GameCardCertificateHash ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x002860] = ($bytes[0x002860] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] Rsa2048ETicketKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x002A90] = ($bytes[0x002A90] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] Rsa2048ETicketCertificate ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x002CE0] = ($bytes[0x002CE0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] BatteryLot ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x002D00] = ($bytes[0x002D00] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] SpeakerCalibrationValue ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003510] = ($bytes[0x003510] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] RegionCode ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003520] = ($bytes[0x003520] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] AmiiboKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003580] = ($bytes[0x003580] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] AmiiboEcqvCertificate ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x0035A0] = ($bytes[0x0035A0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] AmiiboEcdsaCertificate ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003620] = ($bytes[0x003620] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] AmiiboEcqvBlsKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003670] = ($bytes[0x003670] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] AmiiboEcqvBlsCertificate ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x0036A0] = ($bytes[0x0036A0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] AmiiboEcqvBlsRootCertificate ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003740] = ($bytes[0x003740] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] ProductModel ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003750] = ($bytes[0x003750] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] ColorVariation ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003760] = ($bytes[0x003760] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] LcdBacklightBrightnessMapping ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003770] = ($bytes[0x003770] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] ExtendedEccB233DeviceKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x0037D0] = ($bytes[0x0037D0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] ExtendedEccP256ETicketKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003830] = ($bytes[0x003830] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] ExtendedEccB233ETicketKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003890] = ($bytes[0x003890] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] ExtendedRsa2048ETicketKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003AE0] = ($bytes[0x003AE0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] ExtendedSslKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003C20] = ($bytes[0x003C20] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] ExtendedGameCardKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003D60] = ($bytes[0x003D60] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] LcdVendorId ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003D70] = ($bytes[0x003D70] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] ExtendedRsa2048DeviceKey ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x003FC0] = ($bytes[0x003FC0] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] Rsa2048DeviceCertificate ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x004210] = ($bytes[0x004210] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] UsbTypeCPowerSourceCircuit ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x004220] = ($bytes[0x004220] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] HousingSubColor ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x004230] = ($bytes[0x004230] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] HousingBezelColor ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x004240] = ($bytes[0x004240] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] HousingMainColor1 ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x004250] = ($bytes[0x004250] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] HousingMainColor2 ") -ge 0)))
    {
        return $false
    }

    $bytes = $calBytes.Clone()
    $bytes[0x004260] = ($bytes[0x004260] + 1) % 256
    [System.IO.File]::WriteAllBytes($calDatPath, $bytes)
    $result = ((& $CalWriterPath Verify) -join "`n")
    if (-not (($result.IndexOf("[ERROR] Verification failed.") -ge 0) -and
              ($result.IndexOf("[NG] HousingMainColor3 ") -ge 0)))
    {
        return $false
    }

    return $true
}

# FAT 領域へのファイル書き込みが正しく処理されるか
$tests["FatPutFile"] =
{
    # 下準備、ルート以下の全ファイルを消去
    [void](& $CalWriterPath DeleteDirectory cal:/)

    $command = "PutFile"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $srcPath = "${dataDirectoryPath}\\NotExistFile.txt"
    $dstPath = "cal:/NotExistFile.txt"

    $message = "[ERROR] Failed to open the file: ${srcPath} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $srcPath $dstPath)))
    {
        return $false
    }

    $srcPath = "${dataDirectoryPath}\\random.txt"
    $dstPath = "cal:/random.txt"

    $message = "[SUCCESS] Done."

    if (-not ($message -ceq (& $CalWriterPath $command $srcPath $dstPath)))
    {
        return $false
    }

    return $true
}

# FAT 領域へのファイル読み込みが正しく処理されるか
$tests["FatGetFile"] =
{
    # 下準備、ルート以下の全ファイルを消去
    [void](& $CalWriterPath DeleteDirectory cal:/)

    $command = "GetFile"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $srcPath = "cal:/NotExistFile.txt"
    $dstPath = "${dataDirectoryPath}\\NotExistFile.txt"

    $message = "[ERROR] File not found in FAT : ${srcPath}"

    if (-not ($message -ceq (& $CalWriterPath $command $srcPath $dstPath)))
    {
        return $false
    }

    $dstPath = "${dataDirectoryPath}\\random.txt"
    $srcPath = "cal:/random.txt"

    $message = "[SUCCESS] Done."

    [void](& $CalWriterPath PutFile $dstPath $srcPath)
    if (-not ($message -ceq (& $CalWriterPath $command $srcPath $dstPath)))
    {
        Write-Host "c"
        return $false
    }

    return $true
}

# FAT 領域へのファイル削除が正しく処理されるか
$tests["FatDeleteFile"] =
{
    # 下準備、ルート以下の全ファイルを消去
    [void](& $CalWriterPath DeleteDirectory cal:/)

    $command = "DeleteFile"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $deletePath = "cal:/NotExistFile.txt"

    $message = "[ERROR] File not found in FAT : ${deletePath}"

    if (-not ($message -ceq (& $CalWriterPath $command $deletePath)))
    {
        return $false
    }

    $srcPath = "${dataDirectoryPath}\\random.txt"
    $dstPath = "cal:/random.txt"

    $message = "[SUCCESS] Done."

    [void](& $CalWriterPath PutFile $srcPath $dstPath)
    if (-not ($message -ceq (& $CalWriterPath $command $dstPath)))
    {
        return $false
    }

    return $true
}

# FAT 領域へのディレクトリ作成が正しく処理されるか
$tests["FatMakeDirectory"] =
{
    # 下準備、ルート以下の全ファイルを消去
    [void](& $CalWriterPath DeleteDirectory cal:/)

    $command = "MakeDirectory"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "cal:/NotExistDirectory/NotExistDirectory"

    $message = "[ERROR] Failed to create the directory: ${path} (002-0001)"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "cal:/CreatedDirectory"

    $message = "[SUCCESS] Done."

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    return $true
}

# FAT 領域へのディレクトリ削除が正しく処理されるか
$tests["FatDeleteDirectory"] =
{
    # 下準備、ルート以下の全ファイルを消去
    [void](& $CalWriterPath DeleteDirectory cal:/)

    $command = "DeleteDirectory"

    $message = "[ERROR] Invalid arguments count: 0"

    if (-not ($message -ceq (& $CalWriterPath $command)))
    {
        return $false
    }

    $path = "cal:/NotExistDirectory"

    $message = "[ERROR] Directory not found in FAT : ${path}"

    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    $path = "cal:/CreatedDirectory"

    $message = "[SUCCESS] Done."

    [void](& $CalWriterPath MakeDirectory $path)
    if (-not ($message -ceq (& $CalWriterPath $command $path)))
    {
        return $false
    }

    return $true
}

$xml = [xml]@'
<?xml version="1.0" encoding="UTF-8"?>
<testsuites>
  <testsuite></testsuite>
</testsuites>
'@

$classname = "CalWriterBinarySuite"

$failures = 0

$globalWatch = New-Object "System.Diagnostics.StopWatch"
$globalWatch.Start()

foreach ($key in $tests.Keys)
{
    Write-Host -n "${key}... "

    $localWatch = New-Object "System.Diagnostics.StopWatch"
    $localWatch.Start()

    $result = (& $tests[$key])

    $localWatch.Stop()

    $node = $xml.CreateElement("testcase")
    $node.SetAttribute("name", $key)
    $node.SetAttribute("status", "run")
    $node.SetAttribute("time", $localWatch.Elapsed.Milliseconds / 1000.0)
    $node.SetAttribute("classname", $classname)
    [void]$xml.testsuites.FirstChild.AppendChild($node)

    if ($result)
    {
        Write-Host "Pass"
    }
    else
    {
        Write-Host "Failed"
        $failures += 1
        $subnode = $xml.CreateElement("failure")
        $subnode.SetAttribute(
            "message",
            $tests[$key].File +
            ":" +
            $tests[$key].StartPosition.StartLine +
            " ${key} failed")
        $subnode.SetAttribute("type", "")
        [void]$node.AppendChild($subnode)
    }
}

$globalWatch.Stop()

$node = $xml.testsuites
$node.SetAttribute("name", "AllTests")
$node.SetAttribute("tests", $tests.Count)
$node.SetAttribute("failures", $failures)
$node.SetAttribute("disabled", "0")
$node.SetAttribute("errors", "0")
$node.SetAttribute("timestamp", (Get-Date -format s))
$node.SetAttribute("time", $globalWatch.Elapsed.Milliseconds / 1000.0)

$node = $node.FirstChild
$node.SetAttribute("name", $classname)
$node.SetAttribute("tests", $tests.Count)
$node.SetAttribute("failures", $failures)
$node.SetAttribute("disabled", "0")
$node.SetAttribute("errors", "0")
$node.SetAttribute("time", $globalWatch.Elapsed.Milliseconds / 1000.0)

$xml.Save($TestDetailXmlPath)

Write-Host "Done."

exit $failures
