﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using Xunit;

namespace PropertySheetsTest
{
    /// <summary>
    /// 構築時に与えられた変数名から値へのマップの通り、$(変数名) というパターンで記述された文字列を展開します。
    /// </summary>
    public class Expander
    {
        private IReadOnlyDictionary<string, string> expandMap;
        private static readonly Regex variableRegex = new Regex(@"\$\(([\w\d_]+)\)");

        public Expander(IReadOnlyDictionary<string, string> expandMap)
        {
            this.expandMap = expandMap;
        }

        public string Expand(string s)
        {
            return variableRegex.Replace(s, m => GetValue(m.Groups[1].Value));
        }

        private string GetValue(string key)
        {
            string value;
            if (!expandMap.TryGetValue(key, out value))
            {
                throw new KeyNotFoundException(string.Format("{0}: キーがマップ中に存在しません。", key));
            }

            return value;
        }
    }

    public class ExpanderTest
    {
        [Fact]
        public void ExpandTest()
        {
            var expander = new Expander(new Dictionary<string, string>()
            {
                { "hoge", "fuga" },
            });

            Assert.Equal("hoge", expander.Expand("hoge"));
            Assert.Equal("fugahoge", expander.Expand("$(hoge)hoge"));
        }

        [Fact]
        public void ExpandKeyNotFoundTest()
        {
            var expander = new Expander(new Dictionary<string, string>()
            {
                { "hoge", "fuga" },
            });

            Assert.Throws<KeyNotFoundException>(() => expander.Expand("$(fuga)"));
        }
    }
}
