﻿using Microsoft.Build.Evaluation;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace PropertySheetsTest
{
    /// <summary>
    /// 特定の ConfigurationPair に対して評価された VC++ プロジェクトを表します。
    /// </summary>
    public class EvaluatedVcProject
    {
        private Project project;

        public EvaluatedVcProject(
            string path,
            ConfigurationPair configuraionPair,
            string vsVersion,
            IDictionary<string, string> globalProperties = null)
        {
            this.project = EvaluateVcProject(path, configuraionPair, vsVersion, globalProperties);
            this.Path = path;
            this.Properties = this.project.Properties.ToDictionary(p => p.Name, p => p.EvaluatedValue);
            this.Metadata = this.project.ItemDefinitions
                .SelectMany(x => x.Value.Metadata)
                .ToDictionary(MakeMetadataKey, m => m.EvaluatedValue);
            this.ImportFullPaths = this.project.Imports
                .Select(x => x.ImportedProject.FullPath)
                .ToArray();
        }

        public string Path { get; private set; }
        public IReadOnlyDictionary<string, string> Properties { get; private set; }
        public IReadOnlyDictionary<string, string> Metadata { get; private set; }
        public IEnumerable<string> ImportFullPaths { get; private set; }

        private static string MakeMetadataKey(ProjectMetadata m)
        {
            return string.Format("{0}.{1}", m.ItemType, m.Name);
        }

        private static Project EvaluateVcProject(
            string path,
            ConfigurationPair configurationPair,
            string vsVersion,
            IDictionary<string, string> globalProperties = null)
        {
            var finalGlobalProperties = new Dictionary<string, string>
            {
                { "Configuration", configurationPair.Configuration },
                { "Platform", configurationPair.Platform },
                { "VisualStudioVersion", vsVersion }
            };

            if (globalProperties != null)
            {
                foreach (var key in globalProperties.Keys)
                {
                    finalGlobalProperties[key] = globalProperties[key];
                }
            }

            return new Project(
                path,
                finalGlobalProperties,
                GetToolsVersion(vsVersion),
                vsVersion,
                ProjectCollection.GlobalProjectCollection,
                ProjectLoadSettings.Default);
        }

        private static string GetToolsVersion(string vsVersion)
        {
            switch (vsVersion)
            {
                case "11.0":
                    return "4.0";
                case "12.0":
                case "14.0":
                case "15.0":
                    return vsVersion;
                default:
                    throw new ArgumentException($"不明な Visual Studio バージョンです: {vsVersion}");
            }
        }
    }
}
