﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Assert.h>

#include <nn/audio.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/fs/fs_Result.h>
#include <nn/audio/audio_PolyphaseResampler.h>
#include <nn/nn_Log.h>

#include "testAudio_Fft.h"

#include <memory> // std::unique_ptr
#include <string>
#include <cmath>

namespace
{
    char g_WorkBuffer[1024 * 1024];
    const int32_t   defaultAmplitude = 0x4000;

    bool    resampleAndAnalyzeParameterSet(uint32_t numberOfChannels, const double* frequencies,
                                           uint32_t interpolationFactor, uint32_t decimationFactor,
                                           bool highQuality, float gain = 1.0f) NN_NOEXCEPT;

    void    fillBufferWithCosSignal(int16_t* outputBuffer, uint32_t pitch, uint32_t index, uint32_t sampleCount,
                                    double frequency, int32_t amplitude = defaultAmplitude) NN_NOEXCEPT
    {
        uint32_t    loopIndex;
        uint32_t    bufferIndex;
        double      factor;

        factor = 2.0 * piApproximateValue / (double)numberOfPoints;
        bufferIndex = index;
        for(loopIndex = 0; loopIndex < sampleCount; loopIndex++)
        {
            double  argument;
            double  value;

            argument = factor * frequency * (double)loopIndex;
            value = cos(argument);
            outputBuffer[bufferIndex] = static_cast<int16_t>((value * (double)amplitude) + 0.5);
            bufferIndex += pitch;
        }
    }

    typedef struct {
        uint32_t    peakIndex;
        int32_t     peakAmplitude;
        int32_t     noiseAmplitude;
        float       filterGain;
    } analyzerResultsType;

    void    analyzeBuffer(analyzerResultsType* pResults, int16_t* buffer, uint32_t pitch, uint32_t index,
                          int32_t amplitude = defaultAmplitude) NN_NOEXCEPT
    {
        uint32_t    loopIndex;
        uint32_t    bufferIndex;

        calculateFft(buffer, pitch, index);
        pResults->peakIndex = static_cast<uint32_t>(~0);
        pResults->peakAmplitude = 0;
        pResults->noiseAmplitude = 0;
        pResults->filterGain = 0.0f;
        bufferIndex = index;
        for(loopIndex = 0; loopIndex < (numberOfPoints / 2); loopIndex++)
        {
            if(static_cast<int32_t>(buffer[bufferIndex]) > pResults->peakAmplitude)
            {
                pResults->peakAmplitude = static_cast<int32_t>(buffer[bufferIndex]);
                pResults->peakIndex = loopIndex;
            }
            bufferIndex += pitch;
        }
        bufferIndex = index + pitch;
        for(loopIndex = 1; loopIndex < (numberOfPoints / 2); loopIndex++)
        {
            if(loopIndex != pResults->peakIndex)
            {
                pResults->noiseAmplitude += static_cast<int32_t>(buffer[bufferIndex]);
            }
            bufferIndex += pitch;
        }
        if(pResults->peakIndex < (numberOfPoints / 2))
        {
            pResults->filterGain = static_cast<float>(pResults->peakAmplitude) / static_cast<float>(amplitude);
        }
    }

    bool    resampleAndAnalyzeParameterSet(uint32_t numberOfChannels, const double* frequencies,
                                           uint32_t interpolationFactor, uint32_t decimationFactor,
                                           bool highQuality, float gain) NN_NOEXCEPT
    {
        nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
        nn::audio::PolyphaseResamplerType resampler;
        nn::audio::SampleFormat sampleFormat = nn::audio::SampleFormat_PcmInt16;
        const size_t resamplerBufferSize = nn::audio::GetRequiredBufferSizeForPolyphaseResampler(decimationFactor, interpolationFactor, numberOfChannels, highQuality);
        void* resamplerBuffer = allocator.Allocate(resamplerBufferSize, nn::audio::BufferAlignSize);
        nn::audio::InitializePolyphaseResampler(&resampler, resamplerBuffer, resamplerBufferSize, decimationFactor, interpolationFactor, numberOfChannels, highQuality, gain);
        const size_t inFrameSampleCount = 4096;
        const size_t inDataSize = inFrameSampleCount * numberOfChannels * nn::audio::GetSampleByteSize(sampleFormat);
        const size_t outDataSize = nn::audio::GetPolyphaseResamplerOutputSampleCount(&resampler, inFrameSampleCount) * numberOfChannels * nn::audio::GetSampleByteSize(sampleFormat);
        const size_t inBufferSize = nn::util::align_up(inDataSize, nn::audio::AudioOutBuffer::SizeGranularity);
        const size_t outBufferSize = nn::util::align_up(outDataSize, nn::audio::AudioOutBuffer::SizeGranularity);
        void* inBuffer = allocator.Allocate(inBufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        void* outBuffer = allocator.Allocate(outBufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        int outputSamples;
        bool        rval;
        uint32_t    loopIndex;
        int16_t*    outBufferForAnalysis;
        const uint32_t  chunkSize = 16;
        uint32_t    loopCount;
        uint32_t    outDataIndex;
        int16_t*    inDataPointer;
        int16_t*    outDataPointer;
        const uint32_t  transientStopIndex = 64;
        float       lastGainValue = 1.0f;
        const float allowedGainDeviation = 0.01f;
        const float maximumHighQualityNoiseLevel = 0.0002f;
        const float maximumLowQualityNoiseLevel = 0.05f;

        outBufferForAnalysis = static_cast<int16_t*>(outBuffer);
        outBufferForAnalysis = &outBufferForAnalysis[numberOfChannels * transientStopIndex];
        for(loopIndex = 0; loopIndex < numberOfChannels; loopIndex++)
        {
            fillBufferWithCosSignal(static_cast<int16_t*>(inBuffer), numberOfChannels, loopIndex, inFrameSampleCount, frequencies[loopIndex]);
        }
        loopCount = inFrameSampleCount / chunkSize;
        outDataIndex = 0;
        inDataPointer = static_cast<int16_t*>(inBuffer);
        outDataPointer = static_cast<int16_t*>(outBuffer);
        for(loopIndex = 0; loopIndex < loopCount; loopIndex++)
        {
            ProcessPolyphaseResamplerBuffer(&resampler,
                                            &outputSamples, &outDataPointer[outDataIndex], (outBufferSize - (outDataIndex * sizeof(int16_t))),
                                            &inDataPointer[loopIndex * chunkSize * numberOfChannels], chunkSize);
            outDataIndex += outputSamples * numberOfChannels;
        }
        for(loopIndex = 0; loopIndex < numberOfChannels; loopIndex++)
        {
            uint32_t            expectedPeakIndex;
            analyzerResultsType results;
            float               maximumNoiseLevel;
            float               noiseLevel;

            maximumNoiseLevel = (highQuality) ? maximumHighQualityNoiseLevel : maximumLowQualityNoiseLevel;
            expectedPeakIndex = static_cast<uint32_t>(((frequencies[loopIndex]) / static_cast<double>(interpolationFactor) *
                                                        static_cast<double>(decimationFactor)) + 0.5);
            analyzeBuffer(&results, outBufferForAnalysis, numberOfChannels, loopIndex);
            noiseLevel = static_cast<float>(results.noiseAmplitude) / static_cast<float>(defaultAmplitude);
            NN_LOG("\t\tfreq:%f\tl:%3d, m:%3d: --> %3d with gain %f and total noise of %f\n",
                   frequencies[loopIndex], interpolationFactor, decimationFactor,
                   results.peakIndex, results.filterGain, noiseLevel);
            lastGainValue = results.filterGain;
            if(expectedPeakIndex != results.peakIndex)
            {
                break;
            }
            if(noiseLevel > maximumNoiseLevel)
            {
                break;
            }
        }
        rval = (loopIndex == numberOfChannels);
        if((rval) && (numberOfChannels == 1) && (gain < 1.0f))
        {
            float   delta;

            delta = gain - lastGainValue;
            delta = (delta < 0.0f) ? -delta : delta;
            if((delta / gain) > allowedGainDeviation)
            {
                rval = false;
            }
        }
        allocator.Free(resamplerBuffer);
        allocator.Free(outBuffer);
        allocator.Free(inBuffer);
        return rval;
    }

    const double    frequencySet1For6Channel[6] = {75.0, 100.0, 125.0, 150.0, 175.0, 200.0};
    const uint32_t  interpolationFactor1 = 25;
    const uint32_t  decimationFactor1 = 24;

    const double    frequencySet2For2Channel[2] = {112.0, 224.0};
    const uint32_t  interpolationFactor2 = 56;
    const uint32_t  decimationFactor2 = 51;

    const double    frequencySet3For1Channel[1] = {160.0};
    const uint32_t  interpolationFactor3 = 160;
    const uint32_t  decimationFactor3 = 147;

    const double    frequencySet4For1Channel[1] = {147.0};
    const uint32_t  interpolationFactor4 = 147;
    const uint32_t  decimationFactor4 = 160;

    const double    frequencySet5For6Channel[6] = {68.0, 88.0, 108.0, 128.0, 148.0, 168.0};
    const double    frequencySet6For6Channel[6] = {12.0, 18.0, 24.0, 30.0, 36.0, 42.0};
    const uint32_t  interpolationFactor5 = 2;
    const uint32_t  decimationFactor5 = 3;

    const double    frequencySet7For6Channel[6] = {136.0544217687075,
                                                   130.6122448979592,
                                                   125.1700680272109,
                                                   119.7278911564626,
                                                   114.2857142857143,
                                                   108.8435374149660};
    const uint32_t  interpolationFactor7 = 24000;
    const uint32_t  decimationFactor7 = 44100;

    const double    frequencySet8For2Channel[2] = {54.421768707483, 5.4421768707483};
    const double    frequencySet9For2Channel[2] = {183.75, 4.59375};

    const double    maximumOutputRateMismatch = 0.01;

    bool    checkApproximateOutputRate(int inputSampleRate, int outputSampleRate)
    {
        bool    rval;
        double  approximateRate;
        double  error;

        approximateRate = nn::audio::CalculatePolyphaseResamplerActualOutputSampleRate(inputSampleRate, outputSampleRate);
        error = (approximateRate - static_cast<double>(outputSampleRate)) / static_cast<double>(outputSampleRate);
        if(error < 0.0)
        {
            error = -error;
        }
        rval = (error <= maximumOutputRateMismatch);
        NN_LOG("\tactual output rate %f.  Given rates: input %d, output %d\terror %f\n", approximateRate, inputSampleRate, outputSampleRate, error);
        return rval;
    }
}

std::size_t GenerateTestWave(void* data, int channel, int channelCount, int sampleRate, int frequency, int sampleCount)
{
    // Generating square wave
    const int WaveLength = sampleRate / frequency;
    const int16_t Amplitude = std::numeric_limits<int16_t>::max();
    int count = 0;

    int16_t* p = static_cast<int16_t*>(data);
    for (int i = 0; i < sampleCount; ++i)
    {
        p[i * channelCount + channel] = static_cast<int16_t>(count < (WaveLength / 2) ? Amplitude : -Amplitude);
        if (++count == WaveLength)
        {
            count = 0;
        }
    }
    return sampleCount * sizeof(int16_t);
}

std::size_t GenerateSineWave(void* data, int channel, int channelCount, int sampleRate, int frequency, int sampleCount)
{
    int16_t* p = static_cast<int16_t*>(data);
    NN_ABORT_UNLESS_NOT_NULL(p);
    const float Pi = 3.1415926535897932384626433f;
    for (auto i = 0; i < sampleCount; ++i)
    {
        p[i * channelCount + channel] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(Pi / 2 + (2 * Pi * frequency * i / sampleRate)));
    }
    return sampleCount * sizeof(int16_t);
}

TEST(PolyphaseResampler, Precondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::PolyphaseResamplerType resampler;
    int inSampleRate = 48000;
    int outSampleRate = 32000;
    int channelCount = 1;

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetRequiredBufferSizeForPolyphaseResampler(0, 0, 0, true), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetRequiredBufferSizeForPolyphaseResampler(0, 0, 0, false), "");
    const size_t resamplerBufferSize = nn::audio::GetRequiredBufferSizeForPolyphaseResampler(inSampleRate, outSampleRate, channelCount, true);
    const size_t invalidResamplerBufferSize = nn::audio::GetRequiredBufferSizeForPolyphaseResampler(inSampleRate, outSampleRate, channelCount, true) - 150;
    void* resamplerBuffer = allocator.Allocate(resamplerBufferSize, nn::audio::BufferAlignSize);

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializePolyphaseResampler(nullptr, resamplerBuffer, resamplerBufferSize, inSampleRate, outSampleRate, channelCount, true, 1.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializePolyphaseResampler(&resampler, nullptr, resamplerBufferSize, inSampleRate, outSampleRate, channelCount, true, 1.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializePolyphaseResampler(&resampler, resamplerBuffer, invalidResamplerBufferSize, inSampleRate, outSampleRate, channelCount, true, 1.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializePolyphaseResampler(&resampler, resamplerBuffer, resamplerBufferSize, 0, outSampleRate, channelCount, true, 1.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializePolyphaseResampler(&resampler, resamplerBuffer, resamplerBufferSize, inSampleRate, 0, channelCount, true, 1.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializePolyphaseResampler(&resampler, resamplerBuffer, resamplerBufferSize, inSampleRate, outSampleRate, 0, true, 1.0f), "");

    nn::audio::SampleFormat sampleFormat = nn::audio::SampleFormat_PcmInt16;
    const int frameRate = 20;                                   // 20fps
    const int inFrameSampleCount = inSampleRate / frameRate;    // 50msecs (in samples)
    const int outFrameSampleCount = outSampleRate / frameRate;  // +400;  // 50msecs (in samples)
    const size_t inDataSize = inFrameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    const size_t outDataSize = outFrameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    const size_t inBufferSize = nn::util::align_up(inDataSize, nn::audio::AudioOutBuffer::SizeGranularity);
    const size_t outBufferSize = nn::util::align_up(outDataSize, nn::audio::AudioOutBuffer::SizeGranularity);
    void* inBuffer = allocator.Allocate(inBufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
    void* outBuffer = allocator.Allocate(outBufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
    int outputSamples;

    memset(&resampler, 0, sizeof(resampler));
    EXPECT_DEATH_IF_SUPPORTED(ProcessPolyphaseResamplerBuffer(&resampler, &outputSamples, static_cast<int16_t*>(outBuffer), outBufferSize, static_cast<const int16_t*>(inBuffer), inFrameSampleCount), "");
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::InitializePolyphaseResampler(&resampler, resamplerBuffer, resamplerBufferSize, inSampleRate, outSampleRate, channelCount, true, 1.0f));

    EXPECT_DEATH_IF_SUPPORTED(ProcessPolyphaseResamplerBuffer(&resampler, &outputSamples, NULL, outBufferSize, static_cast<const int16_t*>(inBuffer), inFrameSampleCount), "");
    EXPECT_DEATH_IF_SUPPORTED(ProcessPolyphaseResamplerBuffer(&resampler, &outputSamples, static_cast<int16_t*>(outBuffer), outBufferSize, NULL, inFrameSampleCount), "");

    NNT_ASSERT_RESULT_SUCCESS(ProcessPolyphaseResamplerBuffer(&resampler, &outputSamples, static_cast<int16_t*>(outBuffer), outBufferSize, static_cast<const int16_t*>(inBuffer), inFrameSampleCount));
}

TEST(PolyphaseResampler, Success)
{
    initializeFft();

    NN_ASSERT(resampleAndAnalyzeParameterSet(6, frequencySet1For6Channel, interpolationFactor1, decimationFactor1, true));
    NN_ASSERT(resampleAndAnalyzeParameterSet(2, frequencySet2For2Channel, interpolationFactor2, decimationFactor2, true));
    NN_ASSERT(resampleAndAnalyzeParameterSet(1, frequencySet3For1Channel, interpolationFactor3, decimationFactor3, true));
    NN_ASSERT(resampleAndAnalyzeParameterSet(1, frequencySet4For1Channel, interpolationFactor4, decimationFactor4, true));
    NN_ASSERT(resampleAndAnalyzeParameterSet(6, frequencySet5For6Channel, interpolationFactor5, decimationFactor5, true));
    NN_ASSERT(resampleAndAnalyzeParameterSet(6, frequencySet6For6Channel, interpolationFactor5, decimationFactor5, true));
    NN_ASSERT(resampleAndAnalyzeParameterSet(6, frequencySet7For6Channel, interpolationFactor7, decimationFactor7, true));
    NN_ASSERT(resampleAndAnalyzeParameterSet(2, frequencySet8For2Channel, interpolationFactor3, decimationFactor3, true));
    NN_ASSERT(resampleAndAnalyzeParameterSet(2, frequencySet9For2Channel, interpolationFactor4, decimationFactor4, true));
    NN_ASSERT(resampleAndAnalyzeParameterSet(1, frequencySet9For2Channel, interpolationFactor4, decimationFactor4, true));

    NN_ASSERT(resampleAndAnalyzeParameterSet(6, frequencySet1For6Channel, interpolationFactor1, decimationFactor1, false));
    NN_ASSERT(resampleAndAnalyzeParameterSet(2, frequencySet2For2Channel, interpolationFactor2, decimationFactor2, false));
    NN_ASSERT(resampleAndAnalyzeParameterSet(1, frequencySet3For1Channel, interpolationFactor3, decimationFactor3, false));
    NN_ASSERT(resampleAndAnalyzeParameterSet(1, frequencySet4For1Channel, interpolationFactor4, decimationFactor4, false));
    NN_ASSERT(resampleAndAnalyzeParameterSet(6, frequencySet5For6Channel, interpolationFactor5, decimationFactor5, false));
    NN_ASSERT(resampleAndAnalyzeParameterSet(6, frequencySet6For6Channel, interpolationFactor5, decimationFactor5, false));
    NN_ASSERT(resampleAndAnalyzeParameterSet(6, frequencySet7For6Channel, interpolationFactor7, decimationFactor7, false));
    NN_ASSERT(resampleAndAnalyzeParameterSet(2, frequencySet8For2Channel, interpolationFactor3, decimationFactor3, false));
    NN_ASSERT(resampleAndAnalyzeParameterSet(2, frequencySet9For2Channel, interpolationFactor4, decimationFactor4, false));
    NN_ASSERT(resampleAndAnalyzeParameterSet(1, frequencySet9For2Channel, interpolationFactor4, decimationFactor4, false));

    NN_ASSERT(resampleAndAnalyzeParameterSet(1, frequencySet3For1Channel, interpolationFactor3, decimationFactor3, true, 0.33f));
    NN_ASSERT(resampleAndAnalyzeParameterSet(1, frequencySet4For1Channel, interpolationFactor4, decimationFactor4, true, 0.444f));
    NN_ASSERT(resampleAndAnalyzeParameterSet(1, frequencySet3For1Channel, interpolationFactor3, decimationFactor3, false, 0.777f));
    NN_ASSERT(resampleAndAnalyzeParameterSet(1, frequencySet4For1Channel, interpolationFactor4, decimationFactor4, false, 0.86333f));

    destroyFft();

    int inputRate = 48000;
    int outputRate = 44099;
    NN_ASSERT(checkApproximateOutputRate(inputRate, outputRate));
    outputRate++;
    NN_ASSERT(checkApproximateOutputRate(inputRate, outputRate));
    outputRate++;
    NN_ASSERT(checkApproximateOutputRate(inputRate, outputRate));
    outputRate = 48000;
    inputRate = 66150;
    NN_ASSERT(checkApproximateOutputRate(inputRate, outputRate));
}

