﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAudio_Fft.h"
#include <cmath>

namespace {

    typedef struct {
        double  cosValue;
        double  sinValue;
    } cosSinType;

    typedef struct {
        double  real;
        double  img;
    } complexType;

    typedef struct {
        cosSinType*     coefficients;
        complexType*    buffer;
        uint32_t*       indexTable;
        uint32_t        log2N;
        uint32_t        numberOfPoints;
    } fftStateType;

    uint32_t    indexTable[numberOfPoints];
    cosSinType  cosSinTable[numberOfPoints / 2];
    complexType workBuffer[numberOfPoints];

    void    makeIndexTable(fftStateType* pState) NN_NOEXCEPT
    {
        uint32_t    indexI;
        uint32_t    indexJ;
        uint32_t    dstMask;
        uint32_t    srcMask;
        uint32_t    accumulator;

        for(indexI = 0; indexI < pState->numberOfPoints; indexI++) {
            dstMask = 1;
            srcMask = 1 << (pState->log2N - 1);
            accumulator = 0;
            for(indexJ = 0; indexJ < pState->log2N; indexJ++) {
                accumulator |= (indexI & srcMask) ? dstMask : 0;
                dstMask <<= 1;
                srcMask >>= 1;
            }
            pState->indexTable[indexI] = accumulator;
        }
    }

    void    makeCoefficientTable(fftStateType* pState) NN_NOEXCEPT
    {
        uint32_t    index;
        uint32_t    count;
        double      argument;
        double      increment;

        count = pState->numberOfPoints / 2;
        increment = piApproximateValue / static_cast<double>(count);
        argument = 0.0;
        for(index = 0; index < count; index++) {
            pState->coefficients[index].cosValue = cos(argument);
            pState->coefficients[index].sinValue = sin(argument);
            argument += increment;
        }
    }

    int32_t prepareS16InputBuffer(int16_t* inputBuffer, uint32_t pitch, fftStateType* pState) NN_NOEXCEPT
    {
        uint32_t    index;

        for(index = 0; index < pState->numberOfPoints; index++) {
            pState->buffer[index].real = static_cast<double>(*inputBuffer);
            pState->buffer[index].img = 0.0;
            inputBuffer = &inputBuffer[pitch];
        }
        return 0;
    }

    int32_t prepareS16OutputBuffer(int16_t* outputBuffer, uint32_t pitch, fftStateType* pState) NN_NOEXCEPT
    {
        double          scaleFactor;
        double          absoluteAmplitude;
        uint32_t        index;
        complexType*    pSrc;

        scaleFactor = 1.0 / static_cast<double>(1 << (pState->log2N - 1));
        for(index = 0; index < pState->numberOfPoints; index++) {
            pSrc = &pState->buffer[pState->indexTable[index]];
            absoluteAmplitude = sqrt(pSrc->real * pSrc->real + pSrc->img * pSrc->img);
            absoluteAmplitude *= scaleFactor;
            absoluteAmplitude += 0.5;
            *outputBuffer = static_cast<int16_t>(absoluteAmplitude);
            outputBuffer = &outputBuffer[pitch];
        }
        return 0;
    }

    void    inPlaceTwoPointDft(complexType* pArg1, complexType* pArg2) NN_NOEXCEPT
    {
        double  tmpReal;
        double  tmpImg;

        tmpReal = pArg1->real + pArg2->real;
        tmpImg = pArg1->img + pArg2->img;
        pArg2->real = pArg1->real - pArg2->real;
        pArg2->img = pArg1->img - pArg2->img;
        pArg1->real = tmpReal;
        pArg1->img = tmpImg;
    }

    void    complexMultiply(complexType* pDst, cosSinType* pSrc) NN_NOEXCEPT
    {
        double  tmpReal;
        double  tmpImg;

        tmpReal = pSrc->cosValue * pDst->real - pSrc->sinValue * pDst->img;
        tmpImg = pSrc->cosValue * pDst->img + pSrc->sinValue * pDst->real;
        pDst->real = tmpReal;
        pDst->img = tmpImg;
    }

    void    computeFft(fftStateType* pState) NN_NOEXCEPT
    {
        uint32_t    indexI;
        uint32_t    indexJ;
        uint32_t    indexK;
        uint32_t    counterJ;
        uint32_t    counterK;

        counterK = pState->numberOfPoints / 2;
        counterJ = 1;
        for(indexI = 0; indexI < pState->log2N; indexI++) {
            for(indexJ = 0; indexJ < counterJ; indexJ++) {
                for(indexK = 0; indexK < counterK; indexK++) {
                    inPlaceTwoPointDft(&pState->buffer[indexJ * 2 * counterK + indexK],
                                       &pState->buffer[(indexJ * 2 + 1) * counterK + indexK]);
                    complexMultiply(&pState->buffer[(indexJ * 2 + 1) * counterK + indexK],
                                    &pState->coefficients[indexK * counterJ]);
                }
            }
            counterK >>= 1;
            counterJ <<= 1;
        }
    }
}

int32_t initializeFft() NN_NOEXCEPT
{
    uint32_t        rval;
    fftStateType    state;

    rval = 0;
    state.coefficients = cosSinTable;
    state.buffer = workBuffer;
    state.indexTable = indexTable;
    state.log2N = log2N;
    state.numberOfPoints = numberOfPoints;
    makeIndexTable(&state);
    makeCoefficientTable(&state);
    return rval;
}

void    destroyFft() NN_NOEXCEPT
{
}

void    calculateFft(int16_t* pBuffer, uint32_t pitch, uint32_t index) NN_NOEXCEPT
{
    fftStateType    state;

    state.coefficients = cosSinTable;
    state.buffer = workBuffer;
    state.indexTable = indexTable;
    state.log2N = log2N;
    state.numberOfPoints = numberOfPoints;

    if(pBuffer != nullptr) {
        prepareS16InputBuffer(&pBuffer[index], pitch, &state);
        computeFft(&state);
        prepareS16OutputBuffer(&pBuffer[index], pitch, &state);
    }
}
