﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAudio_Effect.h"

#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/fs/fs_Result.h>

#include <memory> // std::unique_ptr

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/audio_EffectCommon.h" // QF_FRACTIONAL_BIT_COUNT
#include <cmath> //pow

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/audio/audio_AudioRendererApi-os.win32.h>
#endif
namespace
{
    static const int QF_FRACTIONAL_BIT_COUNT = 14;
    const std::string g_AudioTestSourceFileName = "/shutter.wav";
    const std::string g_AudioTestI3dl2Reverb1chFileName = "/testAudio_Effect_I3dl2Reverb1chShutter.wav";
    const std::string g_AudioTestI3dl2Reverb2chFileName = "/testAudio_Effect_I3dl2Reverb2chShutter.wav";
    const std::string g_AudioTestI3dl2Reverb6chFileNameBroken[] = { "/testAudio_Effect_I3dl2Reverb6chShutterA.wav",
        "/testAudio_Effect_I3dl2Reverb6chShutterB.wav", "/testAudio_Effect_I3dl2Reverb6chShutterC.wav" };
    const std::string g_AudioTestI3dl2Reverb4chFileName = "/testAudio_Effect_I3dl2Reverb4chShutter.wav";
    const std::string g_AudioTestI3dl2Reverb6chFileName = "/testAudio_Effect_I3dl2Reverb6chShutter.wav";
    const float g_AudioTestQfPrecision = 1.0f / std::powf(2, static_cast<float>(QF_FRACTIONAL_BIT_COUNT));

    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer[1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer2[1024 * 1024];

    char g_RecorderBuffer[16 * 1024 * 1024];
}
// ========================================================================================
// Test cases
// ========================================================================================

namespace nnt {
namespace audio {

using namespace nn::audio;

#if !defined(NN_SDK_BUILD_RELEASE)
namespace {
void ReverbDeathTests(I3dl2ReverbType* pReverb, float diffusion, float density, float hfReference, float gain, float ratio, nn::TimeSpan time, bool enable) NN_NOEXCEPT
{
    { // Death Check
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbRoomGain(pReverb, gain), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbRoomGain(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbRoomHfGain(pReverb, gain), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbRoomHfGain(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDecayTime(pReverb, time), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbLateReverbDecayTime(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbHfDecayRatio(pReverb, ratio), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbLateReverbHfDecayRatio(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbReflectionsGain(pReverb, gain), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbReflectionsGain(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbReflectionDelayTime(pReverb, time), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbReflectionDelayTime(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbReverbGain(pReverb, gain), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbReverbGain(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDelayTime(pReverb, time), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbLateReverbDelayTime(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDiffusion(pReverb, diffusion), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbLateReverbDiffusion(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDensity(pReverb, density), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbLateReverbDensity(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbHfReference(pReverb, hfReference), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbHfReference(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbDryGain(pReverb, gain), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbDryGain(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(IsI3dl2ReverbEnabled(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbEnabled(pReverb, enable), "");
        EXPECT_DEATH_IF_SUPPORTED(GetI3dl2ReverbChannelCountMax(pReverb), "");
    }
}
}; // namespace

TEST(TestI3dl2ReverbParameters, Preconditions)
{
    {
        float diffusion = 0.0f;
        float density = 0.0f;
        float hfReference = 21.0f;
        float gain = 0.0f;
        float ratio = 0.5f;
        nn::TimeSpan time = nn::TimeSpan::FromMilliSeconds(0);
        bool enable = false;

        // specifying nullptr to ReverbType*
        I3dl2ReverbType* pReverb = nullptr;
        ReverbDeathTests(pReverb, diffusion, density, hfReference, gain, ratio, time, enable);

        // specifying uninitialized ReverbType to ReverbType*
        I3dl2ReverbType reverb;
        ::std::memset(&reverb, 0xff, sizeof(reverb));
        ReverbDeathTests(&reverb, diffusion, density, hfReference, gain, ratio, time, enable);
    }

    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    EffectScopedRenderer renderer(allocator);
    int channels[] = {1, 2, 4, 6};
    int8_t input[] = {0, 1, 2, 3, 4, 5};
    int8_t output[] = {0, 1, 2, 3, 4, 5};

    for( auto j = 0; j < sizeof(channels) / sizeof(int);  ++j )
    {
        auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
        param.mixBufferCount = 1 + channels[j]; // SubMix(1) + FinalMix(channels[j])
        renderer.RendererSetup(&param, 1, channels[j]);
        nn::audio::MemoryPoolType effectPool;
        nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
        nn::audio::RequestAttachMemoryPool(&effectPool);

        I3dl2ReverbType reverb;
        I3dl2ReverbType* pReverb = &reverb;
        size_t reverbSize = GetRequiredBufferSizeForI3dl2Reverb(param.sampleRate, channels[j]);
        void* reverbBuffer = effectAllocator.Allocate(reverbSize);
        NN_ASSERT_NOT_NULL(reverbBuffer);
        NNT_ASSERT_RESULT_SUCCESS(AddI3dl2Reverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetFinalMix(), channels[j]));
        nn::audio::SetI3dl2ReverbInputOutput(&reverb, input, output, channels[j]);
        EXPECT_EQ(channels[j], nn::audio::GetI3dl2ReverbChannelCountMax(&reverb));

        {
            nn::TimeSpan invalidTime;
            invalidTime = nn::TimeSpan::FromMilliSeconds(99);
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDecayTime(pReverb, invalidTime) ,"");
            invalidTime = nn::TimeSpan::FromSeconds(21);
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDecayTime(pReverb, invalidTime), "");
            invalidTime = nn::TimeSpan::FromMilliSeconds(-1);
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbReflectionDelayTime(pReverb, invalidTime) ,"");
            invalidTime = nn::TimeSpan::FromMilliSeconds(301);
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbReflectionDelayTime(pReverb, invalidTime), "");
            invalidTime = nn::TimeSpan::FromMilliSeconds(-1);
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDelayTime(pReverb, invalidTime) ,"");
            invalidTime = nn::TimeSpan::FromMilliSeconds(101);
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDelayTime(pReverb, invalidTime), "");

            float val = -10000.1f;
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbRoomGain(pReverb, val), "");
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbRoomHfGain(pReverb, val), "");
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbReflectionsGain(pReverb, val), "");
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbReverbGain(pReverb, val), "");
            val = 0.1f;
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbRoomGain(pReverb, val), "");
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbRoomHfGain(pReverb, val), "");
            val = 1000.1f;
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbReflectionsGain(pReverb, val), "");
            val = 2000.1f;
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbReverbGain(pReverb, val), "");

            val = -0.1f;
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDiffusion(pReverb, val), "");
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDensity(pReverb, val), "");
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbDryGain(pReverb, val), "");
            val = 1.1f;
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbDryGain(pReverb, val), "");
            val = 100.1f;
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDiffusion(pReverb, val), "");
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbDensity(pReverb, val), "");

            val = 19.9f;
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbHfReference(pReverb, val), "");
            val = 20000.1f;
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbHfReference(pReverb, val), "");

            val = 0.0f;
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbHfDecayRatio(pReverb, val), "");
            val = 2.1f;
            EXPECT_DEATH_IF_SUPPORTED(SetI3dl2ReverbLateReverbHfDecayRatio(pReverb, val), "");
        }

        renderer.Start();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        renderer.Stop();
        renderer.RendererShutdown();

        effectAllocator.Free(reverbBuffer);
    }
}

TEST(TestI3dl2ReverbParameters, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    EffectScopedRenderer renderer(allocator);
    int channels[] = {1, 2, 4, 6};
    int8_t input[] = {0, 1, 2, 3, 4, 5};
    int8_t output[] = {0, 1, 2, 3, 4, 5};

    for( auto j = 0; j < sizeof(channels) / sizeof(int);  ++j )
    {
        AudioRendererParameter param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
        param.mixBufferCount = 1 + channels[j]; // SubMix(1) + FinalMix(channels[j])
        renderer.RendererSetup(&param, 1, channels[j]);
        nn::audio::MemoryPoolType effectPool;
        nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
        nn::audio::RequestAttachMemoryPool(&effectPool);

        I3dl2ReverbType reverb;
        I3dl2ReverbType* pReverb = &reverb;
        size_t reverbSize = GetRequiredBufferSizeForI3dl2Reverb(param.sampleRate, channels[j]);
        void* reverbBuffer = effectAllocator.Allocate(reverbSize);
        NN_ASSERT_NOT_NULL(reverbBuffer);
        NNT_ASSERT_RESULT_SUCCESS(AddI3dl2Reverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetFinalMix(), channels[j]));
        nn::audio::SetI3dl2ReverbInputOutput(&reverb, input, output, channels[j]);
        {
            nn::TimeSpan validTime;
            validTime = nn::TimeSpan::FromMilliSeconds(100);
            SetI3dl2ReverbLateReverbDecayTime(pReverb, validTime);
            EXPECT_NEAR(static_cast<float>(validTime.GetMilliSeconds()), static_cast<float>(GetI3dl2ReverbLateReverbDecayTime(&reverb).GetMilliSeconds()), g_AudioTestQfPrecision);

            validTime = nn::TimeSpan::FromMilliSeconds(100);
            SetI3dl2ReverbReflectionDelayTime(pReverb, validTime);
            EXPECT_NEAR(static_cast<float>(validTime.GetMilliSeconds()), static_cast<float>(GetI3dl2ReverbReflectionDelayTime(&reverb).GetMilliSeconds()), 1);

            validTime = nn::TimeSpan::FromMilliSeconds(20);
            SetI3dl2ReverbLateReverbDelayTime(pReverb, validTime);
            EXPECT_NEAR(static_cast<float>(validTime.GetMilliSeconds()), static_cast<float>(GetI3dl2ReverbLateReverbDelayTime(&reverb).GetMilliSeconds()), g_AudioTestQfPrecision);

            float val = -10000.0f;
            SetI3dl2ReverbRoomGain(pReverb, val);
            SetI3dl2ReverbRoomHfGain(pReverb, val);
            SetI3dl2ReverbReflectionsGain(pReverb, val);
            SetI3dl2ReverbReverbGain(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbRoomGain(pReverb), g_AudioTestQfPrecision);
            EXPECT_NEAR(val, GetI3dl2ReverbRoomHfGain(pReverb), g_AudioTestQfPrecision);
            EXPECT_NEAR(val, GetI3dl2ReverbReflectionsGain(pReverb), g_AudioTestQfPrecision);
            EXPECT_NEAR(val, GetI3dl2ReverbReverbGain(pReverb), g_AudioTestQfPrecision);
            val = 0.0f;
            SetI3dl2ReverbRoomGain(pReverb, val);
            SetI3dl2ReverbRoomHfGain(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbRoomGain(pReverb), g_AudioTestQfPrecision);
            EXPECT_NEAR(val, GetI3dl2ReverbRoomHfGain(pReverb), g_AudioTestQfPrecision);
            val = 1000.0f;
            SetI3dl2ReverbReflectionsGain(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbReflectionsGain(pReverb), g_AudioTestQfPrecision);
            val = 2000.0f;
            SetI3dl2ReverbReverbGain(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbReverbGain(pReverb), g_AudioTestQfPrecision);

            val = 0.0f;
            SetI3dl2ReverbLateReverbDiffusion(pReverb, val);
            SetI3dl2ReverbLateReverbDensity(pReverb, val);
            SetI3dl2ReverbDryGain(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbLateReverbDiffusion(pReverb), g_AudioTestQfPrecision);
            EXPECT_NEAR(val, GetI3dl2ReverbLateReverbDensity(pReverb), g_AudioTestQfPrecision);
            EXPECT_NEAR(val, GetI3dl2ReverbDryGain(pReverb), g_AudioTestQfPrecision);
            val = 1.0f;
            SetI3dl2ReverbDryGain(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbDryGain(pReverb), g_AudioTestQfPrecision);
            val = 100.0f;
            SetI3dl2ReverbLateReverbDiffusion(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbLateReverbDiffusion(pReverb), g_AudioTestQfPrecision);
            SetI3dl2ReverbLateReverbDensity(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbLateReverbDensity(pReverb), g_AudioTestQfPrecision);

            val = 20.0f;
            SetI3dl2ReverbHfReference(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbHfReference(pReverb), g_AudioTestQfPrecision);
            val = 20000.0f;
            SetI3dl2ReverbHfReference(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbHfReference(pReverb), g_AudioTestQfPrecision);

            val = 0.1f;
            SetI3dl2ReverbLateReverbHfDecayRatio(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbLateReverbHfDecayRatio(pReverb), g_AudioTestQfPrecision);
            val = 2.0f;
            SetI3dl2ReverbLateReverbHfDecayRatio(pReverb, val);
            EXPECT_NEAR(val, GetI3dl2ReverbLateReverbHfDecayRatio(pReverb), g_AudioTestQfPrecision);
        }

        renderer.Start();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        renderer.Stop();
        renderer.RendererShutdown();

        effectAllocator.Free(reverbBuffer);
    }
}
#endif // !defined(NN_SDK_BUILD_RELEASE)

TEST(TestI3dl2Reverb1chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestI3dl2Reverb1chFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestI3dl2Reverb1chFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(3);
    const int32_t RecorderBufferSize = 100;
    const int32_t ChannelCount = 1;
    const int32_t BitRate = 16;
    const size_t SendReturnBufferSize = GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::Recorder recorder;
    nnt::audio::util::WaveFile testWav;

    //fsUtil->InitializeFileSystem(c_MountName, c_MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    nn::audio::I3dl2ReverbType reverb;
    size_t reverbSize = GetRequiredBufferSizeForI3dl2Reverb(params.sampleRate, ChannelCount);
    void* reverbBuffer = allocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);

    nn::Result result = nn::audio::AddI3dl2Reverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetFinalMix(), 1);
    ASSERT_TRUE(result.IsSuccess());
    int8_t bus[] = { 0 };
    nn::audio::I3dl2ReverbParameterSet parameterSet;
    parameterSet = nn::audio::GetI3dl2ReverbParameters(&reverb);
    I3dl2ReverbType::Preset preset = I3dl2ReverbType::Preset::Preset_MetalCorridor;
    LoadI3dl2ReverbPreset(&parameterSet, preset);
    SetI3dl2ReverbParameters(&reverb, &parameterSet);
    nn::audio::SetI3dl2ReverbInputOutput(&reverb, bus, bus, sizeof(bus));
    nn::audio::SetI3dl2ReverbEnabled(&reverb, true);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer, returnBuffer, bufferSize);
    int8_t inputIndex[ChannelCount] = { 0 };
    int8_t outputIndex[ChannelCount] = { 0 };
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);

    // Attach AuxType to Recorder & WaveFile.
    ASSERT_TRUE(recorder.Prepare(&params, &aux, &testWav, SendReturnBufferSize, CheckDuration, g_RecorderBuffer, sizeof(g_RecorderBuffer)).IsSuccess());

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    recorder.Start();
    renderer.Start();

    // keep playing at least for recording period.
    while (recorder.Record())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
        renderer.Update();
    }

    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    recorder.Stop();
    testWav.Close();

    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    allocator.Finalize();

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);

} // NOLINT(impl/function_size)

TEST(TestI3dl2Reverb2chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestI3dl2Reverb2chFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestI3dl2Reverb2chFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(3);
    const int32_t RecorderBufferSize = 100;
    const int32_t ChannelCount = 2;
    const int32_t BitRate = 16;
    const size_t SendReturnBufferSize = GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::Recorder recorder;
    nnt::audio::util::WaveFile testWav;

    //fsUtil->InitializeFileSystem(c_MountName, c_MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    nn::audio::I3dl2ReverbType reverb;
    size_t reverbSize = GetRequiredBufferSizeForI3dl2Reverb(params.sampleRate, ChannelCount);
    void* reverbBuffer = allocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);

    nn::Result result = nn::audio::AddI3dl2Reverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetFinalMix(), 2);
    ASSERT_TRUE(result.IsSuccess());
    int8_t bus[] = {0, 1};
    nn::audio::I3dl2ReverbParameterSet parameterSet;
    parameterSet = nn::audio::GetI3dl2ReverbParameters(&reverb);
    I3dl2ReverbType::Preset preset = I3dl2ReverbType::Preset::Preset_MetalCorridor;
    LoadI3dl2ReverbPreset(&parameterSet, preset);
    SetI3dl2ReverbParameters(&reverb, &parameterSet);
    nn::audio::SetI3dl2ReverbInputOutput(&reverb, bus, bus, sizeof(bus));
    nn::audio::SetI3dl2ReverbEnabled(&reverb, true);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer,  returnBuffer, bufferSize);
    int8_t inputIndex[ChannelCount] = {0, 1};
    int8_t outputIndex[ChannelCount] = {0, 1};
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);

    // Attach AuxType to Recorder & WaveFile.
    ASSERT_TRUE(recorder.Prepare(&params, &aux, &testWav, SendReturnBufferSize, CheckDuration, g_RecorderBuffer, sizeof(g_RecorderBuffer)).IsSuccess());

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 1);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    recorder.Start();
    renderer.Start();

    // keep playing at least for recording period.
    while( recorder.Record() )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
        renderer.Update();
    }

    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    recorder.Stop();
    testWav.Close();

    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    allocator.Finalize();

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile, 1);

} // NOLINT(impl/function_size)

TEST(TestI3dl2Reverb4chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::os::Event event(nn::os::EventClearMode_AutoClear);
    nn::audio::SetFastRenderingMode(true, &event);
    EXPECT_TRUE(nn::audio::GetFastRenderingMode());
#endif
    std::string TargetFile(nnt::audio::util::GetResultDataDirectory() + g_AudioTestI3dl2Reverb4chFileName);
    std::string GoldenFile(g_RefernerceDataFolder + g_AudioTestI3dl2Reverb4chFileName);
    std::string SourceFile(g_SourceDataFolder + g_AudioTestSourceFileName);
    int testChannelCount = 4;

    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    nnt::audio::WaveComparisonTestBase testBase(parameter, static_cast<uint16_t>(testChannelCount));
    testBase.Initialize();
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(testBase.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int8_t bus[] = { 0, 1, 2, 3 };
    size_t reverbSize = GetRequiredBufferSizeForI3dl2Reverb(parameter.sampleRate, testChannelCount);
    void* reverbBuffer = allocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);
    {
        nn::audio::I3dl2ReverbType reverb;
        nn::Result result = nn::audio::AddI3dl2Reverb(testBase.GetConfig(), &reverb, reverbBuffer, reverbSize, testBase.GetFinalMix(), testChannelCount);
        ASSERT_TRUE(result.IsSuccess());

        nn::audio::I3dl2ReverbParameterSet parameterSet;
        parameterSet = nn::audio::GetI3dl2ReverbParameters(&reverb);
        I3dl2ReverbType::Preset preset = I3dl2ReverbType::Preset::Preset_MetalCorridor;
        LoadI3dl2ReverbPreset(&parameterSet, preset);
        SetI3dl2ReverbParameters(&reverb, &parameterSet);
        nn::audio::SetI3dl2ReverbInputOutput(&reverb, bus, bus, testChannelCount);
        nn::audio::SetI3dl2ReverbEnabled(&reverb, true);
    }

    testBase.PrepareRecording(TargetFile);
    testBase.PrepareSourceVoice(SourceFile);
    testBase.Start();
    while( testBase.Record() )
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::audio::TriggerRendering();
        event.Wait();
#endif
        testBase.Wait();
        testBase.Update();
    }

    testBase.Finalize();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // AudioRenderer がクローズされると FastRenderingMode は解除
    EXPECT_FALSE(nn::audio::GetFastRenderingMode());
#endif
    allocator.Free(reverbBuffer);
    allocator.Finalize();

    // 波形比較
    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile, 1);
}

TEST(TestI3dl2Reverb6chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::os::Event event(nn::os::EventClearMode_AutoClear);
    nn::audio::SetFastRenderingMode(true, &event);
    EXPECT_TRUE(nn::audio::GetFastRenderingMode());
#endif
    std::string TargetFile(nnt::audio::util::GetResultDataDirectory() + g_AudioTestI3dl2Reverb6chFileName);
    std::string GoldenFile(g_RefernerceDataFolder + g_AudioTestI3dl2Reverb6chFileName);
    std::string SourceFile(g_SourceDataFolder + g_AudioTestSourceFileName);
    int testChannelCount = 6;

    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    nnt::audio::WaveComparisonTestBase testBase(parameter, static_cast<uint16_t>(testChannelCount));
    testBase.Initialize();
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(testBase.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int8_t bus[] = { 0, 1, 2, 3, 4, 5 };
    size_t reverbSize = GetRequiredBufferSizeForI3dl2Reverb(parameter.sampleRate, static_cast<uint16_t>(testChannelCount));
    void* reverbBuffer = allocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);
    {
        nn::audio::I3dl2ReverbType reverb;
        nn::Result result = nn::audio::AddI3dl2Reverb(testBase.GetConfig(), &reverb, reverbBuffer, reverbSize, testBase.GetFinalMix(), testChannelCount);
        ASSERT_TRUE(result.IsSuccess());

        nn::audio::I3dl2ReverbParameterSet parameterSet;
        parameterSet = nn::audio::GetI3dl2ReverbParameters(&reverb);
        I3dl2ReverbType::Preset preset = I3dl2ReverbType::Preset::Preset_MetalCorridor;
        LoadI3dl2ReverbPreset(&parameterSet, preset);
        SetI3dl2ReverbParameters(&reverb, &parameterSet);
        nn::audio::SetI3dl2ReverbInputOutput(&reverb, bus, bus, testChannelCount);
        nn::audio::SetI3dl2ReverbEnabled(&reverb, true);
    }

    testBase.PrepareRecording(TargetFile);
    testBase.PrepareSourceVoice(SourceFile);
    testBase.Start();
    while( testBase.Record() )
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::audio::TriggerRendering();
        event.Wait();
#endif
        testBase.Wait();
        testBase.Update();
    }

    testBase.Finalize();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // AudioRenderer がクローズされると FastRenderingMode は解除
    EXPECT_FALSE(nn::audio::GetFastRenderingMode());
#endif
    allocator.Free(reverbBuffer);
    allocator.Finalize();

    // 波形比較
    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile, 1);
}

} // namespace audio
} // namespace nnt

TEST(TestI3dl2Reverb, ResourceLimit)
{
    const int EffectCount = 6;
    const int EffectChannelCount = 6;

    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.effectCount = EffectCount;
    parameter.mixBufferCount = EffectChannelCount;

    nnt::audio::util::ScopedAudioRenderer sar(parameter);
    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, EffectChannelCount);

    nn::audio::I3dl2ReverbType reverb[EffectCount + 1];

    auto size = nn::audio::GetRequiredBufferSizeForI3dl2Reverb(parameter.sampleRate, EffectChannelCount);
    auto buffer = allocator.Allocate(size, nn::audio::BufferAlignSize);
    EXPECT_TRUE(buffer != nullptr);

    for( auto i = 0; i < EffectCount; ++i )
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddI3dl2Reverb(&sar.GetConfig(), &reverb[i], buffer, size, &finalMix, EffectChannelCount));
    }
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultOutOfResource, nn::audio::AddI3dl2Reverb(&sar.GetConfig(), &reverb[EffectCount], buffer, size, &finalMix, EffectChannelCount));

    for( auto i = 0; i < EffectCount; ++i )
    {
        nn::audio::SetI3dl2ReverbEnabled(&reverb[i], false);
        EXPECT_TRUE(buffer == nn::audio::RemoveI3dl2Reverb(&sar.GetConfig(), &reverb[i], &finalMix));
    }

    allocator.Free(buffer);
}

TEST(TestI3dl2Reverb, SuccessAddRemoveManyTimes)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    const auto parameter = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    nnt::audio::EffectScopedRenderer renderer(allocator);
    const int channelCountMax = 4;
    auto reverbSize = nn::audio::GetRequiredBufferSizeForI3dl2Reverb(parameter.sampleRate, channelCountMax);

    renderer.RendererSetup(&parameter);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::I3dl2ReverbType reverb;
    void* reverbBuffer = effectAllocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);
    // Renderer を開始
    renderer.Start();
    renderer.Update();
    renderer.Wait();

    for( auto i = 0; i < 50; ++i ) // 50 は適当な数。深い意味はない。
    {
        const int channelCountTable[] = { 1, 2, 4, 6 };
        const int channelCount = channelCountTable[i % (sizeof(channelCountTable) / sizeof(channelCountTable[0]))];

        // reverb 用バッファに適当な値を書き込んで、ダーティな状態にしておく
        memset(reverbBuffer, i, reverbSize);

        // reverb を renderer に追加し Update を呼び出すことで、reverbBuffer が指すメモリを DSP が触っている状態にする
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddI3dl2Reverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetSubMix(), channelCount));
        renderer.Update();
        renderer.Wait();

        // あえて workBuffer を破壊しても停止することなく動作を続ける
        memset(reverbBuffer, i, reverbSize);
        nn::os::FlushDataCache(reverbBuffer, reverbSize);

        // 外すために無効化
        nn::audio::SetI3dl2ReverbEnabled(&reverb, false);

        // 外れるまで待機
        while( IsI3dl2ReverbRemovable(&reverb) == false )
        {
            renderer.Update();
            renderer.Wait();
        }

        // I3dl2Reverb を外す
        ASSERT_TRUE(nn::audio::RemoveI3dl2Reverb(renderer.GetConfig(), &reverb, renderer.GetSubMix()) == reverbBuffer);
    }

    renderer.RendererShutdown();
    effectAllocator.Free(reverbBuffer);
}

TEST(TestI3dl2Reverb, UpdateAudioRendererOnEachStep)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    const auto parameter = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    nnt::audio::EffectScopedRenderer renderer(allocator);
    const int channelCountMax = 6;
    auto reverbSize = nn::audio::GetRequiredBufferSizeForI3dl2Reverb(parameter.sampleRate, channelCountMax);

    renderer.RendererSetup(&parameter);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::I3dl2ReverbType reverb;
    void* reverbBuffer = effectAllocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);
    // Renderer を開始
    renderer.Start();
    renderer.Update();
    renderer.Wait();

    for( auto i = 0; i < 10; ++i ) // 10 は適当な数。深い意味はない。
    {
        const int8_t bus[] = { 0, 1, 2, 3, 4, 5 };
        const int channelCountTable[] = { 1, 2, 4, 6 };
        const int channelCount = channelCountTable[i % (sizeof(channelCountTable) / sizeof(channelCountTable[0]))];
        EXPECT_LE(channelCount, channelCountMax);

        NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddI3dl2Reverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetSubMix(), channelCount));
        renderer.Update();
        renderer.Wait();

        nn::audio::SetI3dl2ReverbInputOutput(&reverb, bus, bus, channelCount);
        renderer.Update();
        renderer.Wait();

        SetI3dl2ReverbLateReverbDecayTime(&reverb, nn::TimeSpan::FromMilliSeconds(100));
        renderer.Update();
        renderer.Wait();

        SetI3dl2ReverbReflectionDelayTime(&reverb, nn::TimeSpan::FromMilliSeconds(100));
        renderer.Update();
        renderer.Wait();

        SetI3dl2ReverbRoomGain(&reverb, -10000.0f);
        renderer.Update();
        renderer.Wait();

        SetI3dl2ReverbRoomHfGain(&reverb, -10000.0f);
        renderer.Update();
        renderer.Wait();

        SetI3dl2ReverbReflectionsGain(&reverb, -10000.0f);
        renderer.Update();
        renderer.Wait();

        SetI3dl2ReverbReverbGain(&reverb, 2000.0f);
        renderer.Update();
        renderer.Wait();

        SetI3dl2ReverbLateReverbDiffusion(&reverb, 0.0f);
        renderer.Update();
        renderer.Wait();

        SetI3dl2ReverbLateReverbDensity(&reverb, 0.0f);
        renderer.Update();
        renderer.Wait();

        SetI3dl2ReverbDryGain(&reverb, 0.0f);
        renderer.Update();
        renderer.Wait();

        SetI3dl2ReverbHfReference(&reverb, 20.f);
        renderer.Update();
        renderer.Wait();

        SetI3dl2ReverbLateReverbHfDecayRatio(&reverb, 0.1f);
        renderer.Update();
        renderer.Wait();

        // 外すために無効化
        nn::audio::SetI3dl2ReverbEnabled(&reverb, false);
        renderer.Update();
        renderer.Wait();

        // 外れるまで待機
        while( IsI3dl2ReverbRemovable(&reverb) == false )
        {
            renderer.Update();
            renderer.Wait();
        }

        // I3dl2Reverb を外す
        ASSERT_TRUE(nn::audio::RemoveI3dl2Reverb(renderer.GetConfig(), &reverb, renderer.GetSubMix()) == reverbBuffer);

        renderer.Update();
        renderer.Wait();
    }

    renderer.RendererShutdown();
    effectAllocator.Free(reverbBuffer);
} // NOLINT(readability/fn_size)
