﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory> //unique_ptr

#include <nnt.h>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>

#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>

#define NN_TEST_AUDIO_EXPECT_RANGE(value, begin, end)       \
    {                                                       \
        auto nn_test_audio_expect_range_value = (value);    \
        EXPECT_GE(nn_test_audio_expect_range_value, begin); \
        EXPECT_LT(nn_test_audio_expect_range_value, end);   \
    }

namespace nnt {
namespace audio {

const nn::audio::AudioRendererParameter g_SR48K_SampleCount240_Mix2_Sub1_Voice1_Sink1_Effect0 = { 48000, 240, 2, 1, 1, 1, 0, 0 };

class EffectScopedRenderer
{
protected:
    nn::audio::AudioRendererParameter m_Parameter;
    size_t m_SizeForEffect;
    size_t m_WorkBufferSize;
    void*  m_pWorkBuffer;
    void*  m_pConfigBuffer;

    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;
    nn::audio::FinalMixType m_FinalMix;
    nn::audio::SubMixType m_SubMix;
    nn::audio::DeviceSinkType m_DeviceSink;
    nn::mem::StandardAllocator m_AllocatorInstance;
    nn::mem::StandardAllocator& m_Allocator;
    nn::os::SystemEvent m_SystemEvent;
private:
    nn::audio::FinalMixType* GetMixImpl(nn::audio::FinalMixType)
    {
        return &m_FinalMix;
    };

    nn::audio::SubMixType* GetMixImpl(nn::audio::SubMixType)
    {
        return &m_SubMix;
    };

public:
    explicit EffectScopedRenderer(nn::mem::StandardAllocator& allocator);
    virtual ~EffectScopedRenderer();
    void RendererSetup(const nn::audio::AudioRendererParameter* param, const int subMixBufferCount = 1, const int finalMixBufferCount = 1);
    void Start();
    void Update();
    void Wait();
    void Stop();
    void RendererShutdown();
    nn::audio::AudioRendererHandle GetHandle();
    nn::audio::AudioRendererConfig* GetConfig();
    nn::audio::AudioRendererParameter* GetParam();
    nn::audio::FinalMixType* GetFinalMix();
    nn::audio::SubMixType* GetSubMix();

    template<typename T>
    T* GetMix()
    {
        return GetMixImpl(T());
    };
};

class WaveComparisonTestBase
{
public:
    NN_IMPLICIT WaveComparisonTestBase(nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault(), uint16_t channelCount = 2);

    void Initialize();
    nn::audio::SubMixType* GetSubMix();
    nn::audio::FinalMixType* GetFinalMix();
    nn::audio::AudioRendererConfig* GetConfig();
    int GetSampleRate();

    void PrepareRecording(std::string recordingFileName, nn::TimeSpan recordingDuration = nn::TimeSpan::FromSeconds(1), bool trimSilence = false);
    void PrepareSourceVoice(std::string sourceFile);

    void Start();
    bool Record();
    void Update();
    void Wait();
    void Finalize();

private:
    // Test Setting Parameters
    const uint16_t m_WaveFileChannelCount;
    const uint16_t m_WaveFileBitRate;
    const int32_t m_RecoderAudioFrameCount;
    nn::audio::AudioRendererParameter m_Parameter;
    nn::TimeSpan m_RecodingDuration;

    // test components
    nnt::audio::util::SimpleFsUtility m_FsUtil;
    nnt::audio::util::MinimalRenderer m_Renderer;
    nnt::audio::util::Recorder m_Recorder;
    nnt::audio::util::WaveFile m_RecordingWav;
    nn::mem::StandardAllocator m_Allocator;

    // Aux effect
    nn::audio::AuxType m_RecoderAux;
    nn::audio::MemoryPoolType m_RecorderPoolSend;
    nn::audio::MemoryPoolType m_RecorderPoolReturn;
    int32_t* m_SendBuffer;
    int32_t* m_ReturnBuffer;

    // input source voice
    nn::audio::VoiceType m_InputVoice;
    nn::audio::WaveBuffer m_InputWaveBuffer;
    nn::audio::MemoryPoolType m_SourcePool;
    void* m_InputWaveFileDataBuffer;

    void* m_RecorderBuffer;
    size_t m_RecorderBufferSize;
};

} // namespace audio
} // namespace nnt
