﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cmath>
#include <limits>
#include <algorithm>
#include <string>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>

#include <nn/audio.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>

#include <nn/settings/settings_DebugPad.h>

namespace {

    const int RenderRate = 32000;
    const int RenderCount = (RenderRate / 200);

    const int SeCount = 1;

    const char Title[] = "AudioReverbCrossfadeTest";

    const char* g_SeFileNames[SeCount] =
    {
        "asset:/AudioCommon/SampleBgm0-1ch.adpcm",
    };

    NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_EffectBufferPoolMemory[12 * 1024 * 1024];

    nn::mem::StandardAllocator g_Allocator;
    nn::mem::StandardAllocator g_WaveBufferAllocator;
    nn::mem::StandardAllocator g_EffectBufferAllocator;

    const int parameterSetCount = 2;

    struct ReverbParameterSet
    {
        nn::audio::ReverbType::EarlyMode earlyMode;
        float earlyGain;
        nn::TimeSpan predelayTime;
        nn::audio::ReverbType::LateMode lateMode;
        float lateGain;
        nn::TimeSpan decayTime;
        float highFrequencyDecayRatio;
        float coloration;
        float reverbGain;
        float outGain;
        float dryGain;
    };

    ReverbParameterSet reverbParameterSets[parameterSetCount] =
    {
        {
            nn::audio::ReverbType::EarlyMode_SmallRoom,
            0.7f,
            nn::TimeSpan::FromMilliSeconds(20),
            nn::audio::ReverbType::LateMode_Room,
            0.7f,
            nn::TimeSpan::FromSeconds(1),
            0.1f,
            0.8f,
            0.7f,
            1.0f,
            0.7f,
        },
        {
            nn::audio::ReverbType::EarlyMode_Hall,
            0.9f,
            nn::TimeSpan::FromMilliSeconds(2),
            nn::audio::ReverbType::LateMode_Hall,
            0.9f,
            nn::TimeSpan::FromSeconds(10),
            0.7f,
            0.1f,
            0.9f,
            0.0f,
            0.7f,
        }
    };

    char* g_MountRomCacheBuffer = NULL;
}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonX = nn::hid::KeyboardKey::X::Index;
    map.buttonY = nn::hid::KeyboardKey::Y::Index;
    map.buttonLeft = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NN_LOG("---------------------------------------------- \n");
    NN_LOG("%s Sample\n", Title);
    NN_LOG("---------------------------------------------- \n");
    NN_LOG("[Up][Down]      Jump to Reverb1 or Reverb2     \n");
    NN_LOG("[Left][Right]   Pan  to Reverb1 or Reverb2     \n");
    NN_LOG("[X][Y]          Incremental panning            \n");
    NN_LOG("[Start/Space]   Finish Sample Program          \n");
    NN_LOG("---------------------------------------------- \n");
}

float CrossfadeReverb(float delta, nn::audio::ReverbType* pReverbA, nn::audio::ReverbType* pReverbB)
{
    float outGainA = nn::audio::GetReverbOutGain(pReverbA);
    float outGainB = nn::audio::GetReverbOutGain(pReverbB);

    outGainA += delta;
    outGainB -= delta;

    if (outGainA < 0.0f || outGainB > 1.0f) {
        outGainA = 0.0f;
        outGainB = 1.0f;
    }
    if (outGainA > 1.0f || outGainB < 0.0f) {
        outGainA = 1.0f;
        outGainB = 0.0f;
    }

    nn::audio::SetReverbOutGain(pReverbA, outGainA);
    nn::audio::SetReverbOutGain(pReverbB, outGainB);

    if (outGainA <= 0.0f) {
        nn::audio::SetReverbEnabled(pReverbA, false);
    }
    else if (!nn::audio::IsReverbEnabled(pReverbA)) {
        nn::audio::SetReverbEnabled(pReverbA, true);
    }
    if (outGainB <= 0.0f) {
        nn::audio::SetReverbEnabled(pReverbB, false);
    }
    else if (!nn::audio::IsReverbEnabled(pReverbB)) {
        nn::audio::SetReverbEnabled(pReverbB, true);
    }

    NN_LOG("Reverb : Left - %f - Enabled=%s | Right - %f - Enabled=%s \n",
        outGainA,
        nn::audio::IsReverbEnabled(pReverbA) ? "true" : "false",
        outGainB,
        nn::audio::IsReverbEnabled(pReverbB) ? "true" : "false");

    return outGainA;
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_EffectBufferAllocator.Initialize(g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    InitializeFileSystem();
    InitializeHidDevices();

    float reverb = 0.0f;

    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 16;
    parameter.voiceCount = 1;
    parameter.subMixCount = 3;
    parameter.sinkCount = 1;
    parameter.effectCount = 4;

    // ミックスバッファとオーディオバスの関係を定義します。
    int channelCount = 2;
    int8_t mainBus[2] = { 4, 5 };
    int8_t subBusA[2] = { 2, 3 };
    int8_t subBusB[2] = { 0, 1 };

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
        );

    // これらのエフェクトで利用するバッファを AudioRenderer に登録します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;

    // レンダラを初期化します。
    nn::audio::AudioRendererHandle handle;
    nn::Result result = nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 8);
    nn::audio::SubMixType subMix0;
    nn::audio::AcquireSubMix(&config, &subMix0, parameter.sampleRate, 4);
    nn::audio::SubMixType subMix1;
    nn::audio::AcquireSubMix(&config, &subMix1, parameter.sampleRate, 4);

    // レンダラの出力先を用意します。
    nn::audio::DeviceSinkType deviceSink;
    result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::audio::SetSubMixDestination(&config, &subMix0, &finalMix);

    nn::audio::SetSubMixMixVolume(&subMix0, &finalMix, 0.7f, subBusA[0], mainBus[0]);
    nn::audio::SetSubMixMixVolume(&subMix0, &finalMix, 0.7f, subBusA[1], mainBus[1]);

    nn::audio::SetSubMixDestination(&config, &subMix1, &subMix0);

    nn::audio::SetSubMixMixVolume(&subMix1, &subMix0, 0.7f, subBusB[0], subBusA[0]);
    nn::audio::SetSubMixMixVolume(&subMix1, &subMix0, 0.7f, subBusB[1], subBusA[1]);

    nn::audio::MemoryPoolType effectBufferPool;
    bool rt = AcquireMemoryPool(&config, &effectBufferPool, g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    // メモリプールをアタッチします。設定は次の nn::audio::RequestUpdateAudioRenderer() が呼ばれた以降に反映されます。
    rt = RequestAttachMemoryPool(&effectBufferPool);
    NN_ABORT_UNLESS(rt);

    // Reverb エフェクトを用意します。
    nn::audio::ReverbType reverbA;
    nn::audio::ReverbType reverbB;
    size_t reverbASize = nn::audio::GetRequiredBufferSizeForReverb(parameter.sampleRate, channelCount);
    void* reverbABuffer = g_EffectBufferAllocator.Allocate(reverbASize);
    NN_ABORT_UNLESS(reverbABuffer != nullptr);
    size_t reverbBSize = nn::audio::GetRequiredBufferSizeForReverb(parameter.sampleRate, channelCount);
    void* reverbBBuffer = g_EffectBufferAllocator.Allocate(reverbBSize);
    NN_ABORT_UNLESS(reverbBBuffer != nullptr);

    // Reverb を FinalMix に追加します。
    result = nn::audio::AddReverb(&config, &reverbA, reverbABuffer, reverbASize, &subMix0, 2);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::audio::AddReverb(&config, &reverbB, reverbBBuffer, reverbBSize, &subMix1, 2);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // Reverb エフェクトへの入力・出力バッファを設定します。
    // ここでは、入力元のバッファにそのまま書き戻します。
    nn::audio::SetReverbInputOutput(&reverbA, subBusA, subBusA, sizeof(subBusA));
    nn::audio::SetReverbInputOutput(&reverbB, subBusB, subBusB, sizeof(subBusB));

    ReverbParameterSet* parameterSetA = &reverbParameterSets[0];
    ReverbParameterSet* parameterSetB = &reverbParameterSets[1];

    // 初期反射音への設定
    nn::audio::SetReverbEarlyMode(&reverbA, parameterSetA->earlyMode);
    nn::audio::SetReverbEarlyGain(&reverbA, parameterSetA->earlyGain);
    nn::audio::SetReverbPredelayTime(&reverbA, parameterSetA->predelayTime);
    // 後期残響音への設定
    nn::audio::SetReverbLateMode(&reverbA, parameterSetA->lateMode);
    nn::audio::SetReverbLateGain(&reverbA, parameterSetA->lateGain);
    nn::audio::SetReverbDecayTime(&reverbA, parameterSetA->decayTime);
    nn::audio::SetReverbHighFrequencyDecayRatio(&reverbA, parameterSetA->highFrequencyDecayRatio);
    nn::audio::SetReverbColoration(&reverbA, parameterSetA->coloration);
    // エフェクト全体的なゲイン調整
    nn::audio::SetReverbReverbGain(&reverbA, parameterSetA->reverbGain);
    nn::audio::SetReverbOutGain(&reverbA, parameterSetA->outGain);
    nn::audio::SetReverbDryGain(&reverbA, parameterSetA->dryGain);
    nn::audio::SetReverbEnabled(&reverbA, true);

    // 初期反射音への設定
    nn::audio::SetReverbEarlyMode(&reverbB, parameterSetB->earlyMode);
    nn::audio::SetReverbEarlyGain(&reverbB, parameterSetB->earlyGain);
    nn::audio::SetReverbPredelayTime(&reverbB, parameterSetB->predelayTime);
    // 後期残響音への設定
    nn::audio::SetReverbLateMode(&reverbB, parameterSetB->lateMode);
    nn::audio::SetReverbLateGain(&reverbB, parameterSetB->lateGain);
    nn::audio::SetReverbDecayTime(&reverbB, parameterSetB->decayTime);
    nn::audio::SetReverbHighFrequencyDecayRatio(&reverbB, parameterSetB->highFrequencyDecayRatio);
    nn::audio::SetReverbColoration(&reverbB, parameterSetB->coloration);
    // エフェクト全体的なゲイン調整
    nn::audio::SetReverbReverbGain(&reverbB, parameterSetB->reverbGain);
    nn::audio::SetReverbOutGain(&reverbB, parameterSetB->outGain);
    nn::audio::SetReverbDryGain(&reverbB, parameterSetB->dryGain);
    nn::audio::SetReverbEnabled(&reverbB, true);

    // 設定したパラメータをレンダラに反映させます。
    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // レンダリングを開始します。
    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::audio::MemoryPoolType waveBufferMemoryPool;
    rt = AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    // SE を読み込みます。
    nn::audio::VoiceType voiceSe[SeCount];
    nn::audio::WaveBuffer waveBufferSe[SeCount];
    nn::audio::AdpcmHeaderInfo* header = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo) * SeCount, nn::audio::BufferAlignSize));
    void* dataSe[SeCount];

    for (int i = 0; i < SeCount; ++i)
    {
        std::size_t dataSeSize = ReadAdpcmFile(&header[i], &dataSe[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &voiceSe[i], header[i].sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &header[i].parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&config, &voiceSe[i], &subMix1);

        waveBufferSe[i].buffer = dataSe[i];
        waveBufferSe[i].size = dataSeSize;
        waveBufferSe[i].startSampleOffset = 0;
        waveBufferSe[i].endSampleOffset = header[i].sampleCount;
        waveBufferSe[i].loop = true;
        waveBufferSe[i].isEndOfStream = false;
        waveBufferSe[i].pContext = &header[i].loopContext;
        waveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
        nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &subMix1, 0.707f / 2, 0, subBusB[0]);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &subMix1, 0.707f / 2, 0, subBusB[1]);
    }

    PrintUsage();

    int64_t prevPadSamplingNumber = 0;
    nn::hid::DebugPadState debugPadState[nn::hid::DebugPadStateCountMax];

    reverb = 1.0f;
    bool running = true;

    while (running)
    {
        systemEvent.Wait();

        // DebugPad の入力を取得します。
        // debugPad の状態を表す構造体を用意します。
        nn::hid::GetDebugPadStates(debugPadState, nn::hid::DebugPadStateCountMax);

        // 直前から現在までの Pad 入力の更新回数を取得します。
        int64_t padSamplingCount = debugPadState[0].samplingNumber - prevPadSamplingNumber;
        if (padSamplingCount >= nn::hid::DebugPadStateCountMax)
        {
            padSamplingCount = nn::hid::DebugPadStateCountMax - 1;
        }
        nn::hid::DebugPadButtonSet padButtonDown(debugPadState[0].buttons & ~debugPadState[padSamplingCount].buttons);

        // 現在までの Pad 入力の更新回数を取得します。
        prevPadSamplingNumber = debugPadState[0].samplingNumber;

        // 各 Effect のパラメータを更新します。
        {
            if (debugPadState[0].buttons.Test< ::nn::hid::DebugPadButton::Left >())
            {
                reverb = CrossfadeReverb(0.0005f, &reverbA, &reverbB);
            }
            if (debugPadState[0].buttons.Test< ::nn::hid::DebugPadButton::Right >())
            {
                reverb = CrossfadeReverb(-0.0005f, &reverbA, &reverbB);
            }
            if (padButtonDown.Test<::nn::hid::DebugPadButton::X>())
            {
                reverb = CrossfadeReverb(0.05f, &reverbA, &reverbB);
            }
            if (padButtonDown.Test<::nn::hid::DebugPadButton::Y>())
            {
                reverb = CrossfadeReverb(-0.05f, &reverbA, &reverbB);
            }
            if (padButtonDown.Test<::nn::hid::DebugPadButton::Up>())
            {
                reverb = CrossfadeReverb(1.0f, &reverbA, &reverbB);
            }
            if (padButtonDown.Test<::nn::hid::DebugPadButton::Down>())
            {
                reverb = CrossfadeReverb(-1.0f, &reverbA, &reverbB);
            }

            if (padButtonDown.IsAnyOn())
            {
            }
        }

        //SE を再生します（同じ SE を多重再生しません）。
        for (int i = 0; i < SeCount; ++i)
        {
            // SE番号 {0, 1, 2, 3, 4, 5} が、ボタン {A, B, X, Y, L, R} に対応します。
            if (debugPadState[0].buttons.Test(i) && !debugPadState[padSamplingCount].buttons.Test(i))
            {
                if (nn::audio::GetReleasedWaveBuffer(&voiceSe[i]) != nullptr)
                {
                    nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
                }
            }
        }

        // 各エフェクトに施したパラメータ変更をオーディオレンダラに反映します。
        // 反映するまでは、パラメータ変更は有効になりません。
        // nn::audio::RequestUpdateAudioRenderer() の呼び出し中は、各エフェクトへのアクセスは避けてください。
        result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        //  サンプルを終了する
        if (padButtonDown.Test<::nn::hid::DebugPadButton::Start>())
        {
            running = false;
        }
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを解放します。
    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_WaveBufferAllocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
        }
    }
    if (header)
    {
        g_WaveBufferAllocator.Free(header);
        header = nullptr;
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }
    if (reverbABuffer)
    {
        g_EffectBufferAllocator.Free(reverbABuffer);
        reverbABuffer = nullptr;
    }
    if (reverbBBuffer)
    {
        g_EffectBufferAllocator.Free(reverbBBuffer);
        reverbBBuffer = nullptr;
    }

    FinalizeFileSystem();
}  // NOLINT(readability/fn_size)
