﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>  // std::memcmp
#include <cstdlib>  // std::aligned_alloc
#include <mutex>

#include <nnt.h>
#include <nn/nn_Log.h>

#include <atomic>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>

#include <nnt/audioUtil/testAudio_Constants.h>
#include <nnt/audioUtil/testAudio_Util.h>

#include <nn/audio/audio_AudioRendererApi.private.h>
#include "testAudio_SimpleAudioRenderer.h"

namespace {

NN_ALIGNAS(4096) char g_WorkBuffer[1024 * 1024 * 2];  // 4096 == nn::os::MemoryPageSize

void SetDefaultParameter(nn::audio::AudioRendererParameter* pOutParameter)
{
    nn::audio::InitializeAudioRendererParameter(pOutParameter);
    pOutParameter->mixBufferCount = 6;
    pOutParameter->subMixCount = 1;
    pOutParameter->voiceCount = 24;
    pOutParameter->effectCount = 4;
}

const std::string g_AudioTestSourceFileName(g_SourceDataFolder + "/shutter.wav");

}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(MemoryPoolAPIs, preconditions)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    nnt::audio::util::ScopedAudioRenderer sar(parameter);
    sar.Start();

    nn::audio::MemoryPoolType pool;
    const int poolSize = nn::audio::MemoryPoolType::SizeGranularity * 2;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN uint8_t poolBuffer[poolSize];

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireMemoryPool(nullptr, &pool,  poolBuffer, poolSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireMemoryPool(&sar.GetConfig(), nullptr,  poolBuffer, poolSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool,  nullptr, poolSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool,  poolBuffer + 1, poolSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool,  poolBuffer, poolSize - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool,  poolBuffer, 0), "");

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseMemoryPool(nullptr, &pool), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseMemoryPool(&sar.GetConfig(), nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::IsMemoryPoolAttached(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetMemoryPoolSize(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestAttachMemoryPool(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestDetachMemoryPool(nullptr), "");

    pool._pMemoryPoolInfo = nullptr; // invalid address
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseMemoryPool(&sar.GetConfig(), &pool), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::IsMemoryPoolAttached(&pool), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetMemoryPoolSize(&pool), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestAttachMemoryPool(&pool), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestDetachMemoryPool(&pool), "");

    nn::audio::MemoryPoolType pool2;
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool,  poolBuffer, poolSize));
    // Since overlapped region passed.
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool2,  poolBuffer, poolSize), ""); // 同一区間
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool2,  poolBuffer + nn::audio::MemoryPoolType::SizeGranularity, poolSize), ""); // 上にはみ出す
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool2,  poolBuffer + poolSize, poolSize)); // 既存の境界と上端で接する
    nn::audio::ReleaseMemoryPool(&sar.GetConfig(), &pool2);
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool2,  poolBuffer - nn::audio::MemoryPoolType::SizeGranularity, poolSize), ""); // 下にはみ出す
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool2,  poolBuffer - poolSize, poolSize)); // 既存の境界と下端で接する
    nn::audio::ReleaseMemoryPool(&sar.GetConfig(), &pool2);
}
#else
TEST(MemoryPoolAPIs, preconditions)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    nnt::audio::util::ScopedAudioRenderer sar(parameter);
    sar.Start();

    nn::audio::MemoryPoolType pool;
    const int poolSize = nn::audio::MemoryPoolType::SizeGranularity * 2;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN uint8_t poolBuffer[poolSize];

    EXPECT_FALSE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool, poolBuffer, 0));
    EXPECT_EQ(pool._pMemoryPoolInfo, nullptr);
    EXPECT_FALSE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool, nullptr, poolSize));
    EXPECT_EQ(pool._pMemoryPoolInfo, nullptr);
}

#endif

TEST(MemoryPoolAPIs, Success)
{
    nn::os::SystemEvent event;
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    nnt::audio::util::ScopedAudioRenderer sar(parameter, &event);

    nn::audio::MemoryPoolType pool[3];
    const int poolSize = 4096;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN uint8_t poolBuffer0[poolSize];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN uint8_t poolBuffer1[poolSize];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN uint8_t poolBuffer2[poolSize];

    nn::audio::SubMixType submix;
    nn::audio::AcquireSubMix(&sar.GetConfig(), &submix, 48000, 1);

    sar.Start();

    nn::Result result;
    auto poolCount = nn::audio::GetReleasedMemoryPoolCount(&sar.GetConfig());
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool[0], poolBuffer0, poolSize));
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool[1], poolBuffer1, poolSize));
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool[2], poolBuffer2, poolSize));

    EXPECT_EQ(poolCount - 3, nn::audio::GetReleasedMemoryPoolCount(&sar.GetConfig()));

    EXPECT_EQ(nn::audio::GetMemoryPoolAddress(&pool[0]), poolBuffer0);
    EXPECT_EQ(nn::audio::GetMemoryPoolAddress(&pool[1]), poolBuffer1);
    EXPECT_EQ(nn::audio::GetMemoryPoolAddress(&pool[2]), poolBuffer2);

    EXPECT_EQ(nn::audio::GetMemoryPoolSize(&pool[0]), poolSize);
    EXPECT_EQ(nn::audio::GetMemoryPoolSize(&pool[1]), poolSize);
    EXPECT_EQ(nn::audio::GetMemoryPoolSize(&pool[2]), poolSize);

    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[0]));
    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[1]));
    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[2]));
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[1]), nn::audio::MemoryPoolType::State_Detached);

    //// MemoryPool の基本的な動作確認
    // Request Attach
    EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&pool[1]));
    EXPECT_FALSE(nn::audio::RequestAttachMemoryPool(&pool[1])); // 2 回目のコールは false を返す
    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1]));
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[1]), nn::audio::MemoryPoolType::State_RequestAttach);
    EXPECT_TRUE(nn::audio::RequestDetachMemoryPool(&pool[1])); // RequestAttach のキャンセル
    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[1])); // キャンセルしたので Detach 状態
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[1]), nn::audio::MemoryPoolType::State_Detached);

    // these two are not requested
    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[0]));
    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[2]));
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[0]), nn::audio::MemoryPoolType::State_Detached);
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[2]), nn::audio::MemoryPoolType::State_Detached);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[1])); // RequestAttach がキャンセルされているかの確認
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[1]), nn::audio::MemoryPoolType::State_Detached);
    EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&pool[1])); // 通常の Attach 要求
    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1]));
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[1]), nn::audio::MemoryPoolType::State_RequestAttach);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1])); // 通常の Attach 要求の成功確認
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[1]), nn::audio::MemoryPoolType::State_Attached);
    EXPECT_TRUE(nn::audio::RequestDetachMemoryPool(&pool[1]));
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[1]), nn::audio::MemoryPoolType::State_RequestDetach);
    EXPECT_FALSE(nn::audio::RequestDetachMemoryPool(&pool[1])); // 2 回目のコールは false を返す
    EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&pool[1])); // RequestDetach のキャンセル
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[1]), nn::audio::MemoryPoolType::State_Attached);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1])); // RequestDetach がキャンセルされているかの確認
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[1]), nn::audio::MemoryPoolType::State_Attached);
    EXPECT_TRUE(nn::audio::RequestDetachMemoryPool(&pool[1])); // 通常の Detach 要求
    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1])); // Detach については即座には 反映されず まだ true を返す
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[1]), nn::audio::MemoryPoolType::State_RequestDetach);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[1])); // 通常の Detach 要求の成功確認
    EXPECT_EQ(nn::audio::GetMemoryPoolState(&pool[1]), nn::audio::MemoryPoolType::State_Detached);

    //// 利用中の MemoryPool が Detach されない仕様の確認
    nn::audio::DelayType delay;
    nn::TimeSpan delayTime = nn::TimeSpan::FromMilliSeconds(1);
    auto delayWorkBufferSize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, 48000, 1);
    ASSERT_TRUE(delayWorkBufferSize < poolSize);
    nn::audio::AddDelay(&sar.GetConfig(), &delay, poolBuffer1, delayWorkBufferSize, &submix, delayTime, 1);
    nn::audio::SetDelayEnabled(&delay, true);
    EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&pool[1])); // Delay が利用する領域を Attach

    event.Wait(); // コマンド発行を待つ
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1]));
    EXPECT_TRUE(nn::audio::RequestDetachMemoryPool(&pool[1])); // Delay が使用中だが Detach 要求自体は成功する。
    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1]));

    event.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1])); // Delay を remove していないので Attach されたまま
    EXPECT_FALSE(nn::audio::RequestDetachMemoryPool(&pool[1])); // まだ Detach が完了していないので false

    nn::audio::SetDelayEnabled(&delay, false);
    while (nn::audio::IsDelayRemovable(&delay) != true)
    {
        event.Wait();
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    }

    nn::audio::RemoveDelay(&sar.GetConfig(), &delay, &submix);
    event.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    event.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[1])); // Delay が remove されたので detach される


    //// 利用中の MemoryPool への Detach がペンディングされている最中のキャンセルの挙動確認
    nn::audio::AddDelay(&sar.GetConfig(), &delay, poolBuffer1, delayWorkBufferSize, &submix, delayTime, 1);
    nn::audio::SetDelayEnabled(&delay, true);
    EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&pool[1])); // Delay が利用する領域を Attach

    event.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1]));
    EXPECT_TRUE(nn::audio::RequestDetachMemoryPool(&pool[1])); // Delay が使用中だが Detach 要求自体は成功する。
    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1]));

    event.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1])); // Delay を remove していないので Attach されたまま
    EXPECT_FALSE(nn::audio::RequestDetachMemoryPool(&pool[1])); // まだ Detach が完了していないので false
    EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&pool[1])); // Detach をキャンセル
    nn::audio::SetDelayEnabled(&delay, false);

    while (nn::audio::IsDelayRemovable(&delay) != true)
    {
        event.Wait();
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    }
    nn::audio::RemoveDelay(&sar.GetConfig(), &delay, &submix);

    event.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    for (auto i = 0; i < 10; ++i) // 数フレーム待つ
    {
        event.Wait();
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    }

    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1])); // Detach がキャンセルされているので、利用者がいなくなっても Attach のまま
    EXPECT_TRUE(nn::audio::RequestDetachMemoryPool(&pool[1])); // Detach 要求

    event.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[1])); // Detach 成功

    //// data corruption check
    {
        EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[0]));
        EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[2]));
        EXPECT_EQ(nn::audio::GetMemoryPoolAddress(&pool[0]), poolBuffer0);
        EXPECT_EQ(nn::audio::GetMemoryPoolAddress(&pool[2]), poolBuffer2);
        EXPECT_EQ(nn::audio::GetMemoryPoolSize(&pool[0]), poolSize);
        EXPECT_EQ(nn::audio::GetMemoryPoolSize(&pool[2]), poolSize);
    }

    EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&pool[1])); // 一度 detach されたプールも detach状態からなら attach 可能
    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1]));

    //// Voice の play state に応じた動作確認
    nn::audio::VoiceType voice;
    nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
    nn::audio::WaveBuffer waveBuffer;
    const int playPeriod = (48000 / 200) * 5; // 5ms * 5 fr
    const int sampleBufferSize = playPeriod * sizeof(int16_t);
    ASSERT_TRUE(delayWorkBufferSize + sampleBufferSize < poolSize) << delayWorkBufferSize;
    waveBuffer.buffer = poolBuffer1 + delayWorkBufferSize;
    waveBuffer.size = sampleBufferSize;
    waveBuffer.startSampleOffset = 0;
    waveBuffer.endSampleOffset = playPeriod;
    waveBuffer.loop = false;
    waveBuffer.pContext = nullptr;
    waveBuffer.contextSize = 0;
    memset(poolBuffer1 + delayWorkBufferSize, 0, waveBuffer.size);
    // 2 つの WaveBuffer を突っ込む
    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Stop); // stop 状態で追加する

    event.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_TRUE(nn::audio::RequestDetachMemoryPool(&pool[1])); // detach 要求

    event.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1])); // Voice が stop 状態なので再生されていない WaveBuffer が残ったまま
    EXPECT_FALSE(nn::audio::RequestDetachMemoryPool(&pool[1])); // 未だ Detach に成功していないので重複 Request となり false

    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

    while (nn::audio::GetReleasedWaveBuffer(&voice) == nullptr) // 再生完了を待つ
    {
        if (nn::audio::GetWaveBufferCount(&voice) > 0)
        {
            EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1])); // 待つ間、MemoryPool は解放されない
        }
        event.Wait();
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    }

    // タイミング依存で WaveBuffer の消費量が変動するため、
    // 確実に一つだけ回収された時のみ、 if 文内のテストを行う。
    if (nn::audio::GetReleasedWaveBuffer(&voice) == nullptr)
    {
        // 一つ目の WaveBuffer が回収完了。まだ attach 状態が期待される。
        EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1]));
    }

    while (nn::audio::GetReleasedWaveBuffer(&voice) == nullptr) // 再生完了を待つ
    {
        if (nn::audio::GetWaveBufferCount(&voice) > 0)
        {
            EXPECT_TRUE(nn::audio::IsMemoryPoolAttached(&pool[1])); // 待つ間、MemoryPool は解放されない
        }
        event.Wait();
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    }

    // RequestDetach 済なので、利用が完了したと同時に Detach されている。
    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[1]));

    event.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_FALSE(nn::audio::IsMemoryPoolAttached(&pool[1])); // Voice の再生が完了し Detach も完了
    nn::audio::ReleaseMemoryPool(&sar.GetConfig(), &pool[1]);
} // NOLINT(readability/fn_size)

class MemoryPoolFsAccessTest
{
    std::atomic_bool m_TestRunning;
    std::atomic_bool m_InitialLoad;
    void* m_DataBuffer = nullptr;
    size_t m_DataSize = 0;
    int m_SampleRate = 0;

    nn::os::Mutex m_Mutex;
    nnt::audio::util::SimpleFsUtility m_FsUtil;

    enum class TestState
    {
        AttachAndStart,
        Playing,
        Stop,
        Detach,
        Wait,
    };

public:
    MemoryPoolFsAccessTest()
        : m_DataBuffer(nullptr)
        , m_DataSize(0)
        , m_SampleRate(0)
        , m_Mutex(false)
    {
        m_TestRunning = true;
        m_InitialLoad = false;
    }

    static void fileAccessThread(void* pthis);
    static void audioThread(void *pthis);
    void Stop() { m_TestRunning = false; }
};

void MemoryPoolFsAccessTest::fileAccessThread(void* pthis)
{
    auto base = reinterpret_cast<MemoryPoolFsAccessTest*>(pthis);

    // Test 実行中、 WaveBuffer として利用し、かつ MemoryPool の attach / detach が繰り返される領域に対して、リソースの読み込みを繰り返す。
    while (base->m_TestRunning)
    {
        if (base->m_DataBuffer != nullptr &&
            base->m_DataSize > 0 &&
            base->m_InitialLoad)
        {
//             NN_LOG("F");
            std::lock_guard<nn::os::Mutex> lock(base->m_Mutex);
            nnt::audio::util::LoadSourceSamples(g_AudioTestSourceFileName,
                base->m_DataBuffer, base->m_DataSize, &base->m_SampleRate);
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
}

void MemoryPoolFsAccessTest::audioThread(void *pthis)
{
    auto base = reinterpret_cast<MemoryPoolFsAccessTest*>(pthis);
    const int sampleRate = 48000;

    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    std::string mountPath;
    nnt::audio::util::GetMountPath(mountPath);
    NNT_EXPECT_RESULT_SUCCESS(base->m_FsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()));

    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    parameter.sampleRate = 32000;
    parameter.sampleCount = parameter.sampleRate / 200;
    nn::os::SystemEvent systemEvent;
    nnt::audio::util::ScopedAudioRenderer sar(parameter, &systemEvent);

    const int channelCount = 2;
    const int8_t mainBus[channelCount] = { 0, 1 };

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 2);
    nn::audio::DeviceSinkType deviceSink;
    nn::audio::AddDeviceSink(&sar.GetConfig(), &deviceSink, &finalMix, mainBus, sizeof(mainBus), "MainAudioOut");

    sar.Update();
    sar.Start();

    base->m_DataSize = nnt::audio::util::GetWaveSampleSize(g_AudioTestSourceFileName);
    base->m_DataBuffer = allocator.Allocate(base->m_DataSize, nn::audio::MemoryPoolType::AddressAlignment);
    NN_ABORT_UNLESS_NOT_NULL(base->m_DataBuffer);

    size_t dataSize = nnt::audio::util::LoadSourceSamples(g_AudioTestSourceFileName, base->m_DataBuffer, base->m_DataSize, &base->m_SampleRate);
    size_t poolSize = nn::util::align_up(dataSize, nn::audio::MemoryPoolType::SizeGranularity);
    base->m_InitialLoad = true;

    nn::audio::VoiceType voice;
    voice._pVoiceInfo = nullptr;

    nn::audio::WaveBuffer waveBuffer;
    waveBuffer.buffer = base->m_DataBuffer;
    waveBuffer.size = dataSize;
    waveBuffer.startSampleOffset = 0;
    // 再生・停止の頻度を上げるために 1/100 してサンプル数を減らす
    waveBuffer.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t)) / 100;
    waveBuffer.loop = false;
    waveBuffer.pContext = nullptr;
    waveBuffer.contextSize = 0;

    nn::audio::MemoryPoolType memoryPool;

    TestState state(TestState::AttachAndStart);

    bool needLock = false;
    while (base->m_TestRunning)
    {
        // No wait
        switch(state)
        {
        case TestState::AttachAndStart:
//             NN_LOG("A");
            ASSERT_FALSE(nn::audio::IsVoiceValid(&voice));
            ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, base->m_DataBuffer, poolSize));
            ASSERT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, sampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
            ASSERT_FALSE(nn::audio::RequestDetachMemoryPool(&memoryPool));
            ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));
            nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);
            ASSERT_TRUE(nn::audio::AppendWaveBuffer(&voice, &waveBuffer));
            nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
            nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.f, 0, mainBus[0]);
            nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.f, 0, mainBus[1]);
            needLock = true; // Flush 操作との排他の為に、 AppendWaveBuffer 後、最初の update のみ fs との排他が必要。
            systemEvent.Wait();
            state = TestState::Playing;
            break;
        case TestState::Playing:
            systemEvent.Wait();
            if (nn::audio::GetReleasedWaveBuffer(&voice))
            {
                state = TestState::Stop;
            }
            break;
        case TestState::Stop:
//             NN_LOG("S");
            nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Stop);
            nn::audio::ReleaseVoiceSlot(&sar.GetConfig(), &voice);
            state = TestState::Detach;
            break;
        case TestState::Detach:
//             NN_LOG("D");
            ASSERT_FALSE(nn::audio::RequestAttachMemoryPool(&memoryPool));
            ASSERT_TRUE(nn::audio::RequestDetachMemoryPool(&memoryPool));
            state = TestState::Wait;
            break;
        case TestState::Wait:
//             NN_LOG("W");
            if (nn::audio::IsMemoryPoolAttached(&memoryPool) == false)
            {
                nn::audio::ReleaseMemoryPool(&sar.GetConfig(), &memoryPool);
                state = TestState::AttachAndStart;
            }
            else
            {
                // SIGLO-28216 にて RequestAttach/RequestDetach のキャンセルがサポートされたためコメントアウト
                // ASSERT_FALSE(nn::audio::RequestAttachMemoryPool(&memoryPool));
                ASSERT_FALSE(nn::audio::RequestDetachMemoryPool(&memoryPool));
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        if (needLock)
        {
            base->m_Mutex.Lock();
        }
        sar.Update();
        if (needLock)
        {
            base->m_Mutex.Unlock();
            needLock = false;
        }

    }
} // NOLINT(readability/fn_size)

TEST(MemoryPoolAttachAndDetach, Success)
{
    int loopCount = 300; // about 3 sec. duration.

    auto p1 = malloc(1024 * 1024 + nn::audio::MemoryPoolType::AddressAlignment - 1);
    auto p2 = malloc(1024 * 1024 + nn::audio::MemoryPoolType::AddressAlignment - 1);
    NN_ABORT_UNLESS_NOT_NULL(p1);
    NN_ABORT_UNLESS_NOT_NULL(p2);
    nn::os::ThreadType g_MemoryPoolTestThread;
    nn::os::ThreadType g_FileAccessThread;
    MemoryPoolFsAccessTest testBase;

    auto palign1 = nn::util::align_up(reinterpret_cast<uintptr_t>(p1), nn::os::MemoryPageSize);
    NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&g_MemoryPoolTestThread, (nn::os::ThreadFunction)MemoryPoolFsAccessTest::audioThread, &testBase, reinterpret_cast<void*>(palign1), 1024 * 1024, nn::os::DefaultThreadPriority));
    nn::os::SetThreadNamePointer(&g_MemoryPoolTestThread, "MemoryPoolTestThread");
    nn::os::StartThread(&g_MemoryPoolTestThread);

    auto palign2 = nn::util::align_up(reinterpret_cast<uintptr_t>(p2), nn::os::MemoryPageSize);
    NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&g_FileAccessThread, (nn::os::ThreadFunction)MemoryPoolFsAccessTest::fileAccessThread, &testBase, reinterpret_cast<void*>(palign2), 1024 * 1024, nn::os::DefaultThreadPriority));
    nn::os::SetThreadNamePointer(&g_FileAccessThread, "FileAccessThread");
    nn::os::StartThread(&g_FileAccessThread);

    int cnt = 0;
    for (;;)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));

        if (++cnt > loopCount)
        {
            testBase.Stop();
            break;
        }
    }

    nn::os::WaitThread(&g_MemoryPoolTestThread);
    nn::os::DestroyThread(&g_MemoryPoolTestThread);
    nn::os::WaitThread(&g_FileAccessThread);
    nn::os::DestroyThread(&g_FileAccessThread);
    free(p1);
    free(p2);
    SUCCEED();
}

TEST(MemoryPoolErrorCheck, Success)
{
    const std::size_t size = 128 * 1024 * 1024;
    void* buffer = std::malloc(size);
    ASSERT_TRUE(buffer != nullptr);

    {
        SimpleAudioRenderer renderer(buffer, size);
        nn::audio::SetMemoryPoolErrorCheckEnabled(&renderer.GetConfig(), false);
        EXPECT_FALSE(nn::audio::IsMemoryPoolErrorCheckEnabled(&renderer.GetConfig()));

        // preparation: prepare constant value short waveform without MemoryPool
        const auto InputValue = std::numeric_limits<int16_t>::max();
        NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t input[nn::audio::MemoryPoolType::SizeGranularity / sizeof(int16_t)];
        for (auto i = 0; i < static_cast<int>(sizeof(input) / sizeof(input[0])); ++i)
        {
            input[i] = InputValue;
        }

        {
            renderer.AppendPcm16Waveform(input, sizeof(input));
            renderer.SetVoiceVolume(0.0f);

            const auto LoopCount = 4;
            for (auto loop = 0; loop < LoopCount; ++loop)
            {
                renderer.Wait();
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(renderer.GetHandle(), &renderer.GetConfig()));  // renderer.Update();
            }
        }
    }

    std::free(buffer);
}

TEST(CopyMemoryPoolData, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    nnt::audio::util::ScopedAudioRenderer sar(parameter);
    sar.Start();

    auto buffer = reinterpret_cast<char*>(allocator.Allocate(nn::audio::MemoryPoolType::SizeGranularity * 2));
    memset(buffer, 0, nn::audio::MemoryPoolType::SizeGranularity * 2);
    auto middleOfBuffer = &buffer[nn::audio::MemoryPoolType::SizeGranularity];
    memset(middleOfBuffer, 0xF, nn::audio::MemoryPoolType::SizeGranularity);

    nn::audio::MemoryPoolType pool;
    nn::audio::AcquireMemoryPool(&sar.GetConfig(), &pool, buffer, nn::audio::MemoryPoolType::SizeGranularity);
    nn::audio::CopyMemoryPoolData(&pool, buffer, middleOfBuffer, nn::audio::MemoryPoolType::SizeGranularity);

    EXPECT_TRUE(std::strncmp(buffer, middleOfBuffer, nn::audio::MemoryPoolType::SizeGranularity) == 0);

    memset(buffer, 0, nn::audio::MemoryPoolType::SizeGranularity * 2);
    nn::audio::CopyMemoryPoolData(&pool, middleOfBuffer, buffer, nn::audio::MemoryPoolType::SizeGranularity);

    EXPECT_TRUE(std::strncmp(buffer, middleOfBuffer, nn::audio::MemoryPoolType::SizeGranularity) == 0);
}

