﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <cstdlib>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/audio/audio_Applet.h>
#include <nn/audio/audio_Debugger.h>
#include <nn/applet/applet_Apis.h>
#include "../../../../../Programs/Eris/Sources/Libraries/audio/common/audio_NodeIdManager.h"

#include <nn/nn_Log.h>
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include "testAudio_getWorkBufferSize.nx.h"
#endif
#if defined(NN_BUILD_CONFIG_OS_WIN)
#if defined(NN_BUILD_CONFIG_ADDRESS_64)
#include "testAudio_getWorkBufferSize.win64.h"
#else
#include "testAudio_getWorkBufferSize.win32.h"
#endif
#endif
namespace {
void SetDefaultParameter(nn::audio::AudioRendererParameter* pOutParameter)
{
    nn::audio::InitializeAudioRendererParameter(pOutParameter);
    pOutParameter->mixBufferCount = 16;
    pOutParameter->voiceCount = 24;
}

NN_ALIGNAS(4096) char g_WorkBuffer[1024 * 1024];  // 4096 == nn::os::MemoryPageSize

class ScopedAudioRenderer
{
private:
    void* m_WorkBuffer;
    void* m_ConfigBuffer;
    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;

    nn::mem::StandardAllocator m_Allocator;

public:
    NN_IMPLICIT ScopedAudioRenderer(const nn::audio::AudioRendererParameter& parameter, nn::os::SystemEvent* pSystemEvent = nullptr)
    {
        m_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        m_WorkBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

        if (pSystemEvent)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, pSystemEvent, parameter, m_WorkBuffer, workBufferSize));
        }
        else
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, parameter, m_WorkBuffer, workBufferSize));
        }

        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
        m_ConfigBuffer = m_Allocator.Allocate(configBufferSize);
        nn::audio::InitializeAudioRendererConfig(&m_Config, parameter, m_ConfigBuffer, configBufferSize);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(m_Handle));
    }
    ~ScopedAudioRenderer()
    {
        nn::audio::StopAudioRenderer(m_Handle);
        nn::audio::CloseAudioRenderer(m_Handle);

        m_Allocator.Free(m_ConfigBuffer);
        m_Allocator.Free(m_WorkBuffer);
        m_Allocator.Finalize();
    }
    nn::audio::AudioRendererHandle GetHandle() const
    {
        return m_Handle;
    }
    nn::audio::AudioRendererConfig& GetConfig()
    {
        return m_Config;
    }
};

}

/**
 * @brief       IsValidAudioRendererParameter() の異常系テストです。
 */
TEST(IsValidAudioRendererParameterPrivate, Failure)
{
    nn::audio::AudioRendererParameter parameter;

    // SampleCount
    SetDefaultParameter(&parameter);
    parameter.sampleRate = 32000 - 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.sampleRate = 32000 + 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.sampleRate = 48000 - 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.sampleRate = 48000 + 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.sampleRate = 32000;
    parameter.sampleCount = 160 + 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.sampleRate = 48000;
    parameter.sampleCount = 320 + 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.sampleRate = 32000;
    parameter.sampleCount = 160 - 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.sampleRate = 48000;
    parameter.sampleCount = 320 - 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    // MixBufferCount
    SetDefaultParameter(&parameter);
    parameter.mixBufferCount = 0;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.mixBufferCount = nn::audio::AudioRendererParameter::MixBufferCountMax + 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    // VoiceCount
    SetDefaultParameter(&parameter);
    parameter.voiceCount = 0;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.voiceCount = nn::audio::AudioRendererParameter::VoiceCountMax + 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.voiceCount = (1 << nn::audio::common::NodeIdManager::BaseWidth) + 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    // SinkCount
    SetDefaultParameter(&parameter);
    parameter.sinkCount = 0;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.sinkCount = nn::audio::AudioRendererParameter::SinkCountMax + 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    // EffectCount
    SetDefaultParameter(&parameter);
    parameter.effectCount = -1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.effectCount = nn::audio::AudioRendererParameter::EffectCountMax + 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    // SubMix
    SetDefaultParameter(&parameter);
    parameter.subMixCount = -1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));

    parameter.subMixCount = nn::audio::AudioRendererParameter::SubMixCountMax + 1;
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(parameter));
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
/**
 * @brief       Debugger 向け API の正常系テストです
 */
TEST(SuspendAndResumeForDebug, Success)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioRenderersForDebug(appletResourceUserId));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioRenderersForDebug(appletResourceUserId));
}

/**
 * @brief       API call test in suspended status
 */
TEST(SuspendForDebug, Success)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();

    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    {
        ScopedAudioRenderer sar(parameter);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioRenderersForDebug(appletResourceUserId));

        for (auto i = 0; i < 16; ++i)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        }

        // AudioRenderer is stopped and closed without resuming
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioRenderersForDebug(appletResourceUserId));
}

/**
 * @brief       AppletManager 向け API の正常系テストです
 */
TEST(SuspendAndResume, Success)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();

    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    nn::os::SystemEvent systemEvent;

    ScopedAudioRenderer sar(parameter, &systemEvent);

    {
        nn::audio::VoiceType voice;
        nn::audio::FinalMixType finalMix;
        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

        nn::audio::MemoryPoolType memoryPool;
        NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];

        ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
        ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

        nn::audio::WaveBuffer waveBuffer;
        waveBuffer.buffer = pcmBuffer;
        waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]));
        waveBuffer.loop = 0;
        waveBuffer.isEndOfStream = false;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(appletResourceUserId));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioRenderers(appletResourceUserId, nn::TimeSpan::FromMilliSeconds(1)));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioRenderers(appletResourceUserId, nn::TimeSpan::FromMilliSeconds(1)));

    {
        float volume;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::GetAudioRenderersProcessMasterVolume(&volume, appletResourceUserId));
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::SetAudioRenderersProcessMasterVolume(appletResourceUserId, 1.0f, nn::TimeSpan::FromMilliSeconds(1)));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(appletResourceUserId));
}

/**
 * @brief       API call test in suspended status
 */
TEST(Suspend, Success)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();

    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    {
        ScopedAudioRenderer sar(parameter);

        nn::os::SystemEvent systemEvent;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioRenderers(appletResourceUserId, nn::TimeSpan::FromMilliSeconds(10)));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        for (auto i = 0; i < 16; ++i)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        }

        // AudioRenderer is stopped and closed without resuming
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioRenderers(appletResourceUserId, nn::TimeSpan::FromMilliSeconds(0)));
}

/**
 * @brief       AppletManager 向け API の事前条件テストです。
 */
TEST(SuspendAndResume, Precondition)
{
    auto id = nn::applet::GetAppletResourceUserId();

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestSuspendAudioRenderers(id, nn::TimeSpan::FromMilliSeconds(0 - 1)) , "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestSuspendAudioRenderers(id, nn::TimeSpan::FromMilliSeconds(10 + 1)) , "");
}

TEST(RegisterApplet, Success)
{
    auto id = nn::applet::GetAppletResourceUserId();
    float volume = 1.0f;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(id)); // Actually register. (Lookup Count = 1)
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(id)); // Only the lookup count goes up. (Lookup Count = 2)
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(id)); // Only the lookup count goes down. (Lookup Count = 1)

    nn::audio::SetAudioRenderersProcessMasterVolume(id, 0.5f, 0);
    nn::audio::GetAudioRenderersProcessMasterVolume(&volume, id);

    EXPECT_TRUE(volume == 0.5f);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(id)); // Actually cancel registration. (Lookup Count = 0)
}

TEST(RegisterApplet, Failure)
{
    auto id = nn::applet::AppletResourceUserId::GetInvalidId();
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::UnregisterAppletResourceUserId(id));

    float volume = 0.0f;
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::GetAudioRenderersProcessMasterVolume(&volume, id));
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::SetAudioRenderersProcessMasterVolume(id, 0.5f, 0));
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::RequestSuspendAudioRenderers(id, nn::TimeSpan::FromMilliSeconds(10)));
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::RequestResumeAudioRenderers(id, nn::TimeSpan::FromMilliSeconds(10)));
}
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)

// A test process aborts if this test case is success.
#if 0
TEST(OpenAudioRenderer, SystemEventLeak)
{
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioRendererParameter parameter;
    nn::audio::RendereritializeAudioRendererParameter(&parameter);

    for(int i = 0; i < 10000; ++i)
    {
        ScopedAudioRenderer aren(parameter, &systemEvent);
    }
}
#endif

TEST(AudioRendererGetWorkBufferSize, Success)
{
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);
    for( auto testParam : testBufferParams)
    {
        nn::audio::InitializeAudioRendererParameter(&param);
        param.sinkCount = testParam.param.sinkCount;
        param.effectCount = testParam.param.effectCount;
        param.voiceCount = testParam.param.voiceCount;
        param.mixBufferCount = testParam.param.mixBufferCount;
        param.subMixCount = testParam.param.subMixCount;
        param.splitterCount = testParam.param.splitterCount;
        param.splitterSendChannelCount = testParam.param.splitterSendChannelCount;
        param.performanceFrameCount = testParam.param.performanceFrameCount;
        EXPECT_EQ(testParam.expectedSize, nn::audio::GetAudioRendererWorkBufferSize(param));
    }
}

#if 0
TEST(AudioRendererGetWorkBufferSizeRecord, Success)
{
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);

    size_t testStep = 4;
    NN_LOG("WorkBufferSizeParam testBufferParams[] {\n");
    for(int mixBufferIteration = 1; mixBufferIteration < nn::audio::AudioRendererParameter::MixBufferCountMax; mixBufferIteration += nn::audio::AudioRendererParameter::MixBufferCountMax / testStep)
    {
        for(int subMixIteration = 0; subMixIteration < nn::audio::AudioRendererParameter::SubMixCountMax; subMixIteration += nn::audio::AudioRendererParameter::SubMixCountMax / testStep)
        {
            for(int voiceCountIteration = 1; voiceCountIteration < nn::audio::AudioRendererParameter::VoiceCountMax; voiceCountIteration += nn::audio::AudioRendererParameter::VoiceCountMax / testStep)
            {
                for(int sinkCountIteration = 1; sinkCountIteration < nn::audio::AudioRendererParameter::SinkCountMax; sinkCountIteration += nn::audio::AudioRendererParameter::SinkCountMax / testStep)
                {
                    for(int effectCountIteration = 0; effectCountIteration < nn::audio::AudioRendererParameter::EffectCountMax; effectCountIteration += nn::audio::AudioRendererParameter::EffectCountMax / testStep)
                    {
                        for(int perfCountIteration = 0; perfCountIteration < nn::audio::AudioRendererParameter::PerformanceFrameCountMax; perfCountIteration += nn::audio::AudioRendererParameter::PerformanceFrameCountMax / testStep)
                        {
                            for(int splitterCountIteration = 1; splitterCountIteration < nn::audio::AudioRendererParameter::SplitterCountMax; splitterCountIteration += nn::audio::AudioRendererParameter::SplitterCountMax / testStep)
                            {
                                nn::audio::InitializeAudioRendererParameter(&param);
                                param.sinkCount = sinkCountIteration;
                                param.effectCount = effectCountIteration;
                                param.voiceCount = voiceCountIteration;
                                param.mixBufferCount = mixBufferIteration;
                                param.subMixCount = subMixIteration;
                                param.splitterCount = splitterCountIteration;
                                param.performanceFrameCount = perfCountIteration;
                                NN_LOG("{ { %d, %d, %d, %d, %d, %d, %d, %d }, %d, }, \n",
                                param.mixBufferCount, param.subMixCount, param.voiceCount,
                                param.sinkCount, param.effectCount, param.performanceFrameCount, param.splitterCount,
                                param.splitterSendChannelCount, nn::audio::GetAudioRendererWorkBufferSize(param));
                            }
                        }
                    }
                }
            }
        }
    }
    NN_LOG("\n};\n");
}
#endif
