﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <cstdlib>
#include <cstring>

#include <nn/os.h>
#include <nn/audio.h>
#include <nn/audio/audio_Applet.h>
#include <nn/audio/audio_Debugger.h>
#include <nn/applet/applet_Apis.h>
#include <nn/util/util_BitUtil.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nn/nn_SdkLog.h>
#include "../../../Programs/Eris/Sources/Libraries/audio/common/audio_Util.h" // common::AudioOutInitializedMagic

#define NN_TEST_AUDIO_EXPECT_RANGE(value, begin, end)       \
    {                                                       \
        auto nn_test_audio_expect_range_value = (value);    \
        EXPECT_GE(nn_test_audio_expect_range_value, begin); \
        EXPECT_LT(nn_test_audio_expect_range_value, end);   \
    }

#if !defined(NN_SDK_BUILD_RELEASE)
#define NN_TEST_EXPECT_DEATH_OR_RESULT(func, expectedResult) \
    EXPECT_DEATH_IF_SUPPORTED(func, "")
#else
#define NN_TEST_EXPECT_DEATH_OR_RESULT(func, expectedResult) \
    NNT_EXPECT_RESULT_FAILURE(expectedResult, func)
#endif


namespace {

NN_AUDIO_ALIGNAS_AUDIO_OUT_BUFFER_ALIGN char g_Buffer[nn::audio::AudioOutBuffer::SizeGranularity];

}

class ScopedAudioOut
{
public:
    ScopedAudioOut() : m_AudioOut()
    {
    }
    ~ScopedAudioOut()
    {
        if (nn::audio::GetAudioOutState(&m_AudioOut) == nn::audio::AudioOutState_Started)
        {
            nn::audio::StopAudioOut(&m_AudioOut);
        }
        nn::audio::CloseAudioOut(&m_AudioOut);
    }
    nn::audio::AudioOut* operator->()
    {
        return &m_AudioOut;
    }
    nn::audio::AudioOut* Get()
    {
        return &m_AudioOut;
    }
private:
    nn::audio::AudioOut m_AudioOut;
};

/**
 * @brief       InitializeAudioOutParameter() の正常系テストです。
 */
TEST(InitializeAudioOutParameter, Success)
{
    nn::audio::AudioOutParameter parameter;

    nn::audio::InitializeAudioOutParameter(&parameter);
    EXPECT_EQ(parameter.sampleRate, 0);
    EXPECT_EQ(parameter.channelCount, 0);
}

/**
 * @brief       ListAudioOuts() の正常系テストです。
 */
TEST(ListAudioOuts, Success)
{
    int counts[] = { 0, 1, nn::audio::AudioOutCountMax };
    for (auto count : counts)
    {
        nn::audio::AudioOutInfo audioOutInfos[nn::audio::AudioOutCountMax];
        std::memset(audioOutInfos, 0xFF, sizeof(audioOutInfos));

        int listedCount = nn::audio::ListAudioOuts(audioOutInfos, count);
        NN_TEST_AUDIO_EXPECT_RANGE(listedCount, 0, count + 1);

        for (int i = 0; i < listedCount; ++i)
        {
            EXPECT_GT(strnlen(audioOutInfos[i].name, nn::audio::AudioOut::NameLength), 0u);
        }

        // 外部仕様では特に規定していないがサーバ側で 0 クリアするのでその確認
        for (int i = listedCount; i < count; ++i)
        {
            EXPECT_EQ(strnlen(audioOutInfos[i].name, nn::audio::AudioOut::NameLength), 0u);
        }
    }
}

/**
 * @brief       ListAudioOuts() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(ListAudioOuts, PreCondition)
{
    nn::audio::AudioOutInfo audioOutInfos[nn::audio::AudioOutCountMax];

    // outAudioOuts == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::ListAudioOuts(nullptr, sizeof(audioOutInfos) / sizeof(*audioOutInfos)), "");

    // count < 0
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::ListAudioOuts(audioOutInfos, -1), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       OpenAudioOut() の正常系テストです。
 */
TEST(OpenAudioOutTest, Success)
{
    nn::audio::AudioOutInfo audioOutInfo;
    int count = nn::audio::ListAudioOuts(&audioOutInfo, 1);
    EXPECT_EQ(count, 1);

    int sampleRate;

    // With default
    {
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioOut(aout.Get(), audioOutInfo.name, parameter));
        sampleRate = GetAudioOutSampleRate(aout.Get());
        EXPECT_EQ(
            nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Stopped);
    }

    // With systemEvent default
    {
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        nn::os::SystemEvent systemEvent;
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioOut(aout.Get(), &systemEvent, audioOutInfo.name, parameter));
        sampleRate = GetAudioOutSampleRate(aout.Get());
        EXPECT_EQ(
            nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Stopped);
    }

    // With non-default sample rate
    {
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        parameter.sampleRate = sampleRate;
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioOut(aout.Get(), audioOutInfo.name, parameter));
        EXPECT_EQ(
            nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Stopped);
    }

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // With non-default channel count on NX
    {
        int channelCounts[] = { 2, 6 };
        for(auto channelCount : channelCounts)
        {
            nn::audio::AudioOutParameter parameter;
            nn::audio::InitializeAudioOutParameter(&parameter);
            parameter.channelCount = channelCount;
            ScopedAudioOut aout;
            NNT_EXPECT_RESULT_SUCCESS(
                nn::audio::OpenAudioOut(aout.Get(), audioOutInfo.name, parameter));
            EXPECT_EQ(
                nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Stopped);
        }
    }
#endif

    // 同時に 2 つオープンする
    {
        nn::audio::AudioOutParameter parameter0;
        nn::audio::InitializeAudioOutParameter(&parameter0);
        nn::audio::AudioOutParameter parameter1;
        nn::audio::InitializeAudioOutParameter(&parameter1);
        parameter1.sampleRate = sampleRate;
        ScopedAudioOut audioOut0, audioOut1;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioOut(audioOut0.Get(), audioOutInfo.name, parameter0));
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioOut(audioOut1.Get(), audioOutInfo.name, parameter1));
    }
}

/**
 * @brief       OpenAudioOut() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(OpenAudioOutTest, PreCondition)
{
    nn::audio::AudioOutInfo audioOutInfo;
    int count = nn::audio::ListAudioOuts(&audioOutInfo, 1);
    EXPECT_EQ(count, 1);

    nn::audio::AudioOut audioOut;

    // With default
    {
        nn::audio::AudioOutParameter parameter;
        // Uninitialized parameter
        memset(&parameter, 0, sizeof(nn::audio::AudioOutParameter));
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioOut(&audioOut, audioOutInfo.name, parameter), "");
        nn::audio::InitializeAudioOutParameter(&parameter);
        // pAudioOut == nullptr
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioOut(nullptr, audioOutInfo.name, parameter), "");
        // name == nullptr
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioOut(&audioOut, nullptr, parameter), "");
        // pSystemEvent == nullptr
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioOut(&audioOut, nullptr, audioOutInfo.name, parameter), "");
    }

    // With non-default sample rate
    {
        // pAudioOut == nullptr
        nn::audio::AudioOutParameter parameter0;
        nn::audio::InitializeAudioOutParameter(&parameter0);

        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioOut(nullptr, audioOutInfo.name, parameter0), "");

        // name == nullptr
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioOut(&audioOut, nullptr, parameter0), "");

        // sampleRate < 0
        nn::audio::AudioOutParameter parameter1;
        nn::audio::InitializeAudioOutParameter(&parameter1);
        parameter1.sampleRate = -1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioOut(&audioOut, audioOutInfo.name, parameter1), "");
    }
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       OpenAudioOut() のエラー処理テストです。
 */
TEST(OpenAudioOutTest, Error)
{
    // おおよそ ありえない であろう名前
    const char InvalidAudioOutName[] = "____InvalidAudioOutName0123456789";
    const int InvalidSampleRate = 1;  // おおよそ ありえない であろうサンプルレート

    nn::audio::AudioOut audioOut;

    // With default
    {
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        NNT_EXPECT_RESULT_FAILURE(
            nn::audio::ResultNotFound, nn::audio::OpenAudioOut(&audioOut, InvalidAudioOutName, parameter));
    }

    // With non-default sample rate
    {
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        parameter.sampleRate = 48000;
        NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultNotFound,
            nn::audio::OpenAudioOut(&audioOut, InvalidAudioOutName, parameter));
    }

    // with invalid sample rate
    {
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        parameter.sampleRate = InvalidSampleRate;
        nn::audio::AudioOutInfo audioOutInfo;
        int count = nn::audio::ListAudioOuts(&audioOutInfo, 1);
        EXPECT_EQ(count, 1);
        auto result = nn::audio::OpenAudioOut(&audioOut, audioOutInfo.name, parameter);
        NNT_EXPECT_RESULT_FAILURE( nn::audio::ResultInvalidSampleRate, result);
    }
}

TEST(OpenDefaultAudioOutTest, Success)
{
    int sampleRate;

    // With default
    {
        ScopedAudioOut aout;
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
        EXPECT_EQ(
            nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Stopped);
        sampleRate = GetAudioOutSampleRate(aout.Get());
    }

    // With systemEvent default
    {
        ScopedAudioOut aout;
        nn::audio::AudioOutParameter parameter;
        nn::os::SystemEvent systemEvent;
        nn::audio::InitializeAudioOutParameter(&parameter);
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioOut(aout.Get(), &systemEvent, parameter));
        EXPECT_EQ(
            nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Stopped);
        sampleRate = GetAudioOutSampleRate(aout.Get());
    }

    // With non-default sample rate
    {
        ScopedAudioOut aout;
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        parameter.sampleRate = sampleRate;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
        EXPECT_EQ(
            nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Stopped);
    }

    // 同時に 2 つオープンする
    {
        nn::audio::AudioOutParameter parameter0;
        nn::audio::InitializeAudioOutParameter(&parameter0);
        nn::audio::AudioOutParameter parameter1;
        nn::audio::InitializeAudioOutParameter(&parameter1);
        parameter1.sampleRate = sampleRate;

        ScopedAudioOut audioOut0, audioOut1;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioOut(audioOut0.Get(), parameter0));
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioOut(audioOut1.Get(), parameter1));
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(OpenDefaultAudioOutDeathTest, NullAudioOut)
{
    // With default
    {
        // pAudioOut == nullptr
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenDefaultAudioOut(nullptr, parameter), "");
    }

    // initialized parameter.
    {
        nn::audio::AudioOut audioOut;
        nn::audio::AudioOutParameter parameter;
        memset(&parameter, 0, sizeof(nn::audio::AudioOutParameter));
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenDefaultAudioOut(&audioOut, parameter), "");
    }

    // initialized parameter w/o systemEvent
    {
        nn::audio::AudioOut audioOut;
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenDefaultAudioOut(&audioOut, nullptr, parameter), "");
    }

    // With non-default sample rate
    {
        // pAudioOut == nullptr
        nn::audio::AudioOutParameter parameter0;
        nn::audio::InitializeAudioOutParameter(&parameter0);
        parameter0.sampleRate = 48000;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenDefaultAudioOut(nullptr, parameter0), "");

        // sampleRate < 0
        nn::audio::AudioOutParameter parameter1;
        nn::audio::InitializeAudioOutParameter(&parameter1);
        parameter1.sampleRate = -1;
        nn::audio::AudioOut audioOut;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenDefaultAudioOut(&audioOut, parameter1), "");
    }
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(OpenDuplicateAudioOutTest, Error)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    nn::audio::AudioOutInfo audioOutInfos[nn::audio::AudioOutCountMax];
    std::memset(audioOutInfos, 0xFF, sizeof(audioOutInfos));

    int listedCount = nn::audio::ListAudioOuts(audioOutInfos, nn::audio::AudioOutCountMax);
    NN_TEST_AUDIO_EXPECT_RANGE(listedCount, 0, nn::audio::AudioOutCountMax + 1);
    {
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenDefaultAudioOut(aout.Get(), parameter), nn::audio::ResultAlreadyOpen);
    }
    {
        ScopedAudioOut aout;
        nn::os::SystemEvent event;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioOut(aout.Get(), &event, parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenDefaultAudioOut(aout.Get(), &event, parameter), nn::audio::ResultAlreadyOpen);
        nn::os::DestroySystemEvent(event.GetBase());
    }
    {
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioOut(aout.Get(), audioOutInfos[0].name, parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenAudioOut(aout.Get(), audioOutInfos[0].name, parameter), nn::audio::ResultAlreadyOpen);
    }
    {
        ScopedAudioOut aout;
        nn::os::SystemEvent event;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioOut(aout.Get(), &event, audioOutInfos[0].name, parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenAudioOut(aout.Get(), &event, audioOutInfos[0].name, parameter), nn::audio::ResultAlreadyOpen);
        nn::os::DestroySystemEvent(event.GetBase());
    }
}

/**
 * @brief       OpenDefaultAudioOut() のエラー処理テストです。
 */
TEST(OpenDefaultAudioOutTest, Error)
{
    const int InvalidSampleRate = 1;  // おおよそ ありえない であろうサンプルレート

    nn::audio::AudioOut audioOut;

    // With invalid sample rate
    {
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        parameter.sampleRate = InvalidSampleRate;
        NNT_EXPECT_RESULT_FAILURE(
            nn::audio::ResultInvalidSampleRate, nn::audio::OpenDefaultAudioOut(&audioOut, parameter));
    }
}

TEST(CloseAudioOutTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(CloseAudioOutDeathTest, NullAudioOut)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::CloseAudioOut(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

void CloseAudioOutDeathTest_InvalidAudioOutState()
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    nn::audio::AudioOut audioOut;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(&audioOut, parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioOut(&audioOut));
    EXPECT_EQ(
        nn::audio::GetAudioOutState(&audioOut), nn::audio::AudioOutState_Started);
    nn::audio::CloseAudioOut(&audioOut);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(CloseAudioOutDeathTest, InvalidAudioOutState)
{
    EXPECT_DEATH_IF_SUPPORTED(
        CloseAudioOutDeathTest_InvalidAudioOutState(), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

void CloseAudioOutDeathTest_OperationAfterClose()
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    nn::audio::AudioOut audioOut;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(&audioOut, parameter));
    nn::audio::CloseAudioOut(&audioOut);
    nn::audio::StartAudioOut(&audioOut);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(CloseAudioOutDeathTest, OperationAfterClose)
{
    EXPECT_DEATH_IF_SUPPORTED(
        CloseAudioOutDeathTest_OperationAfterClose(), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(StartAudioOutTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioOut(aout.Get()));
    EXPECT_EQ(
        nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Started);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(StartAudioOutDeathTest, NullAudioOut)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::StartAudioOut(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(StartAudioOutTest, InvalidAudioOutState)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioOut(aout.Get()));
    EXPECT_EQ(
        nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Started);

    //  Precondition violation: Start AudioOut when it is AudioOutState_Started.
#if defined(NN_SDK_BUILD_RELEASE)
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultOperationFailed, nn::audio::StartAudioOut(aout.Get()));
#else
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::StartAudioOut(aout.Get()), "");
#endif // defined(NN_SDK_BUILD_RELEASE)
}

TEST(StopAudioOutTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioOut(aout.Get()));
    nn::audio::StopAudioOut(aout.Get());
    EXPECT_EQ(
        nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Stopped);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(StopAudioOutDeathTest, NullAudioOut)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::StopAudioOut(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(StopAudioOutDeathTest, InvalidAudioOutState)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
    EXPECT_EQ(
        nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Stopped);

    //  Precondition violation: Stop AudioOut when it is AudioOutState_Stopped.
#if defined(NN_SDK_BUILD_RELEASE)
    nn::audio::StopAudioOut(aout.Get());
#else
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::StopAudioOut(aout.Get()), "");
#endif

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(aout.Get()));
    nn::audio::StopAudioOut(aout.Get());

    //  Precondition violation: Stop AudioOut when it is AudioOutState_Stopped.
#if defined(NN_SDK_BUILD_RELEASE)
    nn::audio::StopAudioOut(aout.Get());
#else
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::StopAudioOut(aout.Get()), "");
#endif
}

TEST(GetAudioOutState, NullAudioOut)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAudioOutState(nullptr), "");
}

TEST(GetAudioOutNameTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
    EXPECT_TRUE(nn::audio::GetAudioOutName(aout.Get()) != NULL);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioOutNameDeathTest, NullAudioOut)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAudioOutName(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetAudioOutSampleRateTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
    EXPECT_GT(
        nn::audio::GetAudioOutSampleRate(aout.Get()), 0);
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    EXPECT_EQ(
        nn::audio::GetAudioOutSampleRate(aout.Get()), 48000);
#endif
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioOutSampleRateDeathTest, NullAudioOut)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAudioOutSampleRate(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetAudioOutChannelCountTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
    EXPECT_GT(
        nn::audio::GetAudioOutChannelCount(aout.Get()), 0);
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    EXPECT_EQ(
        nn::audio::GetAudioOutChannelCount(aout.Get()), 2);
#endif
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioOutChannelCountDeathTest, NullAudioOut)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAudioOutChannelCount(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetAudioOutSampleFormatTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
    EXPECT_NE(
        nn::audio::GetAudioOutSampleFormat(aout.Get()), nn::audio::SampleFormat_Invalid);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioOutSampleFormatDeathTest, NullAudioOut)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAudioOutSampleFormat(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       AppendAudioOutBuffer() の正常系テストです。
 */
TEST(AppendAudioOutBufferTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));

    nn::audio::AudioOutBuffer audioOutBuffer;
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));

    for(int i = 0; i < nn::audio::AudioOutBufferCountMax; ++i)
    {
        EXPECT_TRUE(nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer));
        EXPECT_EQ(nn::audio::GetAudioOutBufferCount(aout.Get()), i + 1);
    }

    EXPECT_FALSE(nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer));
}

/**
 * @brief       AppendAudioOutBuffer() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(AppendAudioOutBufferTest, PreCondition)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));

    nn::audio::AudioOutBuffer audioOutBuffer;

    // pAudioOut == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::AppendAudioOutBuffer(nullptr, &audioOutBuffer), "");

    // pAudioOutBuffer == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::AppendAudioOutBuffer(aout.Get(), nullptr), "");

    // pAudioOutBuffer.buffer == nullptr
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
    audioOutBuffer.buffer = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer), "");

    // GetAudioOutBufferDataPointer(pAudioOutBuffer) is multiple of nn::audio::BufferAlignSize
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer) - 1);
    audioOutBuffer.buffer = g_Buffer + 1;
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer), "");

    // pAudioOutBuffer->size % (nn::audio::GetAudioOutChannelCount(pAudioOut) * nn::audio::GetSampleByteSize(nn::audio::GetAudioOutSampleFormat(pAudioOut))) == 0
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), 1);
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer), "");
}

TEST(SetAudioOutBufferInfo, PreCondition)
{
    nn::audio::AudioOutBuffer audioOutBuffer;

    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::SetAudioOutBufferInfo(nullptr, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer)), "");
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, nullptr, sizeof(g_Buffer), sizeof(g_Buffer)), "");
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer + 1, sizeof(g_Buffer), sizeof(g_Buffer)), "");
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer) - 1, sizeof(g_Buffer)), "");
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer) + 1), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetReleasedAudioOutBufferTest() の正常系テストです。
 */
TEST(GetReleasedAudioOutBufferTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));

    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);

    nn::audio::AudioOutBuffer audioOutBuffer;
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
    nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer);

    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioOut(aout.Get()));

    nn::audio::AudioOutBuffer* pReleased = nullptr;
    while (pReleased == nullptr)
    {
        pReleased = nn::audio::GetReleasedAudioOutBuffer(aout.Get());
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }

    EXPECT_EQ(pReleased, &audioOutBuffer);
    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);
}

TEST(AppendManyAudioOutBufferTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), &systemEvent, parameter));
    nn::audio::AudioOutBuffer audioOutBuffer;
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioOut(aout.Get()));

    // 1. Append nn::audio::AudioOutBufferCountMax -> Release nn::audio::AudioOutBufferCountMax
    for(int i = 0; i < nn::audio::AudioOutBufferCountMax; ++i)
    {
        EXPECT_TRUE(nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer));
    }

    for(int releasedBufferCount = 0; releasedBufferCount < nn::audio::AudioOutBufferCountMax;)
    {
        systemEvent.Wait();

        while(nn::audio::GetReleasedAudioOutBuffer(aout.Get()) != nullptr)
        {
            ++releasedBufferCount;
        }
    }

    // 2. Append nn::audio::AudioOutBufferCountMax -> Release nn::audio::AudioOutBufferCountMax / 2
    //    -> Append nn::audio::AudioOutBufferCountMax / 2 -> Release nn::audio::AudioOutBufferCountMax
    for(int i = 0; i < nn::audio::AudioOutBufferCountMax; ++i)
    {
        EXPECT_TRUE(nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer));
    }

    for(int releasedBufferCount = 0, releaseBufferCountMax = nn::audio::AudioOutBufferCountMax / 2 ; releasedBufferCount < releaseBufferCountMax;)
    {
        systemEvent.Wait();

        while(releasedBufferCount < releaseBufferCountMax && nn::audio::GetReleasedAudioOutBuffer(aout.Get()) != nullptr)
        {
            ++releasedBufferCount;
        }
    }

    for(int i = 0; i < nn::audio::AudioOutBufferCountMax / 2; ++i)
    {
        EXPECT_TRUE(nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer));
    }

    for(int releasedBufferCount = 0; releasedBufferCount < nn::audio::AudioOutBufferCountMax;)
    {
        systemEvent.Wait();

        while(nn::audio::GetReleasedAudioOutBuffer(aout.Get()) != nullptr)
        {
            ++releasedBufferCount;
        }
    }
}

void ThreadFunc(void* arg)
{
    NN_UNUSED(arg);
}

TEST(AppendAfterStartStopTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    nn::audio::AudioOut audioOut;
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(&audioOut, &systemEvent, parameter));
    nn::audio::AudioOutBuffer audioOutBuffer;
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioOut(&audioOut));

    nn::audio::StopAudioOut(&audioOut);

    // 1. Append nn::audio::AudioOutBufferCountMax -> Release nn::audio::AudioOutBufferCountMax
    for(int i = 0; i < nn::audio::AudioOutBufferCountMax; ++i)
    {
        EXPECT_TRUE(nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer));
    }

    nn::audio::CloseAudioOut(&audioOut);

    //Replaced thread creation with os call to make sure g_Buffer is not mapped
    #if defined(NN_BUILD_CONFIG_OS_HORIZON)
    EXPECT_FALSE(nn::os::IsMemoryLocked(g_Buffer, sizeof(g_Buffer)));
    #endif
}

/**
 * @brief       GetReleasedAudioOutBuffer() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetReleasedAudioOutBufferTest, PreCondition)
{
    // pAudioOut == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetReleasedAudioOutBuffer(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(ReleasedAudioOutBufferSystemEventTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), &systemEvent, parameter));
    EXPECT_NE(systemEvent.GetReadableHandle(), nn::os::InvalidNativeHandle);
    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);

    nn::audio::AudioOutBuffer audioOutBuffer;
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
    nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer);
    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioOut(aout.Get()));
    nn::audio::AudioOutBuffer* pReleased = nullptr;

    systemEvent.Wait();

    pReleased = nn::audio::GetReleasedAudioOutBuffer(aout.Get());
    EXPECT_EQ(pReleased, &audioOutBuffer);
    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);
}

/**
 * @brief       ContainsAudioOutBuffer() の正常系テストです。
 */
TEST(ContainsAudioOutBufferTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));

    nn::audio::AudioOutBuffer audioOutBuffer;
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));

    EXPECT_FALSE(nn::audio::ContainsAudioOutBuffer(aout.Get(), &audioOutBuffer));

    nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer);

    EXPECT_TRUE(nn::audio::ContainsAudioOutBuffer(aout.Get(), &audioOutBuffer));

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioOut(aout.Get()));

    nn::audio::AudioOutBuffer* pReleased = nullptr;
    while (pReleased == nullptr)
    {
        pReleased = nn::audio::GetReleasedAudioOutBuffer(aout.Get());
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }

    EXPECT_FALSE(nn::audio::ContainsAudioOutBuffer(aout.Get(), &audioOutBuffer));
}

/**
 * @brief       ContainsAudioOutBuffer() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(ContainsAudioOutBufferTest, PreCondition)
{
    nn::audio::AudioOut audioOut;
    nn::audio::AudioOutBuffer audioOutBuffer;

    // pAudioOut == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::ContainsAudioOutBuffer(nullptr, &audioOutBuffer), "");
    // pAudioOutBuffer == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::ContainsAudioOutBuffer(&audioOut, nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetSampleFormatTest, ValidValueTest)
{
    EXPECT_EQ(GetSampleByteSize(nn::audio::SampleFormat_PcmInt8), 1u);
    EXPECT_EQ(GetSampleByteSize(nn::audio::SampleFormat_PcmInt16), 2u);
    EXPECT_EQ(GetSampleByteSize(nn::audio::SampleFormat_PcmInt24), 3u);
    EXPECT_EQ(GetSampleByteSize(nn::audio::SampleFormat_PcmInt32), 4u);
    EXPECT_EQ(GetSampleByteSize(nn::audio::SampleFormat_PcmFloat), 4u);
}

TEST(GetSampleFormatTest, InvalidValueTest)
{
    EXPECT_DEATH_IF_SUPPORTED(GetSampleByteSize(nn::audio::SampleFormat_Invalid), "");
    EXPECT_DEATH_IF_SUPPORTED(GetSampleByteSize(static_cast<nn::audio::SampleFormat>(-1)), "");
}

/**
 * @brief       StopAudioOut に対するマルチスレッドアクセスの正常系テストです。
 */
TEST(StopAudioOutMultiThread, Success)
{
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t threadStack[32768];
    nn::os::ThreadType thread;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    nn::audio::AudioOut audioOut;

    nn::os::SystemEvent systemEventOut;

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEventOut, parameter));
    EXPECT_NE(systemEventOut.GetReadableHandle(), nn::os::InvalidNativeHandle);
    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(&audioOut), nullptr);

    nn::audio::AudioOutBuffer audioOutBuffer;
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));
    nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer);
    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(&audioOut), nullptr);
    nn::audio::AudioOutBuffer* pReleased = nullptr;

    systemEventOut.Wait();

    pReleased = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
    EXPECT_EQ(pReleased, &audioOutBuffer);
    pReleased = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
    EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(&audioOut), nullptr);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&thread,
        [](void* args)
        {
            auto pAudioOut = static_cast<nn::audio::AudioOut*>(args);
            nn::audio::StopAudioOut(pAudioOut);
        }, &audioOut, threadStack, sizeof(threadStack), nn::os::DefaultThreadPriority));

    nn::os::StartThread(&thread);
    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);

    nn::audio::CloseAudioOut(&audioOut);
}

/**
 * @brief   複数のインスタンスに対する OpenDefaultAudioOut / CloseAudioOut 呼び出しの正常系テストです。
 */
TEST(AudioOutMultiInstalceOpenClose, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    nn::audio::AudioOut audioOut[nn::audio::AudioOutCountMax];
    nn::os::SystemEvent systemEventOut[nn::audio::AudioOutCountMax];

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut[0], &systemEventOut[0], parameter));

    for(int i = 0; i < 16; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut[nn::audio::AudioOutCountMax - 1], &systemEventOut[nn::audio::AudioOutCountMax - 1], parameter));
        nn::audio::CloseAudioOut(&audioOut[nn::audio::AudioOutCountMax - 1]);
    }

    nn::audio::CloseAudioOut(&audioOut[0]);
}

/**
 * @brief       AudioOut によるマルチスレッドでの再生の正常系テストです。
 */
TEST(PlaybackAudioOutMultiThread, Success)
{
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t threadStack1[32768];
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t threadStack2[32768];
    nn::os::ThreadType thread1, thread2;

    const auto threadFunc = [](void* args)
    {
        NN_UNUSED(args);

        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        nn::audio::AudioOut audioOut;
        nn::os::SystemEvent systemEventOut;

        for(int i = 0; i < 10; ++i)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEventOut, parameter));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));

            nn::audio::AudioOutBuffer audioOutBuffer;
            // Set mininum buffer size
            nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), nn::audio::GetAudioOutChannelCount(&audioOut) * nn::audio::GetSampleByteSize(nn::audio::GetAudioOutSampleFormat(&audioOut)));
            nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer);

            const auto beginTick = nn::os::GetSystemTick();
            while(nn::os::GetSystemTick() - beginTick < nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(100)))
            {
                if(nn::audio::GetReleasedAudioOutBuffer(&audioOut) != nullptr)
                {
                    nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer);
                }

                // 偶数回目の時は SystemEvent がシグナル状態になるのを待つパターンでテスト
                if(i % 2 == 0)
                {
                    systemEventOut.Wait();
                }
            }

            nn::audio::StopAudioOut(&audioOut);
            nn::audio::CloseAudioOut(&audioOut);
        }
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&thread1, threadFunc, nullptr, threadStack1, sizeof(threadStack1), nn::os::DefaultThreadPriority));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&thread2, threadFunc, nullptr, threadStack2, sizeof(threadStack2), nn::os::DefaultThreadPriority));
    nn::os::SetThreadCoreMask(&thread1, 1, 1 << 1);
    nn::os::SetThreadCoreMask(&thread2, 2, 1 << 2);
    nn::os::StartThread(&thread1);
    nn::os::StartThread(&thread2);
    nn::os::WaitThread(&thread1);
    nn::os::WaitThread(&thread2);
    nn::os::DestroyThread(&thread1);
    nn::os::DestroyThread(&thread2);
}

/**
 * @brief       AudioOut で複数回の再生・停止を行う正常系テストです。
 */
TEST(MultipleStartStopAudioOutTest, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));

    for(int i = 0; i < 16; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::StartAudioOut(aout.Get()));
        nn::audio::StopAudioOut(aout.Get());
    }

    EXPECT_EQ(
        nn::audio::GetAudioOutState(aout.Get()), nn::audio::AudioOutState_Stopped);
}

/**
 * @brief       AudioOutBuffer の操作を行う API の事前条件テストです。
 */
TEST(AudioOutBuffer, Precondition)
{
    uint8_t* pBuffer = reinterpret_cast<uint8_t*>(nn::audio::AudioOutBuffer::AddressAlignment);
    const size_t bufferSize = nn::audio::AudioOutBuffer::SizeGranularity * 2;
    const auto channelCount = 2;
    const auto sampleSize = sizeof(int16_t);
    const auto sampleCount = 100;
    const size_t dataSize = channelCount * sampleSize * sampleCount;
    nn::audio::AudioOutBuffer outBuffer;

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutBufferInfo(nullptr, pBuffer, bufferSize, dataSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutBufferInfo(&outBuffer, nullptr, bufferSize, dataSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutBufferInfo(&outBuffer, nullptr, bufferSize, dataSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutBufferInfo(&outBuffer, pBuffer + 1, bufferSize, dataSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutBufferInfo(&outBuffer, pBuffer, bufferSize + 1, dataSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutBufferInfo(&outBuffer, pBuffer, bufferSize, bufferSize + 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutBufferInfo(&outBuffer, pBuffer, bufferSize, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioOutBufferDataPointer(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioOutBufferDataSize(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioOutBufferBufferSize(nullptr), "");
}

/**
 * @brief       AudioOutBuffer の操作を行う API の正常系テストです。
 */
TEST(AudioOutBuffer, Success)
{
    void* pBuffer = reinterpret_cast<void*>(nn::audio::AudioOutBuffer::AddressAlignment);
    const size_t bufferSize = nn::audio::AudioOutBuffer::SizeGranularity * 2;
    const auto channelCount = 2;
    const auto sampleSize = sizeof(int16_t);
    const auto sampleCount = 100;
    const size_t dataSize = channelCount * sampleSize * sampleCount;
    nn::audio::AudioOutBuffer outBuffer;

    nn::audio::SetAudioOutBufferInfo(&outBuffer, pBuffer, bufferSize, dataSize);
    EXPECT_EQ(pBuffer, nn::audio::GetAudioOutBufferDataPointer(&outBuffer));
    EXPECT_EQ(dataSize, nn::audio::GetAudioOutBufferDataSize(&outBuffer));
    EXPECT_EQ(bufferSize, nn::audio::GetAudioOutBufferBufferSize(&outBuffer));
}

TEST(GetAudioOutBufferCount, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioOutBufferCount(nullptr), "");
}

TEST(GetAudioOutBufferCount, Success)
{
    nn::audio::AudioOutParameter parameter;

    // Setup AudioOut
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), &systemEvent, parameter));

    // Check initial count
    EXPECT_EQ(0, nn::audio::GetAudioOutBufferCount(aout.Get()));

    // Append 1 AudioOutBuffer
    nn::audio::AudioOutBuffer audioOutBuffer;
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
    nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer); // 43 ms

    // Check AudioOut's buffer count is 1
    EXPECT_EQ(1, nn::audio::GetAudioOutBufferCount(aout.Get()));

    // Start AudioOut
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioOut(aout.Get()));

    // Wait for consuming AudioOutBuffer by AudioOut
    const auto RetryCountMax = 10; // 500 ms
    int i;
    for(i = 0; i < RetryCountMax; ++i)
    {
        const auto CurrentBufferCount = nn::audio::GetAudioOutBufferCount(aout.Get());

        EXPECT_TRUE(CurrentBufferCount == 1 || CurrentBufferCount == 0);

        if(CurrentBufferCount == 0)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
    }

    if(i == RetryCountMax - 1)
    {
        ASSERT_TRUE(false) << "AudioOutBufferCount is not 0.";
    }

    // Check AudioOutBufferCount is not changed even though GetReleasedAudioOutBuffer() is called
    EXPECT_NE(nullptr, nn::audio::GetReleasedAudioOutBuffer(aout.Get()));
    EXPECT_EQ(0, nn::audio::GetAudioOutBufferCount(aout.Get()));
}

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
TEST(GetAudioOutSampleCountPlayedSleepWake, Success)
{
    uint64_t testSampleCount = 0;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    nn::audio::AudioOut audioOut;
    nn::os::SystemEvent systemEventOut;
    int bufferCount = 1;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEventOut, parameter));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));
    EXPECT_EQ(0, nn::audio::GetAudioOutPlayedSampleCount(&audioOut));

    nn::audio::AudioOutBuffer audioOutBuffer;
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), nn::audio::GetAudioOutChannelCount(&audioOut) * nn::audio::GetSampleByteSize(nn::audio::GetAudioOutSampleFormat(&audioOut)) * 240);
    nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer);
    for(int j = 0; j < 4; ++j)
    {
        for(int i = 0; i < 10; ++i)
        {
            systemEventOut.Wait();
            if(nn::audio::GetReleasedAudioOutBuffer(&audioOut) != nullptr)
            {
                testSampleCount = nn::audio::GetAudioOutPlayedSampleCount(&audioOut);
                EXPECT_EQ(testSampleCount, (240 * (bufferCount)));
                nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer);
                ++bufferCount;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }

        NN_SDK_LOG("Current SampleCount: %llu\n - Please sleep the system\n", testSampleCount);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3000));
        testSampleCount = nn::audio::GetAudioOutPlayedSampleCount(&audioOut);
        NN_SDK_LOG("Please confirm that the current sample count is within 240 samples from the above: %llu\n", testSampleCount);
    }
    nn::audio::StopAudioOut(&audioOut);
    nn::audio::CloseAudioOut(&audioOut);
}

TEST(GetAudioOutSampleCountPlayed, Success)
{
    uint64_t testSampleCount = 0;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    nn::audio::AudioOut audioOut;
    nn::os::SystemEvent systemEventOut;
    int bufferCount = 0;
    nn::audio::AudioOutBuffer audioOutBuffer;
    // Set minimum buffer size
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEventOut, parameter));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));

    EXPECT_EQ(0, nn::audio::GetAudioOutPlayedSampleCount(&audioOut));
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), nn::audio::GetAudioOutChannelCount(&audioOut) * nn::audio::GetSampleByteSize(nn::audio::GetAudioOutSampleFormat(&audioOut)) * 240);
    nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer);
    for(int i = 0; i < 10; ++i)
    {
        systemEventOut.Wait();
        ASSERT_TRUE(nn::audio::GetReleasedAudioOutBuffer(&audioOut) != nullptr);
        testSampleCount = nn::audio::GetAudioOutPlayedSampleCount(&audioOut);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(25));
        EXPECT_EQ(testSampleCount, nn::audio::GetAudioOutPlayedSampleCount(&audioOut));
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer);
        ++bufferCount;
        EXPECT_EQ(testSampleCount, (240 * (bufferCount)));
    }
    nn::audio::StopAudioOut(&audioOut);
    nn::audio::CloseAudioOut(&audioOut);
}
#endif

TEST(FlushAudioOutBuffers, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::FlushAudioOutBuffers(nullptr), "");
}

// TODO: This API has not been implemented for Win32 yet.
#if !defined(NN_BUILD_CONFIG_OS_WIN)
TEST(FlushAudioOutBuffers, Success)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));

    // Setup AudioOutBuffers
    nn::audio::AudioOutBuffer audioOutBuffers[nn::audio::AudioOutBufferCountMax];
    for(auto& audioOutBuffer : audioOutBuffers)
    {
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
    }

    // AudioOut is stopped. (API should returns true if AudioOut is not started.)
    {
        int releasedBufferCount = 0;
        // There is no appended buffer in AudioOut
        EXPECT_TRUE(nn::audio::FlushAudioOutBuffers(aout.Get()));
        while(nn::audio::GetReleasedAudioOutBuffer(aout.Get()) != nullptr)
        {
            ++releasedBufferCount;
        }
        EXPECT_EQ(releasedBufferCount, 0);

        // There is 1 appended buffer in AudioOut
        releasedBufferCount = 0;
        EXPECT_TRUE(nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffers[0]));
        EXPECT_TRUE(nn::audio::FlushAudioOutBuffers(aout.Get()));
        while(nn::audio::GetReleasedAudioOutBuffer(aout.Get()) != nullptr)
        {
            ++releasedBufferCount;
        }
        EXPECT_EQ(releasedBufferCount, 1);

        // There are nn::audio::AudioOutBufferCountMax - 1 appended buffer in AudioOut
        releasedBufferCount = 0;
        for(int i = 0; i < nn::audio::AudioOutBufferCountMax - 1; ++i)
        {
            EXPECT_TRUE(nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffers[i]));
            EXPECT_TRUE(nn::audio::FlushAudioOutBuffers(aout.Get()));
        }
        while(nn::audio::GetReleasedAudioOutBuffer(aout.Get()) != nullptr)
        {
            ++releasedBufferCount;
        }
        EXPECT_EQ(releasedBufferCount, nn::audio::AudioOutBufferCountMax - 1);

        // There are nn::audio::AudioOutBufferCountMax appended buffer in AudioOut
        releasedBufferCount = 0;
        for(int i = 0; i < nn::audio::AudioOutBufferCountMax; ++i)
        {
            EXPECT_TRUE(nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffers[i]));
            EXPECT_TRUE(nn::audio::FlushAudioOutBuffers(aout.Get()));
        }
        while(nn::audio::GetReleasedAudioOutBuffer(aout.Get()) != nullptr)
        {
            ++releasedBufferCount;
        }
        EXPECT_EQ(releasedBufferCount, nn::audio::AudioOutBufferCountMax);
    }

    // AudioOut is started. (Check only that audio process does not abort.)
    // TODO: This API can not be called if AudioOut is PlayState_Started for now.
#if 0
    {
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::StartAudioOut(aout.Get()));

        // There is no appended buffer in AudioOut
        EXPECT_TRUE(nn::audio::FlushAudioOutBuffers(aout.Get()));
        while(nn::audio::GetReleasedAudioOutBuffer(aout.Get()) != nullptr);

        // There is 1 appended buffer in AudioOut
        EXPECT_TRUE(nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffers[0]));
        nn::audio::FlushAudioOutBuffers(aout.Get());
        while(nn::audio::GetReleasedAudioOutBuffer(aout.Get()) != nullptr);

        // There are nn::audio::AudioOutBufferCountMax - 1 appended buffer in AudioOut
        for(int i = 0; i < nn::audio::AudioOutBufferCountMax - 1; ++i)
        {
            EXPECT_TRUE(nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffers[i]));
        }
        nn::audio::FlushAudioOutBuffers(aout.Get());
        while(nn::audio::GetReleasedAudioOutBuffer(aout.Get()) != nullptr);
        return;

        // There are nn::audio::AudioOutBufferCountMax appended buffer in AudioOut
        for(int i = 0; i < nn::audio::AudioOutBufferCountMax; ++i)
        {
            EXPECT_TRUE(nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffers[i]));
        }
        nn::audio::FlushAudioOutBuffers(aout.Get());
        while(nn::audio::GetReleasedAudioOutBuffer(aout.Get()) != nullptr);
    }
#endif
}
#endif // !defined(NN_BUILD_CONFIG_OS_WIN)

TEST(SetAudioOutVolume, Precondition)
{
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);

    ScopedAudioOut aout;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));

#if defined(NN_SDK_BUILD_RELEASE)
    nn::audio::SetAudioOutVolume(aout.Get(), nn::audio::AudioOut::GetVolumeMin() - 0.1f);
    nn::audio::SetAudioOutVolume(aout.Get(), nn::audio::AudioOut::GetVolumeMax() + 0.1f);
#else
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutVolume(nullptr, nn::audio::AudioOut::GetVolumeMin()), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutVolume(aout.Get(), nn::audio::AudioOut::GetVolumeMin() - 0.1f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutVolume(aout.Get(), nn::audio::AudioOut::GetVolumeMax() + 0.1f), "");
#endif // defined(NN_SDK_BUILD_RELEASE)
}

TEST(SetAudioOutVolume, Success)
{
    // Setup AudioOut and playback sound by AudioOut
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    ScopedAudioOut aout;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::StartAudioOut(aout.Get()));
    nn::audio::AudioOutBuffer audioOutBuffer;
    nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), nn::audio::GetAudioOutChannelCount(aout.Get()) * nn::audio::GetSampleByteSize(nn::audio::GetAudioOutSampleFormat(aout.Get())) * 240);
    nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer);

    // Set volume by SetAudioOutVolume()
    const auto minVolume = nn::audio::AudioOut::GetVolumeMin();
    const auto maxVolume = nn::audio::AudioOut::GetVolumeMax();
    const auto midVolume = (maxVolume - minVolume) / 2.0f;

    nn::audio::SetAudioOutVolume(aout.Get(), minVolume);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

    nn::audio::SetAudioOutVolume(aout.Get(), midVolume);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

    nn::audio::SetAudioOutVolume(aout.Get(), maxVolume);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
}
