﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cmath>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/audio.h>

namespace {

    const int RenderRate = 48000;
    const int RenderCount = (RenderRate / 200);

    const char Title[] = "AudioAdspLoad";

    const int MaxVoiceCount = 192;
    int voiceCount = MaxVoiceCount / 2;

    // 96 sine frequencies representing 64 musical pitches to be played
    const int sineFrequencies[MaxVoiceCount] =
        { 261, 329, 523, 659, 1047, 1319, 293, 392,
        587, 784, 1175, 1568, 880, 293, 1760, 2349,
        440, 196, 987, 2093, 1976, 3136, 493, 2637,
        261, 329, 523, 659, 1047, 1319, 293, 392,
        587, 784, 1175, 1568, 880, 293, 1760, 2349,
        440, 196, 987, 2093, 1976, 3136, 493, 2637,
        261, 329, 523, 659, 1047, 1319, 293, 392,
        587, 784, 1175, 1568, 880, 293, 1760, 2349,
        440, 196, 987, 2093, 1976, 3136, 493, 2637,
        261, 329, 523, 659, 1047, 1319, 293, 392,
        587, 784, 1175, 1568, 880, 293, 1760, 2349 };

    char g_WorkBuffer[8 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];

    nn::mem::StandardAllocator g_Allocator;
    nn::mem::StandardAllocator g_WaveBufferAllocator;

}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t GenerateSineWave(void** data, int sampleRate, int frequency, int sampleCount)
{
    int16_t* p = static_cast<int16_t*>(g_WaveBufferAllocator.Allocate(sampleCount * sizeof(int16_t), nn::audio::BufferAlignSize));
    NN_ABORT_UNLESS_NOT_NULL(p);
    const float Pi = 3.1415926535897932384626433f;
    for( auto i = 0; i < sampleCount; ++i )
    {
        p[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i / sampleRate));
    }
    *data = p;
    return sampleCount * sizeof(int16_t);
}

void PrintUsage()
{
    NN_LOG("----------------------------------------------\n");
    NN_LOG("%s Sample\n", Title);
    NN_LOG("Press Start to Exit \n");
    NN_LOG("----------------------------------------------\n");
}

void AddVoice(nn::audio::AudioRendererConfig *pConfig, nn::audio::SubMixType *pSubMix, nn::audio::WaveBuffer *pWaveBufferSine, nn::audio::VoiceType *pVoiceSine, int sineSampleRate, float pitch) {

    nn::audio::AcquireVoiceSlot(pConfig, pVoiceSine, sineSampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
    nn::audio::SetVoiceDestination(pConfig, pVoiceSine, pSubMix);

    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::SetVoicePlayState(pVoiceSine, nn::audio::VoiceType::PlayState_Play);
    nn::audio::BiquadFilterParameter filter = { false,{ 14041, -28083, 14041 },{ 29547, -13563 } };
    nn::audio::SetVoiceBiquadFilterParameter(pVoiceSine, 0, filter);
    nn::audio::SetVoicePitch(pVoiceSine, pitch);
    nn::audio::SetVoiceMixVolume(pVoiceSine, pSubMix, 1.0f / 96.0f, 0, 0);
}

void ReleaseVoice(nn::audio::AudioRendererConfig *pConfig, nn::audio::VoiceType *pVoiceSine)
{
    nn::audio::ReleaseVoiceSlot(pConfig, pVoiceSine);
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));

    nn::hid::InitializeDebugPad();

    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA = nn::hid::KeyboardKey::A::Index;
    map.buttonB = nn::hid::KeyboardKey::B::Index;
    map.buttonX = nn::hid::KeyboardKey::X::Index;
    map.buttonY = nn::hid::KeyboardKey::Y::Index;
    map.buttonL = nn::hid::KeyboardKey::L::Index;
    map.buttonR = nn::hid::KeyboardKey::R::Index;
    map.buttonLeft = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
    nn::hid::DebugPadState debugPadState[nn::hid::DebugPadStateCountMax];

    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 11;
    parameter.voiceCount = MaxVoiceCount;
    parameter.subMixCount = 1;
    parameter.sinkCount = 1;
    parameter.effectCount = 5;
    parameter.performanceFrameCount = 5;

    int channelCount = 2;
    int8_t mainBus[2] = { 2, 3 };
    int8_t subBus[2] = { 0, 1 };

    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
        );

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    size_t alignment = nn::os::MemoryPageSize;
    NN_LOG("%x\n",workBufferSize);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, alignment);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;

    nn::audio::AudioRendererHandle handle;
    nn::Result result = nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 6);

    nn::audio::SubMixType subMix;
    nn::audio::AcquireSubMix(&config, &subMix, parameter.sampleRate, 4);

    nn::audio::DeviceSinkType deviceSink;
    result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::audio::SetSubMixDestination(&config, &subMix, &finalMix);
    nn::audio::SetSubMixMixVolume(&subMix, &finalMix, 0.4f, 0, mainBus[0]);
    nn::audio::SetSubMixMixVolume(&subMix, &finalMix, 0.4f, 0, mainBus[1]);

    nn::audio::DelayType delayA;
    const nn::TimeSpan delayTimeMax = nn::TimeSpan::FromSeconds(1);
    size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTimeMax, parameter.sampleRate, 2);
    void* delayBufferA = g_WaveBufferAllocator.Allocate(delaySize, nn::audio::BufferAlignSize);

    result = nn::audio::AddDelay(&config, &delayA, delayBufferA, delaySize, &subMix, delayTimeMax, 2);
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::audio::SetDelayInputOutput(&delayA, subBus, subBus, channelCount);

    nn::audio::SetDelayEnabled(&delayA, true);
    nn::audio::SetDelayTime(&delayA, nn::TimeSpan::FromMilliSeconds(88));
    nn::audio::SetDelayFeedbackGain(&delayA, 0.2f);
    nn::audio::SetDelayChannelSpread(&delayA, 0.5f);
    nn::audio::SetDelayLowPassAmount(&delayA, 0.6f);
    nn::audio::SetDelayDryGain(&delayA, 1.0f);
    nn::audio::SetDelayInGain(&delayA, 0.9f);

    nn::audio::DelayType delayB;
    void* delayBufferB = g_WaveBufferAllocator.Allocate(delaySize, nn::audio::BufferAlignSize);

    result = nn::audio::AddDelay(&config, &delayB, delayBufferB, delaySize, &subMix, delayTimeMax, 2);
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::audio::SetDelayInputOutput(&delayB, subBus, mainBus, channelCount);

    nn::audio::SetDelayEnabled(&delayB, true);
    nn::audio::SetDelayTime(&delayB, nn::TimeSpan::FromMilliSeconds(166));
    nn::audio::SetDelayFeedbackGain(&delayB, 0.4f);
    nn::audio::SetDelayChannelSpread(&delayB, 0.7f);
    nn::audio::SetDelayLowPassAmount(&delayB, 0.9f);
    nn::audio::SetDelayDryGain(&delayB, 0.9f);
    nn::audio::SetDelayInGain(&delayB, 0.9f);

    nn::audio::DelayType delayC;
    void* delayBufferC = g_WaveBufferAllocator.Allocate(delaySize, nn::audio::BufferAlignSize);

    result = nn::audio::AddDelay(&config, &delayC, delayBufferC, delaySize, &subMix, delayTimeMax, 2);
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::audio::SetDelayInputOutput(&delayC, mainBus, mainBus, channelCount);

    nn::audio::SetDelayEnabled(&delayC, true);
    nn::audio::SetDelayTime(&delayC, nn::TimeSpan::FromMilliSeconds(332));
    nn::audio::SetDelayFeedbackGain(&delayC, 0.6f);
    nn::audio::SetDelayChannelSpread(&delayC, 0.9f);
    nn::audio::SetDelayLowPassAmount(&delayC, 0.9f);
    nn::audio::SetDelayDryGain(&delayC, 0.8f);
    nn::audio::SetDelayInGain(&delayC, 0.9f);

    nn::audio::ReverbType reverbA;
    nn::audio::ReverbType reverbB;
    size_t reverbSize = nn::audio::GetRequiredBufferSizeForReverb(parameter.sampleRate, channelCount);
    void* reverbBufferA = g_WaveBufferAllocator.Allocate(reverbSize);
    NN_ABORT_UNLESS(reverbBufferA != nullptr);
    void* reverbBufferB = g_WaveBufferAllocator.Allocate(reverbSize);
    NN_ABORT_UNLESS(reverbBufferB != nullptr);

    result = nn::audio::AddReverb(&config, &reverbA, reverbBufferA, reverbSize, &subMix, 2);
    result = nn::audio::AddReverb(&config, &reverbB, reverbBufferB, reverbSize, &subMix, 2);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::audio::SetReverbInputOutput(&reverbA, mainBus, mainBus, sizeof(mainBus));
    nn::audio::SetReverbInputOutput(&reverbB, mainBus, mainBus, sizeof(mainBus));

    nn::audio::SetReverbEarlyMode(&reverbA, nn::audio::ReverbType::EarlyMode_Hall);
    nn::audio::SetReverbEarlyGain(&reverbA, 0.9f);
    nn::audio::SetReverbPredelayTime(&reverbA, nn::TimeSpan::FromMilliSeconds(1));
    nn::audio::SetReverbLateMode(&reverbA, nn::audio::ReverbType::LateMode_Hall);
    nn::audio::SetReverbLateGain(&reverbA, 0.0f);
    nn::audio::SetReverbDecayTime(&reverbA, nn::TimeSpan::FromSeconds(15));
    nn::audio::SetReverbHighFrequencyDecayRatio(&reverbA, 0.6f);
    nn::audio::SetReverbColoration(&reverbA, 0.4f);
    nn::audio::SetReverbReverbGain(&reverbA, 0.9f);
    nn::audio::SetReverbOutGain(&reverbA, 0.8f);
    nn::audio::SetReverbDryGain(&reverbA, 0.8f);
    nn::audio::SetReverbEnabled(&reverbA, true);

    nn::audio::SetReverbEarlyMode(&reverbB, nn::audio::ReverbType::EarlyMode_Hall);
    nn::audio::SetReverbEarlyGain(&reverbB, 0.0f);
    nn::audio::SetReverbPredelayTime(&reverbB, nn::TimeSpan::FromMilliSeconds(1));
    nn::audio::SetReverbLateMode(&reverbB, nn::audio::ReverbType::LateMode_Hall);
    nn::audio::SetReverbLateGain(&reverbB, 0.9f);
    nn::audio::SetReverbDecayTime(&reverbB, nn::TimeSpan::FromSeconds(15));
    nn::audio::SetReverbHighFrequencyDecayRatio(&reverbB, 0.6f);
    nn::audio::SetReverbColoration(&reverbB, 0.4f);
    nn::audio::SetReverbReverbGain(&reverbB, 0.9f);
    nn::audio::SetReverbOutGain(&reverbB, 0.8f);
    nn::audio::SetReverbDryGain(&reverbB, 0.8f);
    nn::audio::SetReverbEnabled(&reverbB, true);

    nn::audio::MemoryPoolType waveBufferMemoryPool;
    bool rt = AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::audio::VoiceType voiceSine[MaxVoiceCount];
    nn::audio::WaveBuffer waveBufferSine[MaxVoiceCount];
    void* dataSine[MaxVoiceCount];

    const int sineSampleRate = 48000;
    const int sineSampleCount = 32000;

    for( int i = 0; i < MaxVoiceCount; ++i )
    {
        std::size_t size = GenerateSineWave(&dataSine[i], sineSampleRate, sineFrequencies[i], sineSampleCount);

        waveBufferSine[i].buffer = dataSine[i];
        waveBufferSine[i].size = size;
        waveBufferSine[i].startSampleOffset = 0;
        waveBufferSine[i].endSampleOffset = sineSampleCount;
        waveBufferSine[i].loop = 0;
        waveBufferSine[i].isEndOfStream = false;

        if( i < MaxVoiceCount )
        {
            if( i < 16 )
            {
                AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 0.25f);
            }
            else if( i < 32 )
            {
                AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 0.5f);
            }
            else if( i < 48 )
            {
                AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 1.0f);
            }
            else if( i < 64 )
            {
                AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 2.0f);
            }
            else if( i < 80 )
            {
                AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 4.0f);
            }
            else
            {
                AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 3.0f);
            }
        }
    }

    size_t perfBufferSize = nn::audio::GetRequiredBufferSizeForPerformanceFrames(parameter);
    void* perfBuffer[2];
    perfBuffer[0] = g_Allocator.Allocate(perfBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(perfBuffer[0]);
    perfBuffer[1] = g_Allocator.Allocate(perfBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(perfBuffer[1]);
    int perfBufferIndex = 0;

    PrintUsage();
    int dumpCount = 0;
    int totalDumpCount = 0;

    int64_t prevPadSamplingNumber = 0;
    bool running = true;
    while( running )
    {
        systemEvent.Wait();

        nn::hid::GetDebugPadStates(debugPadState, nn::hid::DebugPadStateCountMax);
        int64_t padSamplingCount = debugPadState[0].samplingNumber - prevPadSamplingNumber;
        if( padSamplingCount >= nn::hid::DebugPadStateCountMax )
        {
            padSamplingCount = nn::hid::DebugPadStateCountMax - 1;
        }
        prevPadSamplingNumber = debugPadState[0].samplingNumber;

        nn::hid::DebugPadButtonSet debugPadButtonDown(debugPadState[0].buttons & ~debugPadState[padSamplingCount].buttons);

        for( int i = 0; i < MaxVoiceCount; ++i )
        {
            if( !nn::audio::IsVoiceValid(&voiceSine[i]) )
            {
                if( i < voiceCount )
                {
                    if( i < 16 )
                    {
                        AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 0.25f);
                    }
                    else if( i < 32 )
                    {
                        AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 0.5f);
                    }
                    else if( i < 48 )
                    {
                        AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 1.0f);
                    }
                    else if( i < 64 )
                    {
                        AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 2.0f);
                    }
                    else if( i < 80 )
                    {
                        AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 4.0f);
                    }
                    else
                    {
                        AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate, 3.0f);
                    }
                }
                else
                {
                    continue;
                }
            }
            else
            {
                if( i >= voiceCount )
                {
                    ReleaseVoice(&config, &voiceSine[i]);
                }
                else if( const nn::audio::WaveBuffer* pWaveBuffer = nn::audio::GetReleasedWaveBuffer(&voiceSine[i]) )
                {
                    nn::audio::AppendWaveBuffer(&voiceSine[i], pWaveBuffer);
                    nn::audio::AppendWaveBuffer(&voiceSine[i], pWaveBuffer);
                    nn::audio::AppendWaveBuffer(&voiceSine[i], pWaveBuffer);
                    nn::audio::AppendWaveBuffer(&voiceSine[i], pWaveBuffer);
                }
            }
        }
        for( int i = 0; i < MaxVoiceCount; ++i )
        {
            if( voiceCount > 0 && nn::audio::IsVoiceValid(&voiceSine[i]) )
            {
                if( nn::audio::GetVoiceMixVolume(&voiceSine[i], &subMix, 0, 0) != (1.0f / voiceCount) )
                {
                    nn::audio::SetVoiceMixVolume(&voiceSine[i], &subMix, (float) 1.0f / voiceCount, 0, 0);
                }
            }
        }
        result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        void* lastBuffer = nn::audio::SetPerformanceFrameBuffer(&config, perfBuffer[perfBufferIndex], perfBufferSize);
        perfBufferIndex++;
        perfBufferIndex %= 2;

        nn::audio::PerformanceInfo performanceInfo;
        if( lastBuffer == nullptr || performanceInfo.SetBuffer(lastBuffer, perfBufferSize) == false )
        {
            continue;
        }

        do
        {
            if( dumpCount++ > 100 )
            {
                if( performanceInfo.GetTotalProcessingTime() <= 4000 && voiceCount < MaxVoiceCount )
                {
                    voiceCount++;
                }
                if( performanceInfo.GetTotalProcessingTime() > 4000 && voiceCount > 0 )
                {
                    voiceCount--;
                }
                NN_LOG("App:[Total   ]:%d[usec] (total Dump Count:%d, Voice Count:%d)\n", performanceInfo.GetTotalProcessingTime(), ++totalDumpCount, voiceCount);
                dumpCount = 0;
            }
        } while( performanceInfo.MoveToNextFrame() );

        if( debugPadButtonDown.Test< ::nn::hid::DebugPadButton::Start >() )
        {
            running = false;
        }
    }

    NN_LOG("Sample Completed \n");

    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを解放します。
    for( int i = 0; i < MaxVoiceCount; ++i )
    {
        if( dataSine[i] )
        {
            g_WaveBufferAllocator.Free(dataSine[i]);
            dataSine[i] = nullptr;
        }
    }
    if( configBuffer )
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if( workBuffer )
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }
}  // NOLINT(readability/fn_size)
