﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/atk.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>


namespace
{

const int MemoryHeapSize = 32 * 1024 * 1024;

nnt::atk::util::FsCommonSetup   g_FsSetup;
nnt::atk::util::AtkCommonSetup   g_AtkSetup;

static char                 g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN static char g_HeapMemoryForMemoryPool[MemoryHeapSize];
nn::mem::StandardAllocator  g_AllocatorForMemoryPool;
nn::audio::MemoryPoolType   g_MemoryPoolForAllocator;

const char* GetResourceCompatibilityDirectory()
{
    static char path[nn::fs::EntryNameLengthMax];
    nn::util::SNPrintf(path, sizeof(path), "%s:/ResourceCompatibility/Outputs", g_FsSetup.TestBinariesMountName);
    return path;
}

const int CommonVersionCount = 6;
// バージョンは昇順に並べる
const char* const CommonVersionList[CommonVersionCount]
{
    "NXAddon/0_6_0",
    "NXAddon/0_9_0",
    "NXAddon/1_0_0",
    "NXAddon/1_1_0",
    "NXAddon/3_0_0",
    "NXAddon/3_1_0"
};

// テストで利用するバージョンインデックス設定
const int CrcCheckVersionIndex = 3;
const int StreamTrackCountChangeVersionIndex = 5;

const int PrefetchSoundArchiveVersionCount = 3;
// バージョンは昇順に並べる
const char* const PrefetchSoundArchiveVersionList[PrefetchSoundArchiveVersionCount]
{
    "NXAddon/1_1_0",
    "NXAddon/3_0_0",
    "NXAddon/3_1_0"
};

void LoadData()
{
    // 音源データの読み込み
    g_AtkSetup.LoadData("SEQ_MARIOKART");
    g_AtkSetup.LoadData("SE_YOSHI");
}

void LoadFileData(void** ppOutFileData, size_t* pOutFileSize, const char* const filePath, nn::mem::StandardAllocator& allocator)
{
    nn::fs::FileHandle fileHandle;
    nn::Result result = nn::fs::OpenFile(&fileHandle, filePath, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t sizeForFile;
    result = nn::fs::GetFileSize(&sizeForFile, fileHandle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *ppOutFileData = nnt::atk::util::AllocateUninitializedMemory(allocator, static_cast<size_t>(sizeForFile));
    result = nn::fs::ReadFile(fileHandle, 0, *ppOutFileData, static_cast<size_t>(sizeForFile));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(fileHandle);
    *pOutFileSize = static_cast<size_t>(sizeForFile);
}

void LoadFileData(void** ppFileData, const char* const filePath, nn::mem::StandardAllocator& allocator)
{
    size_t unusedDataSize = 0;
    LoadFileData(ppFileData, &unusedDataSize, filePath, allocator);
}

void UnloadFileData(void* pFileData, nn::mem::StandardAllocator& allocator)
{
    allocator.Free(pFileData);
}

void UpdateAndWait(nn::atk::SoundArchivePlayer& soundArchivePlayer)
{
    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
}

void WaitForSoundStop(nn::atk::SoundArchivePlayer& soundArchivePlayer)
{
    while(nn::atk::SoundSystem::GetVoiceCount() != 0)
    {
        UpdateAndWait(soundArchivePlayer);
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
void PlayAndUpdateForDeathTest( nn::atk::SoundArchivePlayer& soundArchivePlayer, const char* const labelName, nn::atk::SoundStartable::StartInfo* pStartInfo)
{
    nn::atk::SoundHandle soundHandle;
    soundArchivePlayer.StartSound( &soundHandle, labelName, pStartInfo).IsSuccess();

    bool isCommandReplyed = false;
    while(!nnt::atk::util::SendReplyCommand(&isCommandReplyed))
    {
        UpdateAndWait(soundArchivePlayer);
    }

    while(!isCommandReplyed)
    {
        UpdateAndWait(soundArchivePlayer);
    }

    // ヘッダのロ―ド時にアサーションに引っかかるため、少しだけ再生を続ける
    for(int j = 0; j < 100; j++)
    {
        UpdateAndWait(soundArchivePlayer);
    }
}
#endif

}

void CompareStreamSoundDataInfo(const nn::atk::StreamSoundDataInfo& expect, const nn::atk::StreamSoundDataInfo& actual)
{
    EXPECT_EQ(expect.channelCount, actual.channelCount);
    EXPECT_EQ(expect.loopEnd, actual.loopEnd);
    EXPECT_EQ(expect.loopStart, actual.loopStart);
    EXPECT_EQ(expect.loopFlag, actual.loopFlag);
    EXPECT_EQ(expect.sampleRate, actual.sampleRate);
}

void CompareWaveSoundDataInfo(const nn::atk::WaveSoundDataInfo& expect, const nn::atk::WaveSoundDataInfo& actual)
{
    EXPECT_EQ(expect.channelCount, actual.channelCount);
    EXPECT_EQ(expect.loopEnd, actual.loopEnd);
    EXPECT_EQ(expect.loopStart, actual.loopStart);
    EXPECT_EQ(expect.loopFlag, actual.loopFlag);
    EXPECT_EQ(expect.sampleRate, actual.sampleRate);
}

void CompareSoundArchivePlayerInfo(const nn::atk::SoundArchive::SoundArchivePlayerInfo& expect, const nn::atk::SoundArchive::SoundArchivePlayerInfo& actual,
        bool isStreamTrackCountCompared, int expectStreamTrackCount)
{
    EXPECT_EQ(expect.waveSoundCount, actual.waveSoundCount);
    EXPECT_EQ(expect.waveTrackCount, actual.waveTrackCount);
    EXPECT_EQ(expect.sequenceSoundCount, actual.sequenceSoundCount);
    EXPECT_EQ(expect.sequenceTrackCount, actual.sequenceTrackCount);
    EXPECT_EQ(expect.streamSoundCount, actual.streamSoundCount);
    EXPECT_EQ(expect.streamChannelCount, actual.streamChannelCount);

    if(isStreamTrackCountCompared)
    {
        EXPECT_EQ(expectStreamTrackCount, actual.streamTrackCount);
    }

    EXPECT_EQ(expect.streamBufferTimes, actual.streamBufferTimes);
    EXPECT_EQ(expect.isAdvancedWaveSoundEnabled, actual.isAdvancedWaveSoundEnabled);
}

TEST(ResourceCompatibility, SoundArchiveCompatibilityTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    nnt::atk::util::FsCommonSetup::InitializeParam fsParam;
    fsParam.SetTestBinariesDirectoryMounted(true);
    g_FsSetup.Initialize(fsParam);

    bool isFirstInfoRead = false;

    int firstBankCount = 0;
    int firstGroupCount = 0;
    int firstPlayerCount = 0;
    int firstSoundCount = 0;
    int firstSoundGroupCount = 0;
    int firstWaveArchiveCount = 0;
    nn::atk::StreamSoundDataInfo firstStreamSoundDataInfo;
    nn::atk::WaveSoundDataInfo firstWaveSoundDataInfo;
    nn::atk::SoundArchive::SoundArchivePlayerInfo firstSoundArchivePlayerInfo;
    int expectStreamTrackCount = 0;
    int versionListIndex = 0;

    for(auto version : CommonVersionList)
    {
        NN_LOG("%s SoundArchivePlayer Test.\n", version);

        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        char soundArchivePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(soundArchivePath, sizeof(soundArchivePath), "%s/%s/SoundArchive/ResourceCompatibility.bfsar", GetResourceCompatibilityDirectory(), version);
        atkParam.SetSoundArchivePath(soundArchivePath);
        g_AtkSetup.Initialize(atkParam, g_Allocator);
        g_AtkSetup.LoadLabelStringData(g_Allocator);
        LoadData();

        nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();

        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        // 同じデータを使うので基本的な情報の読み込みは一致している必要がある
        // 1 回目の SoundArchive の情報を取得する際には、後のバージョンの SoundArchive と比較するための情報を代入する
        if(!isFirstInfoRead)
        {
            firstBankCount = soundArchive.GetBankCount();
            firstGroupCount = soundArchive.GetGroupCount();
            firstPlayerCount = soundArchive.GetPlayerCount();
            firstSoundCount = soundArchive.GetSoundCount();
            firstSoundGroupCount =  soundArchive.GetSoundGroupCount();
            firstWaveArchiveCount = soundArchive.GetWaveArchiveCount();

            EXPECT_TRUE(soundArchivePlayer.ReadStreamSoundDataInfo(&firstStreamSoundDataInfo, "STRM_MARIOKART").IsSuccess());
            EXPECT_TRUE(soundArchivePlayer.ReadWaveSoundDataInfo(&firstWaveSoundDataInfo, "SE_YOSHI").IsSuccess());
            EXPECT_TRUE(soundArchive.ReadSoundArchivePlayerInfo(&firstSoundArchivePlayerInfo));

            isFirstInfoRead = true;
        }
        else
        {
            // 2 回目以降は SoundArchive の情報を比較する
            EXPECT_EQ(firstBankCount, soundArchive.GetBankCount());
            EXPECT_EQ(firstGroupCount, soundArchive.GetGroupCount());
            EXPECT_EQ(firstPlayerCount, soundArchive.GetPlayerCount());
            EXPECT_EQ(firstSoundCount, soundArchive.GetSoundCount());
            EXPECT_EQ(firstSoundGroupCount, soundArchive.GetSoundGroupCount());
            EXPECT_EQ(firstWaveArchiveCount, soundArchive.GetWaveArchiveCount());

            nn::atk::StreamSoundDataInfo testStreamSoundDataInfo;
            nn::atk::WaveSoundDataInfo testWaveSoundDataInfo;
            nn::atk::SoundArchive::SoundArchivePlayerInfo testSoundArchivePlayerInfo;

            EXPECT_TRUE(soundArchivePlayer.ReadStreamSoundDataInfo(&testStreamSoundDataInfo, "STRM_MARIOKART").IsSuccess());
            EXPECT_TRUE(soundArchivePlayer.ReadWaveSoundDataInfo(&testWaveSoundDataInfo, "SE_YOSHI").IsSuccess());
            EXPECT_TRUE(soundArchive.ReadSoundArchivePlayerInfo(&testSoundArchivePlayerInfo));

            CompareStreamSoundDataInfo(firstStreamSoundDataInfo, testStreamSoundDataInfo);
            CompareWaveSoundDataInfo(firstWaveSoundDataInfo, testWaveSoundDataInfo);
            // NXAddon 3.1.0 以降ストリームトラック数の修正が行われているため、比較対象の分岐を行います
            if(versionListIndex >= StreamTrackCountChangeVersionIndex)
            {
                if(versionListIndex == StreamTrackCountChangeVersionIndex)
                {
                    // Version 3.1.0 の値を保持して、それを比較対象として利用します
                    expectStreamTrackCount = testSoundArchivePlayerInfo.streamTrackCount;
                }
                CompareSoundArchivePlayerInfo(firstSoundArchivePlayerInfo, testSoundArchivePlayerInfo, true, expectStreamTrackCount);
            }
            else
            {
                CompareSoundArchivePlayerInfo(firstSoundArchivePlayerInfo, testSoundArchivePlayerInfo, false, 0);
            }
        }

        nn::atk::SoundHandle soundHandle[3];
        EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle[0], "SEQ_MARIOKART").IsSuccess());
        EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle[1], "SE_YOSHI").IsSuccess());
        EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle[2], "STRM_MARIOKART").IsSuccess());

        // ブロックロードなどを確かめるため、少しだけ再生を続ける
        for(int i = 0; i < 20; i++)
        {
            soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
            nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }

        g_AtkSetup.Finalize(g_Allocator);
        versionListIndex++;
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(ResourceCompatibility, LoadPrefetchCompatibilityTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_HeapMemoryForMemoryPool, MemoryHeapSize);
    nnt::atk::util::FsCommonSetup::InitializeParam fsParam;
    fsParam.SetTestBinariesDirectoryMounted(true);
    g_FsSetup.Initialize(fsParam);

    for(auto version : PrefetchSoundArchiveVersionList)
    {
        NN_LOG("%s PrefetchStream Test.\n", version);

        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        char soundArchivePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(soundArchivePath, sizeof(soundArchivePath), "%s/%s/PrefetchSoundArchive/PrefetchSoundArchive.bfsar", GetResourceCompatibilityDirectory(), version);
        atkParam.SetSoundArchivePath(soundArchivePath);
        g_AtkSetup.Initialize(atkParam, g_Allocator);
        g_AtkSetup.LoadLabelStringData(g_Allocator);
        g_AtkSetup.LoadData("STRM_MARIOKART_PREFETCH");

        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

        // 音源再生
        nn::atk::SoundHandle soundHandle;
        EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle, "STRM_MARIOKART_PREFETCH").IsSuccess());

        // ブロックロードなども確かめるため、少しだけ再生を続ける
        for(int i = 0; i < 20; i++)
        {
            UpdateAndWait(soundArchivePlayer);
        }

        soundHandle.Stop(0);
        WaitForSoundStop(soundArchivePlayer);

        g_AtkSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_AllocatorForMemoryPool.Finalize();
    g_Allocator.Finalize();
}

TEST(ResourceCompatibility, PrefetchStreamCompatibilityTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_HeapMemoryForMemoryPool, MemoryHeapSize);
    nnt::atk::util::FsCommonSetup::InitializeParam fsParam;
    fsParam.SetTestBinariesDirectoryMounted(true);
    g_FsSetup.Initialize(fsParam);

    for(auto version : CommonVersionList)
    {
        NN_LOG("%s Bfstp Test.\n", version);

        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        char soundArchivePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(soundArchivePath, sizeof(soundArchivePath), "%s/%s/SoundArchive/ResourceCompatibility.bfsar", GetResourceCompatibilityDirectory(), version);
        atkParam.SetSoundArchivePath(soundArchivePath);
        g_AtkSetup.Initialize(atkParam, g_Allocator);
        nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPoolForAllocator, g_HeapMemoryForMemoryPool, MemoryHeapSize);
        g_AtkSetup.LoadLabelStringData(g_Allocator);
        LoadData();

        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

        // ストリームデータ設定
        nn::atk::SoundStartable::StartInfo startInfo;
        startInfo.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
        char streamFilePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(streamFilePath, sizeof(streamFilePath), "%s/%s/PrefetchBfstp/kart_title.32.adpcm.bfstm", GetResourceCompatibilityDirectory(), version);
        startInfo.streamSoundInfo.externalPath = streamFilePath;

        // プリフェッチデータの読み込み
        char prefetchDataPath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(prefetchDataPath, sizeof(prefetchDataPath), "%s/%s/PrefetchBfstp//kart_title.32.adpcm.bfstp", GetResourceCompatibilityDirectory(), version);
        void* pMemoryForPrefetchData = nullptr;
        LoadFileData(&pMemoryForPrefetchData, prefetchDataPath, g_AllocatorForMemoryPool);
        startInfo.streamSoundInfo.prefetchData = pMemoryForPrefetchData;

        // 音源再生
        nn::atk::SoundHandle soundHandle;
        EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle, "STRM_MARIOKART", &startInfo).IsSuccess());

        // ブロックロードなども確かめるため、少しだけ再生を続ける
        for(int i = 0; i < 20; i++)
        {
            UpdateAndWait(soundArchivePlayer);
        }

        soundHandle.Stop(0);
        WaitForSoundStop(soundArchivePlayer);

        nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPoolForAllocator);
        UnloadFileData(pMemoryForPrefetchData, g_AllocatorForMemoryPool);
        g_AtkSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_AllocatorForMemoryPool.Finalize();
    g_Allocator.Finalize();
}

TEST(ResourceCompatibility, PrefetchStreamCrossVersionTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_HeapMemoryForMemoryPool, MemoryHeapSize);
    nnt::atk::util::FsCommonSetup::InitializeParam fsParam;
    fsParam.SetTestBinariesDirectoryMounted(true);
    g_FsSetup.Initialize(fsParam);

    const int checkVersionCount = 2;
    const char* checkVersion[checkVersionCount];
    // NXAddon 1.0.0 (CRC32 なし)
    checkVersion[0] = CommonVersionList[CrcCheckVersionIndex - 1];
    // NXAddon 1.1.0 (CRC32 あり)
    checkVersion[1] = CommonVersionList[CrcCheckVersionIndex];

    for(int i = 0; i < checkVersionCount; i++)
    {
        for(int j = 0; j < checkVersionCount; j++)
        {
            for(int k = 0; k < checkVersionCount; k++)
            {
                if( i == j && j == k)
                {
                    continue;
                }

                nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
                char soundArchivePath[nn::fs::EntryNameLengthMax];
                nn::util::SNPrintf(soundArchivePath, sizeof(soundArchivePath), "%s/%s/SoundArchive/ResourceCompatibility.bfsar", GetResourceCompatibilityDirectory(), checkVersion[i]);
                atkParam.SetSoundArchivePath(soundArchivePath);
                g_AtkSetup.Initialize(atkParam, g_Allocator);
                nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPoolForAllocator, g_HeapMemoryForMemoryPool, MemoryHeapSize);
                g_AtkSetup.LoadLabelStringData(g_Allocator);
                LoadData();

                nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

                // ストリームデータ設定
                nn::atk::SoundStartable::StartInfo startInfo;
                startInfo.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
                char streamFilePath[nn::fs::EntryNameLengthMax];
                nn::util::SNPrintf(streamFilePath, sizeof(streamFilePath), "%s/%s/PrefetchBfstp//kart_title.32.adpcm.bfstm", GetResourceCompatibilityDirectory(), checkVersion[j]);
                startInfo.streamSoundInfo.externalPath = streamFilePath;

                // プリフェッチデータの読み込み
                char prefetchDataPath[nn::fs::EntryNameLengthMax];
                nn::util::SNPrintf(prefetchDataPath, sizeof(prefetchDataPath), "%s/%s/PrefetchBfstp//kart_title.32.adpcm.bfstp", GetResourceCompatibilityDirectory(), checkVersion[k]);
                void* pMemoryForPrefetchData = nullptr;
                LoadFileData(&pMemoryForPrefetchData, prefetchDataPath, g_AllocatorForMemoryPool);
                startInfo.streamSoundInfo.prefetchData = pMemoryForPrefetchData;

                // 音源再生
                nn::atk::SoundHandle soundHandle;
                EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle, "STRM_MARIOKART", &startInfo).IsSuccess());

                // ブロックロードなども確かめるため、少しだけ再生を続ける
                for(int frame = 0; frame < 20; frame++)
                {
                    UpdateAndWait(soundArchivePlayer);
                }

                soundHandle.Stop(0);
                WaitForSoundStop(soundArchivePlayer);

                nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPoolForAllocator);
                UnloadFileData(pMemoryForPrefetchData, g_AllocatorForMemoryPool);
                g_AtkSetup.Finalize(g_Allocator);
            }
        }
    }
    g_FsSetup.Finalize();
    g_AllocatorForMemoryPool.Finalize();
    g_Allocator.Finalize();
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(ResourceCompatibility, PrefetchDataDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_HeapMemoryForMemoryPool, MemoryHeapSize);
    nnt::atk::util::FsCommonSetup::InitializeParam fsParam;
    fsParam.SetTestBinariesDirectoryMounted(true);
    g_FsSetup.Initialize(fsParam);

    // CRC が存在するのは NXAddon1.1.0 以降
    for(int i = CrcCheckVersionIndex; i < CommonVersionCount; i++)
    {
        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        char soundArchivePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(soundArchivePath, sizeof(soundArchivePath), "%s/%s/SoundArchive/ResourceCompatibility.bfsar", GetResourceCompatibilityDirectory(), CommonVersionList[i]);
        atkParam.SetSoundArchivePath(soundArchivePath);
        g_AtkSetup.Initialize(atkParam, g_Allocator);
        nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPoolForAllocator, g_HeapMemoryForMemoryPool, MemoryHeapSize);
        g_AtkSetup.LoadLabelStringData(g_Allocator);
        LoadData();

        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

        // ストリームデータ設定
        nn::atk::SoundStartable::StartInfo startInfo;
        startInfo.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
        char streamFilePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(streamFilePath, sizeof(streamFilePath), "%s/%s/PrefetchBfstp//kart_title.32.adpcm.bfstm", GetResourceCompatibilityDirectory(), CommonVersionList[i]);
        startInfo.streamSoundInfo.externalPath = streamFilePath;

        // プリフェッチデータの読み込み
        char prefetchDataPath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(prefetchDataPath, sizeof(prefetchDataPath), "%s/%s/PrefetchBfstp//kart_title_region.adpcm.bfstp", GetResourceCompatibilityDirectory(), CommonVersionList[i]);
        void* pMemoryForPrefetchData = nullptr;
        LoadFileData(&pMemoryForPrefetchData, prefetchDataPath, g_AllocatorForMemoryPool);
        startInfo.streamSoundInfo.prefetchData = pMemoryForPrefetchData;

        // 音源再生
        nn::atk::SoundHandle soundHandle;
        EXPECT_DEATH_IF_SUPPORTED(PlayAndUpdateForDeathTest(soundArchivePlayer, "STRM_MARIOKART", &startInfo), "");

        nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPoolForAllocator);
        UnloadFileData(pMemoryForPrefetchData, g_AllocatorForMemoryPool);
        g_AtkSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_AllocatorForMemoryPool.Finalize();
    g_Allocator.Finalize();
}
#endif

TEST(ResourceCompatibility, BfwavCompatibilityTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_HeapMemoryForMemoryPool, MemoryHeapSize);
    nnt::atk::util::FsCommonSetup::InitializeParam fsParam;
    fsParam.SetTestBinariesDirectoryMounted(true);
    g_FsSetup.Initialize(fsParam);

    for(auto version : CommonVersionList)
    {
        NN_LOG("%s Bfwav Test.\n", version);

        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        char soundArchivePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(soundArchivePath, sizeof(soundArchivePath), "%s/%s/SoundArchive/ResourceCompatibility.bfsar", GetResourceCompatibilityDirectory(), version);
        atkParam.SetSoundArchivePath(soundArchivePath);
        g_AtkSetup.Initialize(atkParam, g_Allocator);
        nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPoolForAllocator, g_HeapMemoryForMemoryPool, MemoryHeapSize);
        g_AtkSetup.LoadLabelStringData(g_Allocator);
        LoadData();

        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

        // bfwav データ設定
        nn::atk::SoundStartable::StartInfo startInfo;
        startInfo.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_WaveSoundInfo;
        char waveFilePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(waveFilePath, sizeof(waveFilePath), "%s/%s/Cache/ResourceCompatibility/snare.dspadpcm.bfwav", GetResourceCompatibilityDirectory(), version);
        void* pMemoryForBfwav = nullptr;
        LoadFileData(&pMemoryForBfwav, waveFilePath, g_AllocatorForMemoryPool);
        startInfo.waveSoundInfo.waveAddress = pMemoryForBfwav;

        nn::atk::SoundHandle soundHandle;
        EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle, "SE_YOSHI", &startInfo).IsSuccess());

        // 少しだけ再生を続ける
        for(int i = 0; i < 20; i++)
        {
            UpdateAndWait(soundArchivePlayer);
        }

        soundHandle.Stop(0);
        WaitForSoundStop(soundArchivePlayer);

        nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPoolForAllocator);
        UnloadFileData(pMemoryForBfwav, g_AllocatorForMemoryPool);
        g_AtkSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_AllocatorForMemoryPool.Finalize();
    g_Allocator.Finalize();
}

TEST(ResourceCompatibility, WaveArchiveCompatibilityTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_HeapMemoryForMemoryPool, MemoryHeapSize);
    nnt::atk::util::FsCommonSetup::InitializeParam fsParam;
    fsParam.SetTestBinariesDirectoryMounted(true);
    g_FsSetup.Initialize(fsParam);

    for(auto version : CommonVersionList)
    {
        NN_LOG("%s Bfwar Test.\n", version);

        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        char soundArchivePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(soundArchivePath, sizeof(soundArchivePath), "%s/%s/SoundArchive/ResourceCompatibility.bfsar", GetResourceCompatibilityDirectory(), version);
        atkParam.SetSoundArchivePath(soundArchivePath);
        g_AtkSetup.Initialize(atkParam, g_Allocator);
        nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPoolForAllocator, g_HeapMemoryForMemoryPool, MemoryHeapSize);
        g_AtkSetup.LoadLabelStringData(g_Allocator);
        LoadData();

        nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

        // 外部 bfwar ファイルの利用
        char archiveFilePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(archiveFilePath, sizeof(archiveFilePath), "%s/%s/Cache/ResourceCompatibility/WARC_BANK_BGM.bfwar", GetResourceCompatibilityDirectory(), version);
        void* pMemoryForBfwar = nullptr;
        size_t sizeForBfwar = 0;
        LoadFileData(&pMemoryForBfwar, &sizeForBfwar, archiveFilePath, g_AllocatorForMemoryPool);

        uint32_t itemId = soundArchive.GetItemId("WARC_BANK_BGM");
        uint32_t fileId = soundArchive.GetItemFileId(itemId);

        nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();
        soundDataManager.SetFileAddress(fileId, pMemoryForBfwar);

        nn::atk::SoundHandle soundHandle;
        EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle, "SEQ_MARIOKART").IsSuccess());

        // 少しだけ再生を続ける
        for(int i = 0; i < 20; i++)
        {
            UpdateAndWait(soundArchivePlayer);
        }

        soundHandle.Stop(0);
        WaitForSoundStop(soundArchivePlayer);

        soundDataManager.InvalidateSoundData(pMemoryForBfwar, static_cast<size_t>(sizeForBfwar));

        nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPoolForAllocator);
        UnloadFileData(pMemoryForBfwar, g_AllocatorForMemoryPool);
        g_AtkSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_AllocatorForMemoryPool.Finalize();
    g_Allocator.Finalize();
}

TEST(ResourceCompatibility, BankCompatibilityTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_HeapMemoryForMemoryPool, MemoryHeapSize);
    nnt::atk::util::FsCommonSetup::InitializeParam fsParam;
    fsParam.SetTestBinariesDirectoryMounted(true);
    g_FsSetup.Initialize(fsParam);

    for(auto version : CommonVersionList)
    {
        NN_LOG("%s Bfbnk Test.\n", version);

        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        char soundArchivePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(soundArchivePath, sizeof(soundArchivePath), "%s/%s/SoundArchive/ResourceCompatibility.bfsar", GetResourceCompatibilityDirectory(), version);
        atkParam.SetSoundArchivePath(soundArchivePath);
        g_AtkSetup.Initialize(atkParam, g_Allocator);
        nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPoolForAllocator, g_HeapMemoryForMemoryPool, MemoryHeapSize);
        g_AtkSetup.LoadLabelStringData(g_Allocator);
        LoadData();

        nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

        // 外部 bfbnk ファイルの利用
        char bankFilePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(bankFilePath, sizeof(bankFilePath), "%s/%s/Cache/ResourceCompatibility/BANK_BGM.bfbnk", GetResourceCompatibilityDirectory(), version);
        void* pMemoryForBfbnk = nullptr;
        size_t sizeForBfbnk = 0;
        LoadFileData(&pMemoryForBfbnk, &sizeForBfbnk, bankFilePath, g_AllocatorForMemoryPool);

        uint32_t itemId = soundArchive.GetItemId("BANK_BGM");
        uint32_t fileId = soundArchive.GetItemFileId(itemId);

        nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();
        soundDataManager.SetFileAddress(fileId, pMemoryForBfbnk);

        nn::atk::SoundHandle soundHandle;
        EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle, "SEQ_MARIOKART").IsSuccess());

        // 少しだけ再生を続ける
        for(int i = 0; i < 20; i++)
        {
            UpdateAndWait(soundArchivePlayer);
        }

        soundHandle.Stop(0);
        WaitForSoundStop(soundArchivePlayer);

        soundDataManager.InvalidateSoundData(pMemoryForBfbnk, static_cast<size_t>(sizeForBfbnk));

        nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPoolForAllocator);
        UnloadFileData(pMemoryForBfbnk, g_AllocatorForMemoryPool);
        g_AtkSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_AllocatorForMemoryPool.Finalize();
    g_Allocator.Finalize();
}

TEST(ResourceCompatibility, UserGroupCompatibilityTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_HeapMemoryForMemoryPool, MemoryHeapSize);
    nnt::atk::util::FsCommonSetup::InitializeParam fsParam;
    fsParam.SetTestBinariesDirectoryMounted(true);
    g_FsSetup.Initialize(fsParam);

    for(auto version : CommonVersionList)
    {
        NN_LOG("%s Bfgrp Test.\n", version);

        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        char soundArchivePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(soundArchivePath, sizeof(soundArchivePath), "%s/%s/SoundArchive/ResourceCompatibility.bfsar", GetResourceCompatibilityDirectory(), version);
        atkParam.SetSoundArchivePath(soundArchivePath);
        g_AtkSetup.Initialize(atkParam, g_Allocator);
        nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPoolForAllocator, g_HeapMemoryForMemoryPool, MemoryHeapSize);
        g_AtkSetup.LoadLabelStringData(g_Allocator);
        LoadData();

        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

        // 外部 bfgrp ファイルの利用
        char groupFilePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(groupFilePath, sizeof(groupFilePath), "%s/%s/SoundArchive/userManagementFiles/GROUP_USER.bfgrp", GetResourceCompatibilityDirectory(), version);
        void* pMemoryForBfgrp = nullptr;
        size_t sizeForBfgrp = 0;
        LoadFileData(&pMemoryForBfgrp, &sizeForBfgrp, groupFilePath, g_AllocatorForMemoryPool);

        nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();
        soundDataManager.SetFileAddressInGroupFile(pMemoryForBfgrp, sizeForBfgrp);

        nn::atk::SoundHandle soundHandle[2];
        EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle[0], "SE_YOSHI_USERGROUP").IsSuccess());
        EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle[1], "SEQ_MARIOKART_USERGROUP").IsSuccess());

        // 少しだけ再生を続ける
        for(int i = 0; i < 20; i++)
        {
            UpdateAndWait(soundArchivePlayer);
        }

        soundHandle[0].Stop(0);
        soundHandle[1].Stop(0);
        WaitForSoundStop(soundArchivePlayer);

        soundDataManager.InvalidateSoundData(pMemoryForBfgrp, static_cast<size_t>(sizeForBfgrp));

        nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPoolForAllocator);
        UnloadFileData(pMemoryForBfgrp, g_AllocatorForMemoryPool);
        g_AtkSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_AllocatorForMemoryPool.Finalize();
    g_Allocator.Finalize();
}
