﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "testData.h"

/*

//////////////////////////////////////////////////////////////////////////////////////////////
// テストの説明
//
// 外部リソースの利用を想定したプロファイルテストです。
// nact のみの実行を想定しているテストであるため、現状 sln にプロジェクトを含めていません。
// Atk の TestBinaries フォルダを TestBinaries:/ とした時、
// 以下のファイルが存在する状態でビルドすると、テストのビルドおよびプロジェクトのコンバートが行われます。
// 1. TestBinaries:/testAtk_ExternalResourceProfiler/testData.h
// 2. TestBinaries:/testAtk_ExternalResourceProfiler/Resources/testData.fspj
//
// 1. のヘッダファイルおいて 2. に含まれるリソースの音源番号、ラベル、
// および再生のタイムアウト時間を設定することで、テストを動作させることができます。
// 以下に testData.h の書き方の例を示します。
//////////////////////////////////////////////////////////////////////////////////////////////

#pragma once
#include "testData.fsid"

#include <nn/nn_TimeSpan.h>

namespace {
    const unsigned int TestDataCount = 2;
    const unsigned int TestData[TestDataCount] = { SEQ_MARIOKART, STRM_MARIOKART };
    const char TestLabel[TestDataCount][128] = { "SeqMarioKart", "StrmMarioKart" };
    const nn::TimeSpan TestTimeOut[TestDataCount] = { nn::TimeSpan::FromSeconds(10), nn::TimeSpan::FromSeconds(10) };
}

*/

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_FileLogger.h>
#include <nnt/atkUtil/testAtk_ProfileStatistics.h>
#include <nnt/atkUtil/testAtk_Constants.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>

#include <nn/mem.h>

namespace {

    const int MemoryHeapSize = 32 * 1024 * 1024;
    const char ResourcesMountName[] = "Resources";
    const char InitialPerformanceLog[] = "Number of Voices,Atk Frame(us),Ren Total Frame(us),Voice Frame(us),Voice Frame Per Voice(us),Sub Mix Frame(us),Final Mix Frame(us),Sink Frame(us),Sound Thread Interval(us)\n";
    const int ProfileCountMax = 32;
    nn::atk::SoundProfile  g_SoundProfile[ProfileCountMax];
    nn::atk::ProfileReader g_ProfileReader;
    nn::os::Tick           g_PreviousAtkFrameProcessBegin;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    const char* TestNameSuffix = "vcmd";
#else
    const char* TestNameSuffix = "";
#endif

    class FsExternalProfilerSetup : public nnt::atk::util::FsCommonSetup
    {
    public:
        virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            // リソースデータの存在するディレクトリのマウント
            // + 1 は終端文字の分
            char testBinariesDirectory[nn::fs::EntryNameLengthMax + 1];
            NN_ABORT_UNLESS(nnt::atk::util::GetAtkTestBinariesDirectoryPath(testBinariesDirectory, sizeof(testBinariesDirectory)), "Cannot get TestBinaries directory path.");
            char resourceDirectory[nn::fs::EntryNameLengthMax + 1];
            nn::util::SNPrintf(resourceDirectory, sizeof(resourceDirectory), "%sExternalResourceProfiler\\Resources\\output\\", testBinariesDirectory);
            nnt::atk::util::MountHostPcFs(ResourcesMountName, resourceDirectory);

            nnt::atk::util::FsCommonSetup::InitializeParam param;
            param.SetTestResultsDirectoryMounted(true);
            param.SetRomMounted(false);
            nnt::atk::util::FsCommonSetup::Initialize(param);
        }
        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
        {
            nnt::atk::util::UnmountFs(ResourcesMountName);
            nnt::atk::util::FsCommonSetup::Finalize();
        }
    };

    class AtkExternalProfilerSetup : public nnt::atk::util::AtkCommonSetup
    {
    public:
        virtual void Initialize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
            soundSystemParam.enableProfiler = true;
            soundSystemParam.enableVoiceDrop = false;
            nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
            initializeParam.SetSoundSystemParam(soundSystemParam);
            // + 1 は終端文字の分
            char soundArchivePath[nn::fs::EntryNameLengthMax + 1];
            nn::util::SNPrintf(soundArchivePath, sizeof(soundArchivePath), "%s:/testData.bfsar", ResourcesMountName);
            initializeParam.SetSoundArchivePath(soundArchivePath);

            AtkCommonSetup::Initialize(initializeParam, allocator);

            nn::atk::SoundSystem::RegisterProfileReader(g_ProfileReader);
        }
        virtual void Finalize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundSystem::UnregisterProfileReader(g_ProfileReader);
            AtkCommonSetup::Finalize(allocator);
        }
    };

    FsExternalProfilerSetup g_FsSetup;
    AtkExternalProfilerSetup g_AtkSetup;

    static char g_HeapMemory[MemoryHeapSize];
    nn::mem::StandardAllocator  g_Allocator;

void LoadData(int i)
{
    // 音源データの読み込み
    g_AtkSetup.LoadData(TestData[i], TestLabel[i]);
}

void WriteProfileLog(nnt::atk::util::FileLogger& logger, nn::atk::SoundProfile& profile)
{
    float voiceFramePerVoice = profile.totalVoiceCount == 0 ? 0.0f : static_cast<float>(profile.voiceProcess.GetSpan().GetMicroSeconds()) / profile.totalVoiceCount;

    nn::Result result = logger.Write("%d,%lld,%lld,%lld,%f,%lld,%lld,%lld,%lld\n",
                profile.totalVoiceCount,
                profile.nwFrameProcess.GetSpan().GetMicroSeconds(),
                profile.rendererFrameProcess.GetSpan().GetMicroSeconds(),
                profile.voiceProcess.GetSpan().GetMicroSeconds(),
                voiceFramePerVoice,
                profile.mainMixProcess.GetSpan().GetMicroSeconds(),
                profile.finalMixProcess.GetSpan().GetMicroSeconds(),
                profile.sinkProcess.GetSpan().GetMicroSeconds(),
                profile.GetNwFrameProcessInterval(g_PreviousAtkFrameProcessBegin).GetMicroSeconds());
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot write file.");
}

void WaitForEffectClear(nn::atk::EffectBase& effect)
{
    const auto WaitTime = nn::TimeSpan::FromMilliSeconds(16);
    const auto TimeOut = nn::TimeSpan::FromMilliSeconds(5000);

    for(auto elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TimeOut; elapsed += WaitTime)
    {
        nn::os::SleepThread(WaitTime);
        if(effect.IsRemovable()) break;
    }
    EXPECT_TRUE(effect.IsRemovable()) << "Effect clear timed out.";
}

}

TEST(ExternalResourceProfiler, DataProfileTest)
{
    for(int i = 0; i < TestDataCount; i++)
    {
        g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
        g_FsSetup.Initialize();

        char targetName[32];
        NN_ABORT_UNLESS(nnt::atk::util::GetTargetName(targetName, sizeof(targetName)), "Cannot get target name.");
        // + 1 は終端文字の分
        char filePath[nn::fs::EntryNameLengthMax + 1];
        nn::util::SNPrintf(filePath, sizeof(filePath), "%s:/Atk%s_ExternalResourceProfiler_%s_%s_%s_None.csv",
            g_FsSetup.TestResultsMountName, TestNameSuffix, targetName, nnt::atk::util::BuildType, TestLabel[i]);
        nnt::atk::util::FileLogger fileLogger(filePath);
        NN_ABORT_UNLESS(fileLogger.IsOpened(), "Cannot open file.");

        g_AtkSetup.Initialize(g_Allocator);
        LoadData(i);

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        bool startResult = soundArchivePlayer.StartSound( &soundHandle, TestData[i]).IsSuccess();
        NN_ABORT_UNLESS(startResult, "StartSound(%s) failed.", TestLabel[i]);
        NN_LOG("Start %s.\n", TestLabel[i]);

        nn::Result result = fileLogger.Write(InitialPerformanceLog);
        NN_ABORT_UNLESS(result.IsSuccess(), "Cannot write file.");

        const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
        int totalProfileCount = 0;

        // 最初の 1 サンプル目は GetNwFrameProcessInterval() が異常値を示すため除外
        nnt::atk::util::ProfileStatistics statistics(1);

        for(nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TestTimeOut[i]; elapsed += WaitTime)
        {
            if(!soundHandle.IsAttachedSound())
            {
                break;
            }
            soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
            nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
            nn::os::SleepThread(WaitTime);

            int profileCount = g_ProfileReader.Read(g_SoundProfile, ProfileCountMax);
            EXPECT_GE(profileCount, 0);
            EXPECT_LE(profileCount, ProfileCountMax);
            for (int j = 0; j < profileCount; j++)
            {
                nn::atk::SoundProfile profile = g_SoundProfile[j];

                WriteProfileLog(fileLogger, profile);
                statistics.AddProfile(profile, g_PreviousAtkFrameProcessBegin);

                g_PreviousAtkFrameProcessBegin = profile.nwFrameProcess.begin;
            }

            totalProfileCount += profileCount;
        }

        EXPECT_NE(0, totalProfileCount);

        char labelName[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(labelName, sizeof(labelName) - 1, "%s%s", TestLabel[i], TestNameSuffix);
        statistics.PrintMeanProfile(labelName);
        statistics.PrintMaxProfile(labelName);

        g_AtkSetup.Finalize(g_Allocator);
        fileLogger.Close();
        g_FsSetup.Finalize();
        g_Allocator.Finalize();
    }
}

TEST(ExternalResourceProfiler, DataProfileTestWithDelay)
{
    for(int i = 0; i < TestDataCount; i++)
    {
        g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
        g_FsSetup.Initialize();

        char targetName[32];
        NN_ABORT_UNLESS(nnt::atk::util::GetTargetName(targetName, sizeof(targetName)), "Cannot get target name.");
        // + 1 は終端文字の分
        char filePath[nn::fs::EntryNameLengthMax + 1];
        nn::util::SNPrintf(filePath, sizeof(filePath), "%s:/Atk%s_ExternalResourceProfiler_%s_%s_%s_Delay.csv",
            g_FsSetup.TestResultsMountName, TestNameSuffix, targetName, nnt::atk::util::BuildType, TestLabel[i]);
        nnt::atk::util::FileLogger fileLogger(filePath);
        NN_ABORT_UNLESS(fileLogger.IsOpened(), "Cannot open file.");

        g_AtkSetup.Initialize(g_Allocator);
        LoadData(i);

        nn::atk::EffectDelay delay;
        delay.SetChannelMode(nn::atk::EffectDelay::ChannelMode_2Ch);
        delay.SetSampleRate(nn::atk::EffectDelay::SampleRate_48000);

        size_t delayBufferSize = nn::util::align_up(delay.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity);
        void* delayBuffer = g_Allocator.Allocate(delayBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(delayBuffer);
        nn::audio::MemoryPoolType memoryPoolForDelay;
        nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForDelay, delayBuffer, delayBufferSize);
        delay.SetEnabled(true);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &delay, delayBuffer, delayBufferSize));

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        bool startResult = soundArchivePlayer.StartSound( &soundHandle, TestData[i]).IsSuccess();
        NN_ABORT_UNLESS(startResult, "StartSound(%s) failed.", TestLabel[i]);
        soundHandle.SetMainSend(-1.0f);
        soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);

        NN_LOG("Start %s.\n", TestLabel[i]);
        nn::Result result = fileLogger.Write(InitialPerformanceLog);
        NN_ABORT_UNLESS(result.IsSuccess(), "Cannot write file.");

        const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
        int totalProfileCount = 0;

        // 最初の 1 サンプル目は GetNwFrameProcessInterval() が異常値を示すため除外
        nnt::atk::util::ProfileStatistics statistics(1);

        for(nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TestTimeOut[i]; elapsed += WaitTime)
        {
            if(!soundHandle.IsAttachedSound())
            {
                break;
            }
            soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
            nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
            nn::os::SleepThread(WaitTime);

            int profileCount = g_ProfileReader.Read(g_SoundProfile, ProfileCountMax);
            EXPECT_GE(profileCount, 0);
            EXPECT_LE(profileCount, ProfileCountMax);
            for (int j = 0; j < profileCount; j++)
            {
                nn::atk::SoundProfile profile = g_SoundProfile[j];

                WriteProfileLog(fileLogger, profile);
                statistics.AddProfile(profile, g_PreviousAtkFrameProcessBegin);

                g_PreviousAtkFrameProcessBegin = profile.nwFrameProcess.begin;
            }

            totalProfileCount += profileCount;
        }

        EXPECT_NE(0, totalProfileCount);

        char labelName[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(labelName, sizeof(labelName) - 1, "%s%sDelay", TestLabel[i], TestNameSuffix);
        statistics.PrintMeanProfile(labelName);
        statistics.PrintMaxProfile(labelName);

        delay.SetEnabled(false);
        WaitForEffectClear(delay);

        nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
        nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForDelay);
        g_AtkSetup.Finalize(g_Allocator);
        fileLogger.Close();
        g_FsSetup.Finalize();
        g_Allocator.Finalize();
    }
}

TEST(ExternalResourceProfiler, DataProfileTestWithReverb)
{
    for(int i = 0; i < TestDataCount; i++)
    {
        g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
        g_FsSetup.Initialize();

        char targetName[32];
        NN_ABORT_UNLESS(nnt::atk::util::GetTargetName(targetName, sizeof(targetName)), "Cannot get target name.");
        // + 1 は終端文字の分
        char filePath[nn::fs::EntryNameLengthMax + 1];
        nn::util::SNPrintf(filePath, sizeof(filePath), "%s:/Atk%s_ExternalResourceProfiler_%s_%s_%s_Reverb.csv",
            g_FsSetup.TestResultsMountName, TestNameSuffix, targetName, nnt::atk::util::BuildType, TestLabel[i]);

        nnt::atk::util::FileLogger fileLogger(filePath);
        NN_ABORT_UNLESS(fileLogger.IsOpened(), "Cannot open file.");

        g_AtkSetup.Initialize(g_Allocator);
        LoadData(i);

        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode(nn::atk::EffectReverb::ChannelMode_2Ch);
        reverb.SetSampleRate(nn::atk::EffectReverb::SampleRate_48000);

        size_t reverbBufferSize = nn::util::align_up(reverb.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity);
        void* reverbBuffer = g_Allocator.Allocate(reverbBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(reverbBuffer);
        nn::audio::MemoryPoolType memoryPoolForReverb;
        nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForReverb, reverbBuffer, reverbBufferSize);
        reverb.SetEnabled(true);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &reverb, reverbBuffer, reverbBufferSize));

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        bool startResult = soundArchivePlayer.StartSound( &soundHandle, TestData[i]).IsSuccess();
        NN_ABORT_UNLESS(startResult, "StartSound(%s) failed.", TestLabel[i]);
        soundHandle.SetMainSend(-1.0f);
        soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);

        NN_LOG("Start %s.\n", TestLabel[i]);

        nn::Result result = fileLogger.Write(InitialPerformanceLog);
        NN_ABORT_UNLESS(result.IsSuccess(), "Cannot write file.");

        const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
        int totalProfileCount = 0;

        // 最初の 1 サンプル目は GetNwFrameProcessInterval() が異常値を示すため除外
        nnt::atk::util::ProfileStatistics statistics(1);

        for(nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TestTimeOut[i]; elapsed += WaitTime)
        {
            if(!soundHandle.IsAttachedSound())
            {
                break;
            }
            soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
            nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
            nn::os::SleepThread(WaitTime);

            int profileCount = g_ProfileReader.Read(g_SoundProfile, ProfileCountMax);
            EXPECT_GE(profileCount, 0);
            EXPECT_LE(profileCount, ProfileCountMax);
            for (int j = 0; j < profileCount; j++)
            {
                nn::atk::SoundProfile profile = g_SoundProfile[j];

                WriteProfileLog(fileLogger, profile);
                statistics.AddProfile(profile, g_PreviousAtkFrameProcessBegin);

                g_PreviousAtkFrameProcessBegin = profile.nwFrameProcess.begin;
            }

            totalProfileCount += profileCount;
        }

        EXPECT_NE(0, totalProfileCount);

        char labelName[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(labelName, sizeof(labelName) - 1, "%s%sReverb", TestLabel[i], TestNameSuffix);
        statistics.PrintMeanProfile(labelName);
        statistics.PrintMaxProfile(labelName);

        reverb.SetEnabled(false);
        WaitForEffectClear(reverb);

        nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
        nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForReverb);
        g_AtkSetup.Finalize(g_Allocator);
        fileLogger.Close();
        g_FsSetup.Finalize();
        g_Allocator.Finalize();
    }
}

TEST(ExternalResourceProfiler, DataProfileTestWithI3dl2Reverb)
{
    for(int i = 0; i < TestDataCount; i++)
    {
        g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
        g_FsSetup.Initialize();

        char targetName[32];
        NN_ABORT_UNLESS(nnt::atk::util::GetTargetName(targetName, sizeof(targetName)), "Cannot get target name.");
        // + 1 は終端文字の分
        char filePath[nn::fs::EntryNameLengthMax + 1];
        nn::util::SNPrintf(filePath, sizeof(filePath), "%s:/Atk%s_ExternalResourceProfiler_%s_%s_%s_I3dl2Reverb.csv",
            g_FsSetup.TestResultsMountName, TestNameSuffix, targetName, nnt::atk::util::BuildType, TestLabel[i]);

        nnt::atk::util::FileLogger fileLogger(filePath);
        NN_ABORT_UNLESS(fileLogger.IsOpened(), "Cannot open file.");

        g_AtkSetup.Initialize(g_Allocator);
        LoadData(i);

        nn::atk::EffectI3dl2Reverb reverb;
        reverb.SetChannelMode(nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch);
        reverb.SetSampleRate(nn::atk::EffectI3dl2Reverb::SampleRate_48000);

        size_t reverbBufferSize = nn::util::align_up(reverb.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity);
        void* reverbBuffer = g_Allocator.Allocate(reverbBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(reverbBuffer);
        nn::audio::MemoryPoolType memoryPoolForI3dl2Reverb;
        nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForI3dl2Reverb, reverbBuffer, reverbBufferSize);
        reverb.SetEnabled(true);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &reverb, reverbBuffer, reverbBufferSize));

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        bool startResult = soundArchivePlayer.StartSound( &soundHandle, TestData[i]).IsSuccess();
        NN_ABORT_UNLESS(startResult, "StartSound(%s) failed.", TestLabel[i]);
        soundHandle.SetMainSend(-1.0f);
        soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);

        NN_LOG("Start %s.\n", TestLabel[i]);

        nn::Result result = fileLogger.Write(InitialPerformanceLog);
        NN_ABORT_UNLESS(result.IsSuccess(), "Cannot write file.");

        const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
        int totalProfileCount = 0;

        // 最初の 1 サンプル目は GetNwFrameProcessInterval() が異常値を示すため除外
        nnt::atk::util::ProfileStatistics statistics(1);

        for(nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TestTimeOut[i]; elapsed += WaitTime)
        {
            if(!soundHandle.IsAttachedSound())
            {
                break;
            }
            soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
            nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
            nn::os::SleepThread(WaitTime);

            int profileCount = g_ProfileReader.Read(g_SoundProfile, ProfileCountMax);
            EXPECT_GE(profileCount, 0);
            EXPECT_LE(profileCount, ProfileCountMax);
            for (int j = 0; j < profileCount; j++)
            {
                nn::atk::SoundProfile profile = g_SoundProfile[j];

                WriteProfileLog(fileLogger, profile);
                statistics.AddProfile(profile, g_PreviousAtkFrameProcessBegin);

                g_PreviousAtkFrameProcessBegin = profile.nwFrameProcess.begin;
            }

            totalProfileCount += profileCount;
        }

        EXPECT_NE(0, totalProfileCount);

        char labelName[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(labelName, sizeof(labelName) - 1, "%s%sI3dl2Reverb", TestLabel[i], TestNameSuffix);
        statistics.PrintMeanProfile(labelName);
        statistics.PrintMaxProfile(labelName);

        reverb.SetEnabled(false);
        WaitForEffectClear(reverb);

        nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
        nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForI3dl2Reverb);
        g_AtkSetup.Finalize(g_Allocator);
        fileLogger.Close();
        g_FsSetup.Finalize();
        g_Allocator.Finalize();
    }
}
