﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// viewer ライブラリには Release ビルドにおける実態がないので、テストすべてを Release でビルドしないようにする
#if !defined(NN_SDK_BUILD_RELEASE)
#include "common.fsid"

#include <nn/atk.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>
#include <nn/atk/viewer/atk_SoundEdit.h>
#include <nn/atk/viewer/atk_Viewer.h>

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>

namespace {

const size_t MemoryHeapSize = 64 * 1024 * 1024;

nnt::atk::util::FsCommonSetup   g_FsSetup;
nnt::atk::util::AtkCommonSetup   g_AtkSetup;

static char                 g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

}

TEST(SoundArchiveEditor, InitializationDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    size_t bufferSize = 32 * 1024 * 1024;
    void* buffer = g_Allocator.Allocate(bufferSize);
    NN_ABORT_UNLESS_NOT_NULL(buffer);

    nn::atk::viewer::SoundArchiveEditor editor;
    nn::atk::viewer::SoundArchiveEditor::Configs configs;

    nn::atk::viewer::Result result;
    EXPECT_DEATH_IF_SUPPORTED(editor.Initialize(nullptr, bufferSize, &g_AtkSetup.GetSoundArchive(), &g_AtkSetup.GetSoundArchivePlayer(), configs) , "");
    EXPECT_DEATH_IF_SUPPORTED(editor.Initialize(buffer, 0, &g_AtkSetup.GetSoundArchive(), &g_AtkSetup.GetSoundArchivePlayer(), configs), "");
    EXPECT_DEATH_IF_SUPPORTED(editor.Initialize(buffer, bufferSize, nullptr, &g_AtkSetup.GetSoundArchivePlayer(), configs), "");
    EXPECT_DEATH_IF_SUPPORTED(editor.Initialize(buffer, bufferSize, &g_AtkSetup.GetSoundArchive(), nullptr, configs), "");

    g_Allocator.Free(buffer);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundArchiveEditor, InitializationTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    size_t bufferSize = 32 * 1024 * 1024;
    void* buffer = g_Allocator.Allocate(bufferSize);
    NN_ABORT_UNLESS_NOT_NULL(buffer);

    nn::atk::viewer::SoundArchiveEditor editor;
    nn::atk::viewer::SoundArchiveEditor::Configs configs;

    // 通常の初期化
    nn::atk::viewer::Result result;
    result = editor.Initialize(buffer, bufferSize, &g_AtkSetup.GetSoundArchive(), &g_AtkSetup.GetSoundArchivePlayer(), configs);
    EXPECT_TRUE(result.IsSucceeded());
    editor.Finalize();

    // 1byte バッファ位置をずらした初期化
    void* alignedBuffer = static_cast<void*>(static_cast<char*>(buffer) + 1);
    size_t alignedBufferSize = bufferSize - 1;
    result = editor.Initialize(alignedBuffer, alignedBufferSize, &g_AtkSetup.GetSoundArchive(), &g_AtkSetup.GetSoundArchivePlayer(), configs);
    EXPECT_TRUE(result.IsSucceeded());
    editor.Finalize();

    // バッファサイズが 4096 の倍数, バッファアドレスの下三桁が 0xF3E - 0xFFF の間,
    // SoundArhchiveEditor::Configs で設定する maxItemName が 48 以上である初期化の退行テスト (for SIGLO-36107)
    void* specificBuffer = static_cast<void*>(static_cast<char*>(buffer) + 0xFA8);
    size_t specificBufferSize = 10 * 1024 * 1024;
    result = editor.Initialize(specificBuffer, specificBufferSize, &g_AtkSetup.GetSoundArchive(), &g_AtkSetup.GetSoundArchivePlayer(), configs);
    EXPECT_TRUE(result.IsSucceeded());
    editor.Finalize();

    g_Allocator.Free(buffer);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif
#if defined( NN_ATK_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION )
TEST(SoundEdit, ChangeWorkerThreadCoreNumber)
{
    nnt::atk::util::OnPreAtkTest();
    const int WorkerThreadCoreNumber = 1;

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    // インゲーム編集機能の初期化
    nn::atk::viewer::SoundEdit soundEdit;
    nn::atk::viewer::SoundEdit::Option option;
    std::size_t memSizeForSoundEdit = soundEdit.GetRequiredMemorySize( option );
    void* pMemoryForSoundEdit = g_Allocator.Allocate( memSizeForSoundEdit );

    nn::atk::viewer::SoundEdit::InitializeArg initArgs;
    initArgs.buffer                 = pMemoryForSoundEdit;
    initArgs.bufferSize             = memSizeForSoundEdit;
    initArgs.soundArchive           = &g_AtkSetup.GetSoundArchive();
    initArgs.soundArchivePlayer     = &g_AtkSetup.GetSoundArchivePlayer();
    initArgs.workerThreadCoreNumber = WorkerThreadCoreNumber;
    soundEdit.Initialize( initArgs, option );
    soundEdit.Start();

    // ワーカースレッドの動作コア番号の確認
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // ワーカースレッドの動作開始待ち
    EXPECT_EQ(soundEdit.GetWorkerThreadCoreNumber(), WorkerThreadCoreNumber);

    // インゲーム編集機能の終了処理
    soundEdit.Stop();
    soundEdit.Finalize();
    g_Allocator.Free(pMemoryForSoundEdit);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif
