﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string> // std::string
#include <algorithm> // std::min

#include <nn/nn_Macro.h> // NN_*
#include <nn/nn_Assert.h> // NN_ASSERT*
#include <nn/nn_Log.h> // NN_LOG
#include <nn/util/util_FormatString.h> // nn::util::SNPrintf

#include <nn/codec/codec_OpusCommon.h>
#include <nn/codec/codec_OpusEncoderTypes.h>
#include <nn/codec/detail/codec_OpusEncoderTypesInternal.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>

#include <opus.h>

namespace nnt {
namespace codec {
namespace util {
namespace {

nn::codec::OpusCodingMode GetCodingModeFromToc(uint8_t toc) NN_NOEXCEPT
{
    const auto config = toc >> 3;
    if (config < 12)
    {
        return nn::codec::OpusCodingMode_Silk;
    }
    else if (config < 16)
    {
        return nn::codec::OpusCodingMode_Hybrid;
    }
    return nn::codec::OpusCodingMode_Celt;
}

nn::codec::OpusBandwidth GetBandwidthFromToc(uint8_t toc) NN_NOEXCEPT
{
    const auto config = toc >> 3;
    if (config < 12) // Silk
    {
        switch(config >> 2)
        {
        case 0:
            return nn::codec::OpusBandwidth_NarrowBand;
        case 1:
            return nn::codec::OpusBandwidth_MediumBand;
        case 2:
            return nn::codec::OpusBandwidth_WideBand;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    else if (config < 16) // Hybrid
    {
        switch(config >> 1)
        {
        case 6:
            return nn::codec::OpusBandwidth_SuperWideBand;
        case 7:
            return nn::codec::OpusBandwidth_FullBand;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    else // Celt
    {
        switch((config >> 2) & 0x3)
        {
        case 0:
            return nn::codec::OpusBandwidth_NarrowBand;
        case 1:
            return nn::codec::OpusBandwidth_WideBand;
        case 2:
            return nn::codec::OpusBandwidth_SuperWideBand;
        case 3:
            return nn::codec::OpusBandwidth_FullBand;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

int GetFrameFromToc(uint8_t toc) NN_NOEXCEPT
{
    const auto config = toc >> 3;
    if (config < 12) // Silk
    {
        switch(config & 0x3)
        {
        case 0:
            return 10000;
        case 1:
            return 20000;
        case 2:
            return 40000;
        case 3:
            return 60000;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    else if (config < 16) // Hybrid
    {
        switch(config & 0x1)
        {
        case 0:
            return 10000;
        case 1:
            return 20000;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    else // Celt
    {
        switch(config & 0x3)
        {
        case 0:
            return 2500;
        case 1:
            return 5000;
        case 2:
            return 10000;
        case 3:
            return 20000;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

int GetChannelCountFromToc(uint8_t toc) NN_NOEXCEPT
{
    return ((toc & 0x4) >> 2) + 1;
}

uint8_t GetFrameCodeFromToc(uint8_t toc) NN_NOEXCEPT
{
    return toc & 0x3;
}

std::string GetCodingModeString(nn::codec::OpusCodingMode codingMode) NN_NOEXCEPT
{
    switch(codingMode)
    {
    case nn::codec::OpusCodingMode_Celt:
        return "Celt";
    case nn::codec::OpusCodingMode_Silk:
        return "Silk";
    case nn::codec::OpusCodingMode_Hybrid:
        return "Hybrid";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

std::string GetBandwidthString(nn::codec::OpusBandwidth bandwidth) NN_NOEXCEPT
{
    switch(bandwidth)
    {
    case nn::codec::OpusBandwidth_NarrowBand:
        return "Narrow Band";
    case nn::codec::OpusBandwidth_MediumBand:
        return "Medium Band";
    case nn::codec::OpusBandwidth_WideBand:
        return "Wide Band";
    case nn::codec::OpusBandwidth_SuperWideBand:
        return "Super-Wide Band";
    case nn::codec::OpusBandwidth_FullBand:
        return "Full Band";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

std::string GetFrameCodeString(uint8_t frameCode) NN_NOEXCEPT
{
    switch(frameCode)
    {
    case 0:
        return "One Frame";
    case 1:
        return "Two Frames (Same Size)";
    case 2:
        return "Two Frames (Different Size)";
    case 3:
        return "An Arbitrary Number of Frames";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void DumpOpusPacketToc(uint8_t toc) NN_NOEXCEPT
{
    const auto codingMode = GetCodingModeFromToc(toc);
    const std::string codingModeString = GetCodingModeString(codingMode);
    const auto bandwidth = GetBandwidthFromToc(toc);
    const std::string bandwidthString = GetBandwidthString(bandwidth);
    const auto frame = GetFrameFromToc(toc);
    const int channelCount = GetChannelCountFromToc(toc);
    const auto frameCode = GetFrameCodeFromToc(toc);
    const std::string frameCodeString = GetFrameCodeString(frameCode);
    NN_LOG("\tCodingMode  : %s\n", codingModeString.c_str());
    NN_LOG("\tFrame       : %d\n", frame);
    NN_LOG("\tBandwidth   : %s\n", bandwidthString.c_str());
    NN_LOG("\tChannelCount: %d\n", channelCount);
    NN_LOG("\tFrameCode   : %s\n", frameCodeString.c_str());
}

std::size_t GetOpusPacketSize(std::size_t *pSizeRead, const uint8_t* frame, std::size_t size) NN_NOEXCEPT
{
    const auto frameBase = frame;
    auto packetSize = 0u;
    *pSizeRead = 0;
    const auto lowerPartOfSize = *frame++;
    packetSize += lowerPartOfSize;
    if (lowerPartOfSize >= 252 && size >= 2)
    {
        const auto higherPartOfSize = *frame++;
        packetSize += higherPartOfSize * 4;
    }
    *pSizeRead = frame - frameBase;
    return packetSize;
}

}

////////////////////////////////////////////////////////////////////////////////
// libopus ==> nn::codec
////////////////////////////////////////////////////////////////////////////////
nn::codec::OpusResult ConvertResult(int result) NN_NOEXCEPT
{
    switch(result)
    {
    case OPUS_OK:
        return nn::codec::OpusResult_Success;
    case OPUS_BAD_ARG:
        return nn::codec::OpusResult_InvalidChannelCount;
    case OPUS_BUFFER_TOO_SMALL:
        return nn::codec::OpusResult_InsufficientOpusBuffer;
    case OPUS_INTERNAL_ERROR:
        return nn::codec::OpusResult_InternalError;
    case OPUS_INVALID_PACKET:
        return nn::codec::OpusResult_BrokenData;
    case OPUS_UNIMPLEMENTED:
        return nn::codec::OpusResult_UnsupportedFormat;
    case OPUS_INVALID_STATE:
        return nn::codec::OpusResult_InvalidSampleRate;
    case OPUS_ALLOC_FAIL:
        return nn::codec::OpusResult_InvalidWorkBuffer;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

nn::codec::OpusBandwidth ConvertBandwidth(int bandwidth) NN_NOEXCEPT
{
    switch(bandwidth)
    {
    case OPUS_BANDWIDTH_NARROWBAND:
        return nn::codec::OpusBandwidth_NarrowBand;
    case OPUS_BANDWIDTH_MEDIUMBAND:
        return nn::codec::OpusBandwidth_MediumBand;
    case OPUS_BANDWIDTH_WIDEBAND:
        return nn::codec::OpusBandwidth_WideBand;
    case OPUS_BANDWIDTH_SUPERWIDEBAND:
        return nn::codec::OpusBandwidth_SuperWideBand;
    case OPUS_BANDWIDTH_FULLBAND:
        return nn::codec::OpusBandwidth_FullBand;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void DumpOpusPacket(const uint8_t* packet, std::size_t size, int streamCount) NN_NOEXCEPT
{
    NN_UNUSED(size);
    using OpusHeader = nn::codec::detail::OpusPacketInternal::Header;
    const auto opusHeaderSize = sizeof(OpusHeader);
    OpusHeader header;
    nn::codec::detail::OpusPacketInternal::GetHeaderFromPacket(&header, packet);
    NN_ASSERT_LESS_EQUAL(header.packetSize + opusHeaderSize, size);
    auto p = packet + opusHeaderSize;
    std::size_t remainingBytes = header.packetSize;
    NN_LOG("* Total Packet Size : %d\n", static_cast<int>(header.packetSize + opusHeaderSize));
    NN_LOG("* Stream Count      : %d\n", streamCount);
    for (auto s = 0; s < streamCount; ++s)
    {
        NN_LOG("[Stream-%d]\n", s);
        const auto packetHead = p;
        // TOC のダンプ
        const auto toc = *p++;
        remainingBytes--;
        DumpOpusPacketToc(toc);
        // パケット長の解析
        const auto frameCode = GetFrameCodeFromToc(toc);
        const bool isSelfDelimited = s != streamCount - 1;
        std::size_t packetBodySize = 0u;
        std::size_t paddingSize = 0u;
        int frameCount = 1;
        bool isVbr = false; // "cbr" at default
        switch(frameCode)
        {
        case 0:
            // Do nothing.
            break;
        case 1:
            frameCount = 2;
            break;
        case 2:
            {
                isVbr = true; // "vbr"
                std::size_t readSize = 0u;
                packetBodySize += GetOpusPacketSize(&readSize, p, remainingBytes);
                p += readSize;
                remainingBytes -= readSize;
            }
            break;
        case 3:
            {
                const auto frameCountByte = *p++;
                remainingBytes--;
                isVbr = (frameCountByte & 0x80) != 0;
                const bool isPadded = (frameCountByte & 0x40) != 0;
                frameCount = frameCountByte & 0x3f;
                if (isPadded)
                {
                    std::size_t readSize = 0u;
                    paddingSize = GetOpusPacketSize(&readSize, p, remainingBytes);
                    p += readSize;
                    remainingBytes -= readSize;
                }
                if (isVbr)
                {
                    for (auto k = 0; k < frameCount - 1; ++k)
                    {
                        std::size_t readSize = 0u;
                        packetBodySize += GetOpusPacketSize(&readSize, p, remainingBytes);
                        p += readSize;
                        remainingBytes -= readSize;
                    }
                }
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
        // Not last
        if (isSelfDelimited)
        {
            std::size_t readSize = 0u;
            auto packetBodySizeInFrame = GetOpusPacketSize(&readSize, p, remainingBytes);
            p += readSize;
            remainingBytes -= readSize;
            if (!isVbr)
            {
                packetBodySizeInFrame *= frameCount;
            }
            packetBodySize += packetBodySizeInFrame;
        }
        // Last
        else
        {
            packetBodySize += remainingBytes;
        }
        NN_LOG("\tFrameCount  : %d\n", frameCount);

        const auto skipCount = packetBodySize + paddingSize;
        p += skipCount;
        remainingBytes -= skipCount;

        const auto packetSize = static_cast<std::size_t>(p - packetHead);
        NN_LOG("\tPacketSize  : %d\n", static_cast<int>(packetSize));
        NN_LOG("\t\tHeaderSize  : %d\n", static_cast<int>(packetSize - packetBodySize - paddingSize));
        NN_LOG("\t\tDataSize    : %d\n", static_cast<int>(packetBodySize));
        NN_LOG("\t\tPaddingSize : %d\n", static_cast<int>(paddingSize));
        {
            const std::size_t packetDumpCountMax = 8u;
            std::string packetDumpString = "\tPacketData  : ";

            for (auto k = 0u; k < std::min(packetDumpCountMax, packetSize); ++k)
            {
                const std::size_t twoDigitSize = 3u;
                char twoDigitString[twoDigitSize];
                nn::util::SNPrintf(twoDigitString, twoDigitSize, "%02x", packetHead[k]);
                packetDumpString += std::string(twoDigitString) + " ";
            }
            if (packetSize > packetDumpCountMax)
            {
                packetDumpString += "...";
            }
            NN_LOG("%s\n", packetDumpString.c_str());
        }
    }
    NN_ASSERT(remainingBytes == 0, "%d", static_cast<int>(remainingBytes));
} // NOLINT(readability/fn_size)

}}} // nnt::codec::util

