﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{StatTracker.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a helper class that sets up command buffers to report GPU
 *  statistic counters and reset those counters. It also provides functionality
 *  to print those counters to the screen using a given DebugTextRenderer.
 */

#include <nvn/nvn_FuncPtrInline.h>
#include <nn/nn_Assert.h>
#include <nvntutorial/StatTracker.h>
#include <nvntutorial/MemoryPool.h>
#include <nvntutorial/TutorialUtil.h>

static const size_t g_CommandMemorySize = 1024;
static const size_t g_ControlMemorySize = 1024;

/*
 * StatTracker Constructor
 * -----------------------
 * Setup some default values for the tracker.
 */
StatTracker::StatTracker() : m_pDevice(NULL),
                             m_CounterFlags(CounterFlags_All_Counters),
                             m_pReportControlMemoryPool(NULL),
                             m_pResetControlMemoryPool(NULL)
{
}

/*
 * StatTracker::Init
 * -----------------
 * Allocates memory for, initializes, and records the command
 * buffers to reset and report the the GPU counters.
 */
void StatTracker::Init(NVNdevice* pDevice, int flags)
{
    m_pDevice = pDevice;
    m_CounterFlags = flags;

        /* Query the device for the counter alignment. */
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_COUNTER_ALIGNMENT, &m_CounterAlignment);
    size_t counterSize = (sizeof(StatCounters) / sizeof(uint64_t)) * m_CounterAlignment;

        /* Query the device for the command and control memory alignment. */
    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

        /* Allocate the command and control memory. */
    m_MemoryPool.Init(NULL,
                      Align(g_CommandMemorySize, commandBufferCommandAlignment) * 2 +
                      Align(counterSize, 4),
                      NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
                      m_pDevice);

    m_pReportControlMemoryPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);
    m_pResetControlMemoryPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

        /* Initialize the sync object. */
    nvnSyncInitialize(&m_CommandBufferSync, m_pDevice);

        /* Initialize the command buffers. */
    nvnCommandBufferInitialize(&m_ReportCommandBuffer, m_pDevice);
    nvnCommandBufferInitialize(&m_ResetCommandBuffer, m_pDevice);

        /* Add the command and control memory to the command buffers. */
    size_t offset = m_MemoryPool.GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);
    nvnCommandBufferAddCommandMemory(&m_ReportCommandBuffer, m_MemoryPool.GetMemoryPool(), offset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_ReportCommandBuffer, m_pReportControlMemoryPool, g_ControlMemorySize);

    offset = m_MemoryPool.GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);
    nvnCommandBufferAddCommandMemory(&m_ResetCommandBuffer, m_MemoryPool.GetMemoryPool(), offset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_ResetCommandBuffer, m_pResetControlMemoryPool, g_ControlMemorySize);


        /* Setup and NVNbuffer for the counters to be reported into. */
    NVNbufferBuilder bufferBuilder;
    nvnBufferBuilderSetDevice(&bufferBuilder, m_pDevice);
    nvnBufferBuilderSetDefaults(&bufferBuilder);
    nvnBufferBuilderSetStorage(&bufferBuilder, m_MemoryPool.GetMemoryPool(), m_MemoryPool.GetNewMemoryChunkOffset(counterSize, 4), counterSize);

    if (!nvnBufferInitialize(&m_CounterBuffer, &bufferBuilder))
    {
        NN_ASSERT(0, "Failed to initialize buffer for counter tracker");
    }

        /* Record the counter reporting command buffer. */
    nvnCommandBufferBeginRecording(&m_ReportCommandBuffer);
    {
        unsigned end = NVN_COUNTER_TYPE_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN;
        unsigned commandOffset = 0;
        unsigned flagChecker = CounterFlags_Timestamp;
        NVNbufferAddress address = nvnBufferGetAddress(&m_CounterBuffer);

            /* Check for each potential counter and report it if needed. */
        for (unsigned i = 0; i <= end; ++i)
        {
            if (m_CounterFlags & flagChecker)
            {
                nvnCommandBufferReportCounter(&m_ReportCommandBuffer, (NVNcounterType)i, address + commandOffset);
            }

            flagChecker <<= 1;
            commandOffset += m_CounterAlignment;
        }
    }
    m_ReportHandle = nvnCommandBufferEndRecording(&m_ReportCommandBuffer);

        /* Record the counter reset command buffer. */
    nvnCommandBufferBeginRecording(&m_ResetCommandBuffer);
    {
        unsigned end = NVN_COUNTER_TYPE_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN;
        unsigned flagChecker = CounterFlags_Samples_Passed;

            /* Check for each potential counter and reset it if needed. */
        for (unsigned i = 1; i <= end; ++i)
        {
            if (m_CounterFlags & flagChecker)
            {
                nvnCommandBufferResetCounter(&m_ResetCommandBuffer, (NVNcounterType)i);
            }

            flagChecker <<= 1;
        }
    }
    m_ResetHandle = nvnCommandBufferEndRecording(&m_ResetCommandBuffer);
}

/*
 * StatTracker::GetCounters
 * ------------------------
 * Waits for the counters to finish being
 * reported and stores the returned values
 * in a struct.
 */
void StatTracker::GetCounters(StatCounters* pOutCounters, NVNqueue* pQueue)
{
        /* Wait for the counters to be reported. */
    nvnQueueFenceSync(pQueue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, NVN_SYNC_FLAG_FLUSH_FOR_CPU_BIT);
    nvnQueueFlush(pQueue);
    NVNsyncWaitResult waitRes = nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);
    NN_ASSERT(waitRes != NVNsyncWaitResult::NVN_SYNC_WAIT_RESULT_FAILED || waitRes != NVNsyncWaitResult::NVN_SYNC_WAIT_RESULT_FAILED,
        "Failed to wait on StatTracker sync\n");

    unsigned end = NVN_COUNTER_TYPE_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN;
    uint64_t* counterArray = (uint64_t*)pOutCounters;

    NVNcounterData* mappedBuffer = (NVNcounterData*)nvnBufferMap(&m_CounterBuffer);

        /* Grab the counter first. */
    NVNcounterData* counterData = mappedBuffer;
    counterArray[0] = counterData->timestamp;

        /* Write the rest of the counters into the output struct. */
    for(unsigned i = 1; i <= end; ++i)
    {
        counterData = mappedBuffer + i;
        counterArray[i] = counterData->counter;
    }
}

/*
 * StatTracker::ReportCounters
 * ---------------------------
 * Submits the previously recorded command buffer
 * to query for the current counter values.
 */
void StatTracker::ReportCounters(NVNqueue* pQueue)
{
    nvnQueueSubmitCommands(pQueue, 1, &m_ReportHandle);
}

/*
 * StatTracker::ResetCounters
 * --------------------------
 * Submit the previously recorded command buffer
 * to reset the counter values.
 */
void StatTracker::ResetCounters(NVNqueue* pQueue)
{
    nvnQueueSubmitCommands(pQueue, 1, &m_ResetHandle);
}

/*
 * StatTracker::PrintCounters
 * --------------------------
 * Print out the previously queried counters on
 * the screen using the provided debug text renderer.
 */
void StatTracker::PrintCounters(StatCounters* counters, DebugTextRenderer* textRenderer, float column, float startingLine)
{
    unsigned flagChecker = CounterFlags_Timestamp;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "Timestamp:                %llu", counters->m_Timestamp);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "SamplesPassed:            %llu", counters->m_SamplesPassed);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "Input Vertices:           %llu", counters->m_InputVertices);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "InputPrimitives:          %llu", counters->m_InputPrimitives);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "VertexShader Invocations: %llu", counters->m_VertexShaderInvocations);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "TesCntrlShdr Invocations: %llu", counters->m_TessControlShaderInvocations);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "TesEvalShdr Invocations:  %llu", counters->m_TessEvalShaderInvocations);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "GeomShdr Invocations:     %llu", counters->m_GeometryShaderInvocations);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "FragShader Invocations:   %llu", counters->m_FragmentShaderInvocations);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "TesEvalShdr Primitives :  %llu", counters->m_TessEvaluationShaderPrimitives);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "GeomShdr Primitives:      %llu", counters->m_GeometryShaderPrimitives);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "ClipperInput Primitives:  %llu", counters->m_ClipperInputPrimitives);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "ClipperOutput Primitives: %llu", counters->m_ClipperOutputPrimitives);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "Primitives Generated:     %llu", counters->m_PrimitivesGenerated);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "TrnsfFeedback Prims Wrtn: %llu", counters->m_TransformFeedbackPrimitivesWritten);
    }
}

/*
 * StatTracker::Shutdown
 * ---------------------
 * Clean up allocated memory and NVN data.
 */
void StatTracker::Shutdown()
{
    nvnBufferFinalize(&m_CounterBuffer);

    nvnSyncFinalize(&m_CommandBufferSync);

    if (m_pResetControlMemoryPool)
    {
        AlignedDeallocate(m_pResetControlMemoryPool);
        m_pResetControlMemoryPool = NULL;
    }

    if (m_pReportControlMemoryPool)
    {
        AlignedDeallocate(m_pReportControlMemoryPool);
        m_pReportControlMemoryPool = NULL;
    }

    nvnCommandBufferFinalize(&m_ReportCommandBuffer);
    nvnCommandBufferFinalize(&m_ResetCommandBuffer);

    m_MemoryPool.Shutdown();
}

/*
 * StatTracker Destructor
 * ----------------------
 * Empty destructor.
 */
StatTracker::~StatTracker()
{
}
