﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include <nn/sf/sf_Types.h>

// 実装
#include <nn/nn_Common.h>
#include <type_traits>
#include <utility>
#include <nn/sf/impl/detail/sf_ServiceObjectImpl.h>
#include <nn/sf/detail/sf_AutogenInterfaceIncludes.h>
#include <nn/sf/impl/sf_AllocationPolicies.h>
#include <nn/sf/sf_ObjectImplFactory.h>

#define MY_EXPECT_EQ(a, b) EXPECT_TRUE((a) == (b))

class IFoo : public nn::sf::IServiceObject
{
private:

    virtual int sync_GetInt() NN_NOEXCEPT = 0;

public:

    int GetInt() NN_NOEXCEPT
    {
        return this->sync_GetInt();
    }

};

namespace {

int64_t g_AllocationBuffer[1024 * 1024];
int64_t* g_AllocationHead = g_AllocationBuffer;
void* AllocateForTest(size_t size)
{
    auto n = (size - 1) / sizeof(int64_t) + 1;
    auto ret = g_AllocationHead;
    g_AllocationHead += n;
    return ret;
}
void DeallocateForTest(void* p, size_t size)
{
    NN_UNUSED(p);
    NN_UNUSED(size);
}

template <class Impl>
class FooTemplateObjectBase : public Impl, public IFoo
{
private:

    virtual int sync_GetInt() NN_NOEXCEPT NN_OVERRIDE
    {
        return Impl::GetInt();
    }

protected:

    using Impl::Impl;

};

template <typename Impl, class AllocationPolicy>
class FooTemplate
    : public nn::sf::ObjectImplFactory<FooTemplateObjectBase<Impl>, AllocationPolicy>
{
};

struct StatelessAllocator
{
    static void* Allocate(size_t size)
    {
        void* ret = AllocateForTest(size);
        return ret;
    }
    static void Deallocate(void* p, size_t size)
    {
        DeallocateForTest(p, size);
    }
};

int g_ConstructCount = 0;
int g_DestructCount = 0;

class Foo : public FooTemplate<Foo,  nn::sf::impl::StatelessAllocationPolicy<StatelessAllocator>>
{
private:

    int m_N;

public:

    explicit Foo(int n)
        : m_N(n)
    {
        ++g_ConstructCount;
    }

    virtual ~Foo() NN_NOEXCEPT
    {
        ++g_DestructCount;
    };

    static nn::sf::SharedPointer<IFoo> CreateShared(int n = 0)
    {
        return nn::sf::SharedPointer<IFoo>(FooTemplate::Create(n), false);
    }

    int GetInt() NN_NOEXCEPT
    {
        return m_N;
    }

};

TEST(sf, ServiceObject_Basic)
{
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        Foo::CreateShared();
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            auto p = Foo::CreateShared();
            EXPECT_TRUE(p);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            const auto& p = Foo::CreateShared();
            EXPECT_TRUE(p);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            auto&& p = Foo::CreateShared();
            EXPECT_TRUE(p);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            auto p1 = Foo::CreateShared();
            auto p2 = p1;
            EXPECT_TRUE(p2);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        decltype(Foo::CreateShared()) p1;
        EXPECT_TRUE(!p1);
        {
            auto p2 = Foo::CreateShared();
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            p1 = p2;
            EXPECT_TRUE(p1);
            EXPECT_TRUE(p2);
        }
        EXPECT_TRUE(p1);
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(0, g_DestructCount);
    }

    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        decltype(Foo::CreateShared()) p1;
        {
            auto p2 = Foo::CreateShared();
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            p1 = std::move(p2);
            EXPECT_TRUE(p1);
            EXPECT_TRUE(!p2);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(0, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        auto f = [] ()
        {
            return Foo::CreateShared();
        };
        f();
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        auto f = [] ()
        {
            return Foo::CreateShared();
        };
        auto p = f();
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(0, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        auto f = [] ()
        {
            return Foo::CreateShared();
        };
        const auto& p = f();
        EXPECT_TRUE(p);
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(0, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        auto f = [] ()
        {
            return Foo::CreateShared();
        };
        auto&& p = f();
        EXPECT_TRUE(p);
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(0, g_DestructCount);
    }
} // NOLINT(readability/fn_size)

TEST(sf, ServiceObject_AssignNullptr)
{
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            auto p = Foo::CreateShared();
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            EXPECT_TRUE(p);
            p = 0;
            EXPECT_TRUE(!p);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(1, g_DestructCount);
        }
    }
}

TEST(sf, ServiceObject_AssignSelf)
{
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            auto p = Foo::CreateShared();
            p = p;
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            EXPECT_TRUE(p);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            auto p = Foo::CreateShared();
            p = std::move(p);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            EXPECT_TRUE(p);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
}

TEST(sf, ServiceObject_Detach)
{
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            auto p = Foo::CreateShared();
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            EXPECT_TRUE(p);
            auto raw = p.Detach();
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            EXPECT_TRUE(!p);
            raw->Release();
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(1, g_DestructCount);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            auto p = Foo::CreateShared();
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            EXPECT_TRUE(p);
            auto q = decltype(p)(p.Detach(), false);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            EXPECT_TRUE(!p);
            EXPECT_TRUE(q);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
}

TEST(sf, ServiceObject_AssignReset)
{
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            auto p = Foo::CreateShared();
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            EXPECT_TRUE(p);
            p.Reset();
            EXPECT_TRUE(!p);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(1, g_DestructCount);
        }
    }
}

TEST(sf, ServiceObject_Swap)
{
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        decltype(Foo::CreateShared()) p1;
        EXPECT_TRUE(!p1);
        {
            auto p2 = Foo::CreateShared();
            EXPECT_TRUE(p2);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            p1.swap(p2);
            EXPECT_TRUE(p1);
            EXPECT_TRUE(!p2);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(0, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        auto p1 = Foo::CreateShared();
        EXPECT_TRUE(p1);
        {
            decltype(Foo::CreateShared()) p2;
            EXPECT_TRUE(!p2);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            p1.swap(p2);
            EXPECT_TRUE(!p1);
            EXPECT_TRUE(p2);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        auto p1 = Foo::CreateShared(1);
        auto p2 = Foo::CreateShared(2);
        EXPECT_TRUE(p1);
        EXPECT_TRUE(p2);
        {
            MY_EXPECT_EQ(2, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
            MY_EXPECT_EQ(1, p1->GetInt());
            MY_EXPECT_EQ(2, p2->GetInt());
            p1.swap(p2);
            EXPECT_TRUE(p1);
            EXPECT_TRUE(p2);
            MY_EXPECT_EQ(2, p1->GetInt());
            MY_EXPECT_EQ(1, p2->GetInt());
            MY_EXPECT_EQ(2, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        EXPECT_TRUE(p1);
        EXPECT_TRUE(p2);
        MY_EXPECT_EQ(2, g_ConstructCount);
        MY_EXPECT_EQ(0, g_DestructCount);
    }
}

TEST(sf, ServiceObject_Polymorphic)
{
    typedef nn::sf::SharedPointer<nn::sf::IServiceObject> ServiceObjectPointer;
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            ServiceObjectPointer p = Foo::CreateShared();
            EXPECT_TRUE(p);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            auto p1 = Foo::CreateShared();
            ServiceObjectPointer p2 = p1;
            EXPECT_TRUE(p1);
            EXPECT_TRUE(p2);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        {
            auto p1 = Foo::CreateShared();
            ServiceObjectPointer p2 = std::move(p1);
            EXPECT_TRUE(!p1);
            EXPECT_TRUE(p2);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(1, g_DestructCount);
    }
    {
        g_ConstructCount = 0;
        g_DestructCount = 0;
        ServiceObjectPointer p1;
        {
            auto p2 = Foo::CreateShared();
            p1 = std::move(p2);
            EXPECT_TRUE(p1);
            EXPECT_TRUE(!p2);
            MY_EXPECT_EQ(1, g_ConstructCount);
            MY_EXPECT_EQ(0, g_DestructCount);
        }
        EXPECT_TRUE(p1);
        MY_EXPECT_EQ(1, g_ConstructCount);
        MY_EXPECT_EQ(0, g_DestructCount);
    }
}

} // anonymous namespace
