﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/sf/sf_Types.h>
#include <nn/result/result_HandlingUtility.h>
#include "testSf_IBlockingMap.h"

namespace nnt { namespace testsf {

void RunContextControlClients(const char* serviceName) NN_NOEXCEPT;
nn::sf::SharedPointer<nnt::testsf::IBlockingMap> MakeBlockingMapViaHipc(const char* serviceName) NN_NOEXCEPT;

class BlockingMapImplBase
{
private:

    struct Entry
    {
        bool valid;
        int32_t key;
        int32_t value;
    };

    Entry m_Entries[32];

    Entry* GetEntryByKey(int32_t key) NN_NOEXCEPT
    {
        for (auto&& e : m_Entries)
        {
            if (e.valid && e.key == key)
            {
                return &e;
            }
        }
        return nullptr;
    }

    Entry* GetInvalidEntry() NN_NOEXCEPT
    {
        for (auto&& e : m_Entries)
        {
            if (!e.valid)
            {
                return &e;
            }
        }
        return nullptr;
    }

    virtual void OnSetKeyValue() NN_NOEXCEPT
    {
        // nop
    }

    virtual nn::Result HandleGetValueFailed() NN_NOEXCEPT
    {
        NN_ABORT("");
    }

protected:

    BlockingMapImplBase() NN_NOEXCEPT
    {
        for (auto&& e : m_Entries)
        {
            e.valid = false;
        }
    }

    ~BlockingMapImplBase() NN_NOEXCEPT
    {
    }

public:

    nn::Result SetKeyValue(std::int32_t key, std::int32_t value) NN_NOEXCEPT
    {
        auto p = GetEntryByKey(key);
        if (!p)
        {
            p = GetInvalidEntry();
            NN_ABORT_UNLESS_NOT_NULL(p);
            p->valid = true;
            p->key = key;
        }
        p->value = value;
        OnSetKeyValue();
        NN_RESULT_SUCCESS;
    }

    nn::Result TryGetValue(nn::sf::Out<bool> pOutValid, nn::sf::Out<std::int32_t> pOutValue, std::int32_t key) NN_NOEXCEPT
    {
        auto p = GetEntryByKey(key);
        if (!p)
        {
            *pOutValid = false;
            *pOutValue = false;
            NN_RESULT_SUCCESS;
        }
        else
        {
            *pOutValid = true;
            *pOutValue = p->value;
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result GetValue(nn::sf::Out<std::int32_t> pOut, std::int32_t key) NN_NOEXCEPT
    {
        auto p = GetEntryByKey(key);
        if (!p)
        {
            NN_RESULT_DO(HandleGetValueFailed());
        }
        *pOut = p->value;
        NN_RESULT_SUCCESS;
    }

    uint32_t TestContext(uint32_t) NN_NOEXCEPT
    {
        return 0;
    }

};

}}
