﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include "testSf_IBlockingMap.h"
#include "testSf_ContextControlCommon.h"

// hipc
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ContextControl.h>
#include <nn/sf/sf_HipcServerSessionManagerHandler.h>
#include <nn/sf/cmif/sf_InlineContext.h>

// 実装
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <type_traits>
#include <new>
#include <nnt/sfutil/sfutil_ThreadUtils.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <thread>

namespace {

struct MyAllocatorTag
{
};

typedef nn::sf::ExpHeapStaticAllocator<1024 * 1024, MyAllocatorTag> MyAllocator;

class MyAllocatorInitializer
{
public:
    MyAllocatorInitializer() NN_NOEXCEPT
    {
        MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} g_MyAllocatorInitializer;

class MyHipcServerManager;

struct RequestInfo
{
    nn::os::MultiWaitHolderType* pSession;
    MyHipcServerManager* pServerManager;

    void Resume() NN_NOEXCEPT;
};

class SuspendedRequestManager
{
private:

    struct RequestEntry
    {
        bool valid;
        RequestInfo info;
    };

    RequestEntry m_Entries[32];
    RequestEntry* m_CurrentResumingEntry;

public:

    SuspendedRequestManager() NN_NOEXCEPT
        : m_CurrentResumingEntry(nullptr)
    {
        for (auto&& e : m_Entries)
        {
            e.valid = false;
        }
    }

    void SuspendRequest(const RequestInfo& info) NN_NOEXCEPT
    {
        if (m_CurrentResumingEntry)
        {
            // Resume から再度 Suspend された場合にはフラグを上げるのみ
            m_CurrentResumingEntry->valid = true;
            return;
        }
        else
        {
            for (auto&& e : m_Entries)
            {
                if (!e.valid)
                {
                    e.valid = true;
                    e.info = info;
                    return;
                }
            }
            NN_ABORT("");
        }
    }

    void ResumeAll() NN_NOEXCEPT
    {
        for (auto&& e : m_Entries)
        {
            if (e.valid)
            {
                e.valid = false;
                m_CurrentResumingEntry = &e;
                e.info.Resume();
                m_CurrentResumingEntry = nullptr;
            }
        }
    }

};

class MyBlockingMapImpl final
    : public nnt::testsf::BlockingMapImplBase
{
private:

    virtual nn::Result HandleGetValueFailed() NN_NOEXCEPT NN_OVERRIDE
    {
        // 値が見つからなかったときは、後回しにすることを通知する
        NN_RESULT_THROW(nn::sf::DeferProcess());
    }

public:

    uint32_t TestContext(uint32_t n) NN_NOEXCEPT;

};

struct MyHipcServerManagerOption
{
    static const bool CanDeferInvokeRequest = true;
};

class MyHipcServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<100, 1, MyHipcServerManagerOption>
{
private:

    SuspendedRequestManager m_SuspendedRequestManager;

public:

    nn::Result ProcessMyInvokeRequest(nn::os::MultiWaitHolderType* p, bool resuming) NN_NOEXCEPT
    {
        NN_RESULT_TRY(ProcessInvokeRequestWithDefer(p))
            NN_RESULT_CATCH(nn::sf::ResultProcessDeferred)
            {
                RequestInfo info = {};
                info.pServerManager = this;
                info.pSession = p;
                m_SuspendedRequestManager.SuspendRequest(info);
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
        if (!resuming)
        {
            m_SuspendedRequestManager.ResumeAll();
        }
        NN_RESULT_SUCCESS;
    }

    static void Loop(MyHipcServerManager* pManager) NN_NOEXCEPT;

};

void RequestInfo::Resume() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(pServerManager->ProcessMyInvokeRequest(pSession, true));
}

namespace {

volatile uint32_t g_GlobalState = 0;

}

class ContextTestServerSessionManagerHandler
    : public nn::sf::IHipcServerSessionManagerHandler
{
    virtual nn::Result BeforeInvoke() NN_NOEXCEPT
    {
        auto inlineContext = nn::sf::cmif::GetInlineContext();
        if (g_GlobalState == 99 && inlineContext == 100)
        {
            return nn::sf::DeferProcess();
        }
        NN_RESULT_SUCCESS;
    }

    virtual void AfterInvoke() NN_NOEXCEPT
    {
        auto inlineContext = nn::sf::cmif::GetInlineContext();
        if (inlineContext == 101)
        {
            g_GlobalState = 200;
        }
    }
};

uint32_t MyBlockingMapImpl::TestContext(uint32_t n) NN_NOEXCEPT
{
    auto ret = g_GlobalState;
    g_GlobalState = n;
    return ret;
}

void MyHipcServerManager::Loop(MyHipcServerManager* pManager) NN_NOEXCEPT
{
    while (auto p = pManager->Wait())
    {
        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
            case MyHipcServerManager::InvokeTag:
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->ProcessMyInvokeRequest(p, false));
                continue;
            }
            case MyHipcServerManager::AcceptTag:
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->ProcessAcceptRequest(p));
                continue;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

NN_OS_ALIGNAS_THREAD_STACK char g_ServerStack[nn::os::StackRegionAlignment * 4];
nnt::sfutil::TestThread g_ServerThread;
const char ServiceName[] = "sf_test1";
std::aligned_storage<sizeof(MyHipcServerManager), NN_ALIGNOF(MyHipcServerManager)>::type g_HipcServerManagerStorage;

void TestServerSessionManagerHandler() NN_NOEXCEPT
{
    auto p = nnt::testsf::MakeBlockingMapViaHipc(ServiceName);
    /*
      p->TestContext(x) の呼び出しは以下の処理を行う。
        - g_GlobalState == 99 && GetInlineContext() == 100 のとき defer する
        - g_GlobalState に x を代入する
        - g_GlobalState の元の値を返す
        - GetInlineContext() == 101 だったら g_GlobalState に 200 を代入する
    */
    {
        {
            nn::sf::cmif::ScopedInlineContextChanger sicc{10};
            auto n = p->TestContext(90);
            ASSERT_EQ(0u, n);
        }
        ASSERT_EQ(90u, g_GlobalState);
        {
            nn::sf::cmif::ScopedInlineContextChanger sicc{10};
            auto n = p->TestContext(99);
            ASSERT_EQ(90u, n);
        }
        ASSERT_EQ(99u, g_GlobalState);
        std::thread blockedThread([]
        {
            auto q = nnt::testsf::MakeBlockingMapViaHipc(ServiceName);
            nn::sf::cmif::ScopedInlineContextChanger sicc{100};
            // この呼び出しは一旦 defer され、その後別スレッドでの p->TestContext(30) によって Resume されて返る
            auto n = q->TestContext(10);
            // 実処理は p->TestContext(30) のあとに行われるため 30 が返る
            ASSERT_EQ(30u, n);
        });
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        {
            nn::sf::cmif::ScopedInlineContextChanger sicc{20};
            auto n = p->TestContext(30);
            ASSERT_EQ(99u, n);
        }
        blockedThread.join();
        ASSERT_EQ(10u, g_GlobalState);
        {
            nn::sf::cmif::ScopedInlineContextChanger sicc{101};
            auto n = p->TestContext(20);
            ASSERT_EQ(10u, n);
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        ASSERT_EQ(200u, g_GlobalState);
    }
}

TEST(sf_AllInOne, ContextControl2)
{
    nn::sf::InitializeForHipcServerSessionManagerHandler();
    nn::sf::UnmanagedServiceObject<nnt::testsf::IBlockingMap, MyBlockingMapImpl> blockingMapImpl;
    auto pServer = new (&g_HipcServerManagerStorage) MyHipcServerManager;
    ContextTestServerSessionManagerHandler handler;
    pServer->SetManagerHandler(&handler);
    NN_UTIL_SCOPE_EXIT
    {
        pServer->~MyHipcServerManager();
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(pServer->RegisterObjectForPort(blockingMapImpl.GetShared(), 10, ServiceName));
    pServer->Start();
    g_ServerThread.Initialize(MyHipcServerManager::Loop, pServer, g_ServerStack, sizeof(g_ServerStack), 10);
    g_ServerThread.Start();

    TestServerSessionManagerHandler();
    nnt::testsf::RunContextControlClients(ServiceName);

    pServer->RequestStop();
    g_ServerThread.Finalize();
}

}
