﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include "testSf_IBlockingMap.h"
#include "testSf_ContextControlCommon.h"

// hipc
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ContextControl.h>

// 実装
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <type_traits>
#include <new>
#include <nnt/sfutil/sfutil_ThreadUtils.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace {

struct MyAllocatorTag
{
};

typedef nn::sf::ExpHeapStaticAllocator<1024 * 1024, MyAllocatorTag> MyAllocator;

class MyAllocatorInitializer
{
public:
    MyAllocatorInitializer() NN_NOEXCEPT
    {
        MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} g_MyAllocatorInitializer;

class MyHipcServerManager;

struct RequestInfo
{
    nn::os::MultiWaitHolderType* pSession;
    MyHipcServerManager* pServerManager;

    static RequestInfo GetCurrent() NN_NOEXCEPT;
    void Resume() NN_NOEXCEPT;
};

class SuspendedRequestManager
{
private:

    struct RequestEntry
    {
        bool valid;
        RequestInfo info;
    };

    RequestEntry m_Entries[32];
    RequestEntry* m_CurrentResumingEntry;

public:

    SuspendedRequestManager() NN_NOEXCEPT
        : m_CurrentResumingEntry(nullptr)
    {
        for (auto&& e : m_Entries)
        {
            e.valid = false;
        }
    }

    void SuspendCurrentRequest() NN_NOEXCEPT
    {
        if (m_CurrentResumingEntry)
        {
            // Resume から再度 Suspend された場合にはフラグを上げるのみ
            m_CurrentResumingEntry->valid = true;
            return;
        }
        else
        {
            for (auto&& e : m_Entries)
            {
                if (!e.valid)
                {
                    e.valid = true;
                    e.info = RequestInfo::GetCurrent();
                    return;
                }
            }
            NN_ABORT("");
        }
    }

    void ResumeAll() NN_NOEXCEPT
    {
        for (auto&& e : m_Entries)
        {
            if (e.valid)
            {
                e.valid = false;
                m_CurrentResumingEntry = &e;
                e.info.Resume();
                m_CurrentResumingEntry = nullptr;
            }
        }
    }

};

class MyBlockingMapImpl final
    : public nnt::testsf::BlockingMapImplBase
{
private:

    typedef nnt::testsf::BlockingMapImplBase Base;

    virtual void OnSetKeyValue() NN_NOEXCEPT NN_OVERRIDE;

    virtual nn::Result HandleGetValueFailed() NN_NOEXCEPT NN_OVERRIDE
    {
        // 値が見つからなかったときは、リクエストをサスペンドし、現コンテキストを破棄する
        SuspendCurrentRequest();
        NN_RESULT_THROW(nn::sf::InvalidateCurrentContext());
    }

    void SuspendCurrentRequest() NN_NOEXCEPT;

};

struct MyHipcServerManagerOption
{
    static const bool CanDeferInvokeRequest = true;
};

class MyHipcServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<100, 1, MyHipcServerManagerOption>
{
private:

    void SetCurrentRequestInfo(nn::os::MultiWaitHolderType* p) NN_NOEXCEPT;

public:

    static void Loop(MyHipcServerManager* pManager) NN_NOEXCEPT
    {
        while (auto p = pManager->Wait())
        {
            switch (nn::os::GetMultiWaitHolderUserData(p))
            {
                case MyHipcServerManager::InvokeTag:
                {
                    pManager->SetCurrentRequestInfo(p);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->ProcessInvokeRequestWithDefer(p));
                    continue;
                }
                case MyHipcServerManager::AcceptTag:
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->ProcessAcceptRequest(p));
                    continue;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }

};

RequestInfo g_CurrentRequestInfo;
SuspendedRequestManager g_SuspendedRequestManager;

void MyHipcServerManager::SetCurrentRequestInfo(nn::os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    g_CurrentRequestInfo.pServerManager = this;
    g_CurrentRequestInfo.pSession = p;
}

RequestInfo RequestInfo::GetCurrent() NN_NOEXCEPT
{
    return g_CurrentRequestInfo;
}

void RequestInfo::Resume() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(pServerManager->ProcessInvokeRequestWithDefer(pSession));
}

void MyBlockingMapImpl::OnSetKeyValue() NN_NOEXCEPT
{
    g_SuspendedRequestManager.ResumeAll();
}

void MyBlockingMapImpl::SuspendCurrentRequest() NN_NOEXCEPT
{
    g_SuspendedRequestManager.SuspendCurrentRequest();
}

NN_OS_ALIGNAS_THREAD_STACK char g_ServerStack[nn::os::StackRegionAlignment * 4];
nnt::sfutil::TestThread g_ServerThread;
const char ServiceName[] = "sf_test1";
std::aligned_storage<sizeof(MyHipcServerManager), NN_ALIGNOF(MyHipcServerManager)>::type g_HipcServerManagerStorage;

TEST(sf_AllInOne, ContextControl)
{
    nn::sf::UnmanagedServiceObject<nnt::testsf::IBlockingMap, MyBlockingMapImpl> blockingMapImpl;
    auto pServer = new (&g_HipcServerManagerStorage) MyHipcServerManager;
    NN_UTIL_SCOPE_EXIT
    {
        pServer->~MyHipcServerManager();
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(pServer->RegisterObjectForPort(blockingMapImpl.GetShared(), 10, ServiceName));
    pServer->Start();
    g_ServerThread.Initialize(MyHipcServerManager::Loop, pServer, g_ServerStack, sizeof(g_ServerStack), 10);
    g_ServerThread.Start();

    nnt::testsf::RunContextControlClients(ServiceName);

    pServer->RequestStop();
    g_ServerThread.Finalize();
}

}
