﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/settings/settings_Language.h>
#include <nn/settings/fwdbg/settings_SettingsManagement.h>
#include <nn/settings/system/settings_Audio.h>
#include <nn/settings/system/settings_ErrorReport.h>
#include <nn/settings/system/settings_Hid.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/system/settings_Ldn.h>
#include <nn/settings/system/settings_Network.h>
#include <nnt/nntest.h>

//!< システム設定の読み出しが正常に動作するか
TEST(ZeroInitializationSuite, ReadSettingsTest1)
{
    const size_t bufferSize = 512 * 1024;

    auto buffer1 = new ::nn::Bit8[bufferSize * 2]();
    auto buffer2 = new ::nn::Bit8[bufferSize]();

    // システム設定をリセット
    ::nn::settings::fwdbg::ResetSettings(
        ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem);

    // システム設定の初期値を保存
    EXPECT_EQ(
        bufferSize,
        ::nn::settings::fwdbg::ReadSettings(
            buffer1, bufferSize * 2,
            ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem));

    // システム設定の書き換え
    ::nn::settings::system::SetForceMuteOnHeadphoneRemoved(false);
    ::nn::settings::system::SetErrorReportSharePermission(
        ::nn::settings::system::ErrorReportSharePermission_Granted);
    ::nn::settings::system::SetVibrationMasterVolume(0.1f);
    ::nn::settings::system::SetLanguageCode(
        ::nn::settings::LanguageCode::Make(
            ::nn::settings::Language_LatinAmericanSpanish));
    ::nn::settings::system::SetLdnChannel(2);
    ::nn::settings::system::SetNetworkSettings(nullptr, 0);

    // システム設定をリセット
    ::nn::settings::fwdbg::ResetSettings(
        ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem);

    // システム設定のリセット後の値を保存
    EXPECT_EQ(
        bufferSize,
        ::nn::settings::fwdbg::ReadSettings(
            buffer2, bufferSize,
            ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem));

    // 値が一致していることを確認
    for (size_t i = 0; i < 64; ++i)
    {
        SCOPED_TRACE(
            ::testing::Message() << "LanguageSettingsDatabase[" << i << "]");
        EXPECT_EQ(buffer1[16 + i], buffer2[16 + i]);
    }

    for (size_t i = 0; i < 64; ++i)
    {
        SCOPED_TRACE(
            ::testing::Message() << "LdnSettingsDatabase[" << i << "]");
        EXPECT_EQ(buffer1[38000 + i], buffer2[38000 + i]);
    }

    for (size_t i = 0; i < 64; ++i)
    {
        SCOPED_TRACE(
            ::testing::Message() << "ErrorReportSettingsDatabase[" << i << "]");
        EXPECT_EQ(buffer1[39280 + i], buffer2[39280 + i]);
    }

    for (size_t i = 0; i < 128; ++i)
    {
        SCOPED_TRACE(
            ::testing::Message() << "AudioSettingsDatabase[" << i << "]");
        EXPECT_EQ(buffer1[169712 + i], buffer2[169712 + i]);
    }

    for (size_t i = 0; i < 64; ++i)
    {
        SCOPED_TRACE(
            ::testing::Message() << "HidSettingsDatabase[" << i << "]");
        EXPECT_EQ(buffer1[170560 + i], buffer2[170560 + i]);
    }

    // 部分的な読み出し
    ::std::fill(buffer2, buffer2 + bufferSize, ::nn::Bit8());
    EXPECT_EQ(
        bufferSize / 2,
        ::nn::settings::fwdbg::ReadSettings(
            buffer2, bufferSize / 2,
            ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem));

    // 未使用領域がゼロのままであることを確認
    for (size_t i = bufferSize / 2; i < bufferSize; ++i)
    {
        EXPECT_EQ(0, buffer2[i]);
    }

    delete[] buffer1;
    delete[] buffer2;
}

//!< システム設定のリセットが正常に動作するか
TEST(ZeroInitializationSuite, ResetSettingsTest1)
{
    // システム設定の書き換え
    ::nn::settings::system::SetForceMuteOnHeadphoneRemoved(false);
    ::nn::settings::system::SetErrorReportSharePermission(
        ::nn::settings::system::ErrorReportSharePermission_Granted);
    ::nn::settings::system::SetVibrationMasterVolume(0.1f);
    ::nn::settings::system::SetLanguageCode(
        ::nn::settings::LanguageCode::Make(
            ::nn::settings::Language_LatinAmericanSpanish));
    ::nn::settings::system::SetLdnChannel(2);
    ::nn::settings::system::SetNetworkSettings(nullptr, 0);

    // システム設定をリセット
    ::nn::settings::fwdbg::ResetSettings(
        ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem);

    // テスト用の設定のリセットを確認
    EXPECT_TRUE(::nn::settings::system::IsForceMuteOnHeadphoneRemoved());
    EXPECT_EQ(
        ::nn::settings::system::ErrorReportSharePermission_NotConfirmed,
        ::nn::settings::system::GetErrorReportSharePermission());
    EXPECT_FLOAT_EQ(1.0f, ::nn::settings::system::GetVibrationMasterVolume());
    ::nn::settings::LanguageCode languageCode = {};
    ::nn::settings::GetLanguageCode(&languageCode);
    EXPECT_STREQ("en-US", languageCode.string);
    EXPECT_EQ(-1, ::nn::settings::system::GetLdnChannel());
    ::nn::settings::system::NetworkSettings networkSettings = {};
    ::nn::settings::system::GetNetworkSettings(&networkSettings, 1);
    EXPECT_STREQ("default-ethernet", networkSettings.name);
    EXPECT_EQ(static_cast<int32_t>(::nn::settings::system::NicType_Ethernet),
              networkSettings.nicType);
}
