﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/settings/system/settings_Audio.h>
#include <nnt/nntest.h>

namespace {

//!< テスト用本体音量設定 A
const ::nn::settings::system::AudioVolume AudioVolumeA = {
    { { 2 } },
    static_cast<int8_t>(0x5A)
};

//!< テスト用本体音量設定 B
const ::nn::settings::system::AudioVolume AudioVolumeB = {
    { { 1 } },
    static_cast<int8_t>(0x2E)
};

//!< テスト用本体オーディオ出力設定 A
const ::nn::settings::system::AudioOutputMode AudioOutputModesA[] = {
    ::nn::settings::system::AudioOutputMode_1ch,
    ::nn::settings::system::AudioOutputMode_2ch,
    ::nn::settings::system::AudioOutputMode_5_1ch,
    ::nn::settings::system::AudioOutputMode_7_1ch,
};

//!< テスト用本体オーディオ出力設定 B
const ::nn::settings::system::AudioOutputMode AudioOutputModesB[] = {
    ::nn::settings::system::AudioOutputMode_2ch,
    ::nn::settings::system::AudioOutputMode_5_1ch,
    ::nn::settings::system::AudioOutputMode_7_1ch,
    ::nn::settings::system::AudioOutputMode_5_1ch_Forced,
};

//!< ヘッドフォン音量更新用本体音量設定 A
::nn::settings::system::AudioVolume HeadphoneAudioVolumeA = {
    { { 0 } },
    static_cast<int8_t>(12)
};

//!< ヘッドフォン音量更新用本体音量設定 B
::nn::settings::system::AudioVolume HeadphoneAudioVolumeB = {
    { { 2 } },
    static_cast<int8_t>(7)
};

} // namespace

namespace nn { namespace settings { namespace system {

bool GetHeadphoneVolumeUpdateFlag() NN_NOEXCEPT;

void SetHeadphoneVolumeUpdateFlag(bool value) NN_NOEXCEPT;

}}} // namespace nn::settings::system

//!< スピーカーの本体音量設定を設定可能か
TEST(AudioSuite, AudioVolumeTest1)
{
    ::nn::settings::system::AudioVolume settings = {};

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Speaker);

    ::nn::settings::system::SetAudioVolume(
        AudioVolumeA,
        ::nn::settings::system::AudioVolumeTarget_Speaker);

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Usb);

    ::nn::settings::system::GetAudioVolume(
        &settings,
        ::nn::settings::system::AudioVolumeTarget_Speaker);

    EXPECT_EQ(
        AudioVolumeA.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>(),
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>());

    EXPECT_EQ(
        AudioVolumeA.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>(),
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>());

    EXPECT_EQ(AudioVolumeA.volume, settings.volume);
}

//!< ヘッドフォンの本体音量設定を設定可能か
TEST(AudioSuite, AudioVolumeTest2)
{
    ::nn::settings::system::AudioVolume settings = {};

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    ::nn::settings::system::SetAudioVolume(
        AudioVolumeB,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Speaker);

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Usb);

    ::nn::settings::system::GetAudioVolume(
        &settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    EXPECT_EQ(
        AudioVolumeB.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>(),
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>());

    EXPECT_EQ(
        AudioVolumeB.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>(),
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>());

    EXPECT_EQ(AudioVolumeB.volume, settings.volume);
}

//!< USB オーディオの本体音量設定を設定可能か
TEST(AudioSuite, AudioVolumeTest3)
{
    ::nn::settings::system::AudioVolume settings = {};

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Usb);

    ::nn::settings::system::SetAudioVolume(
        AudioVolumeA,
        ::nn::settings::system::AudioVolumeTarget_Usb);

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Speaker);

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    ::nn::settings::system::GetAudioVolume(
        &settings,
        ::nn::settings::system::AudioVolumeTarget_Usb);

    EXPECT_EQ(
        AudioVolumeA.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>(),
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>());

    EXPECT_EQ(
        AudioVolumeA.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>(),
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>());

    EXPECT_EQ(AudioVolumeA.volume, settings.volume);
}

//!< 本体オーディオ出力設定を設定可能か
TEST(AudioSuite, AudioOutputModeTest1)
{
    ::nn::settings::system::AudioOutputModeTarget targets[] = {
        ::nn::settings::system::AudioOutputModeTarget_Hdmi,
        ::nn::settings::system::AudioOutputModeTarget_Speaker,
        ::nn::settings::system::AudioOutputModeTarget_Headphone,
        ::nn::settings::system::AudioOutputModeTarget_Usb,
    };

    const size_t targetCount = NN_ARRAY_SIZE(targets);

    for (size_t i = 0; i < targetCount; ++i)
    {
        ::nn::settings::system::SetAudioOutputMode(
            AudioOutputModesA[i], targets[i]);
    }

    for (size_t i = 0; i < targetCount; ++i)
    {
        EXPECT_EQ(
            AudioOutputModesA[i],
            ::nn::settings::system::GetAudioOutputMode(targets[i]));
    }

    for (size_t i = 0; i < targetCount; ++i)
    {
        ::nn::settings::system::SetAudioOutputMode(
            AudioOutputModesB[i], targets[i]);
    }

    for (size_t i = 0; i < targetCount; ++i)
    {
        EXPECT_EQ(
            AudioOutputModesB[i],
            ::nn::settings::system::GetAudioOutputMode(targets[i]));
    }
}

//!< ヘッドフォン抜け時の強制ミュート設定を取得可能か
TEST(AudioSuite, ForceMuteOnHeadphoneRemovedTest1)
{
    ::nn::settings::system::IsForceMuteOnHeadphoneRemoved();
}

//!< ヘッドフォン抜け時の強制ミュート設定を設定可能か
TEST(AudioSuite, ForceMuteOnHeadphoneRemovedTest2)
{
    ::nn::settings::system::SetForceMuteOnHeadphoneRemoved(false);

    EXPECT_FALSE(::nn::settings::system::IsForceMuteOnHeadphoneRemoved());

    ::nn::settings::system::SetForceMuteOnHeadphoneRemoved(true);

    EXPECT_TRUE(::nn::settings::system::IsForceMuteOnHeadphoneRemoved());
}

//!< ヘッドフォン音量の警告表示回数を設定可能か
TEST(AudioSuite, HeadphoneVolumeWarningCountTest1)
{
    const int count =
        ::nn::settings::system::GetHeadphoneVolumeWarningDisplayedEventCount();

    const int countA = 10;

    const int countB = 100;

    ::nn::settings::system::SetHeadphoneVolumeWarningDisplayedEventCount(
        countA);

    ::nn::settings::system::SetHeadphoneVolumeWarningDisplayedEventCount(
        countB);

    EXPECT_EQ(
        countB,
        ::nn::settings::system::GetHeadphoneVolumeWarningDisplayedEventCount());

    ::nn::settings::system::SetHeadphoneVolumeWarningDisplayedEventCount(
        countA);

    EXPECT_EQ(
        countA,
        ::nn::settings::system::GetHeadphoneVolumeWarningDisplayedEventCount());

    ::nn::settings::system::SetHeadphoneVolumeWarningDisplayedEventCount(
        count);
}

//!< 本体ヘッドホン音量の更新がペアコン設定を正しく処理するか
TEST(AudioSuite, HeadphoneVolumeUpdateTest1)
{
    // ペアコン無効
    ::nn::settings::system::SetHeadphoneVolumeUpdateFlag(false);

    EXPECT_FALSE(
        ::nn::settings::system::GetHeadphoneVolumeUpdateFlag());

    ::nn::settings::system::AudioVolume settings = HeadphoneAudioVolumeA;

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    auto isHighPowerAvailable = bool();
    auto volume = int8_t();

    EXPECT_TRUE(
        ::nn::settings::system::NeedsToUpdateHeadphoneVolume(
            &isHighPowerAvailable, &volume, false));
    EXPECT_TRUE(isHighPowerAvailable);
    EXPECT_EQ(volume, HeadphoneAudioVolumeB.volume);

    settings = ::nn::settings::system::AudioVolume();
    ::nn::settings::system::GetAudioVolume(
        &settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>());
    EXPECT_TRUE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>());
    EXPECT_EQ(settings.volume, volume);

    EXPECT_TRUE(
        ::nn::settings::system::GetHeadphoneVolumeUpdateFlag());

    // ペアコン有効
    ::nn::settings::system::SetHeadphoneVolumeUpdateFlag(false);

    EXPECT_FALSE(
        ::nn::settings::system::GetHeadphoneVolumeUpdateFlag());

    settings = HeadphoneAudioVolumeA;

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    isHighPowerAvailable = bool();
    volume = int8_t();

    EXPECT_FALSE(
        ::nn::settings::system::NeedsToUpdateHeadphoneVolume(
            &isHighPowerAvailable, &volume, true));

    settings = ::nn::settings::system::AudioVolume();
    ::nn::settings::system::GetAudioVolume(
        &settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>());
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>());
    EXPECT_EQ(settings.volume, HeadphoneAudioVolumeA.volume);

    EXPECT_TRUE(
        ::nn::settings::system::GetHeadphoneVolumeUpdateFlag());
}

//!< 本体ヘッドホン音量の更新が更新済みの状態で正しく処理されるか
TEST(AudioSuite, HeadphoneVolumeUpdateTest2)
{
    // 大音量無効
    ::nn::settings::system::SetHeadphoneVolumeUpdateFlag(true);

    EXPECT_TRUE(
        ::nn::settings::system::GetHeadphoneVolumeUpdateFlag());

    ::nn::settings::system::AudioVolume settings = HeadphoneAudioVolumeA;

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    auto isHighPowerAvailable = bool();
    auto volume = int8_t();

    EXPECT_FALSE(
        ::nn::settings::system::NeedsToUpdateHeadphoneVolume(
            &isHighPowerAvailable, &volume, false));

    settings = ::nn::settings::system::AudioVolume();
    ::nn::settings::system::GetAudioVolume(
        &settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>());
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>());
    EXPECT_EQ(settings.volume, HeadphoneAudioVolumeA.volume);

    EXPECT_TRUE(
        ::nn::settings::system::GetHeadphoneVolumeUpdateFlag());

    // 大音量有効
    settings = HeadphoneAudioVolumeB;

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    isHighPowerAvailable = bool();
    volume = int8_t();

    EXPECT_FALSE(
        ::nn::settings::system::NeedsToUpdateHeadphoneVolume(
            &isHighPowerAvailable, &volume, false));

    settings = ::nn::settings::system::AudioVolume();
    ::nn::settings::system::GetAudioVolume(
        &settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>());
    EXPECT_TRUE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>());
    EXPECT_EQ(settings.volume, HeadphoneAudioVolumeB.volume);

    EXPECT_TRUE(
        ::nn::settings::system::GetHeadphoneVolumeUpdateFlag());
}

//!< 本体ヘッドホン音量が境界値付近で正しく更新されるか
TEST(AudioSuite, HeadphoneVolumeUpdateTest3)
{
    // 上限の確認
    ::nn::settings::system::SetHeadphoneVolumeUpdateFlag(false);

    EXPECT_FALSE(
        ::nn::settings::system::GetHeadphoneVolumeUpdateFlag());

    ::nn::settings::system::AudioVolume settings = HeadphoneAudioVolumeA;
    settings.volume = 15;
    settings.flags.Set<::nn::settings::system::AudioVolumeFlag::Mute>();

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    auto isHighPowerAvailable = bool();
    auto volume = int8_t();

    EXPECT_TRUE(
        ::nn::settings::system::NeedsToUpdateHeadphoneVolume(
            &isHighPowerAvailable, &volume, false));
    EXPECT_TRUE(isHighPowerAvailable);
    EXPECT_EQ(volume, 9);

    settings = ::nn::settings::system::AudioVolume();
    ::nn::settings::system::GetAudioVolume(
        &settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);
    EXPECT_TRUE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>());
    EXPECT_TRUE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>());
    EXPECT_EQ(settings.volume, volume);

    EXPECT_TRUE(
        ::nn::settings::system::GetHeadphoneVolumeUpdateFlag());

    // 下限の確認
    ::nn::settings::system::SetHeadphoneVolumeUpdateFlag(false);

    EXPECT_FALSE(
        ::nn::settings::system::GetHeadphoneVolumeUpdateFlag());

    settings = HeadphoneAudioVolumeA;
    settings.volume = 0;

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    isHighPowerAvailable = bool();
    volume = int8_t();

    EXPECT_TRUE(
        ::nn::settings::system::NeedsToUpdateHeadphoneVolume(
            &isHighPowerAvailable, &volume, false));
    EXPECT_TRUE(isHighPowerAvailable);
    EXPECT_EQ(volume, 0);

    settings = ::nn::settings::system::AudioVolume();
    ::nn::settings::system::GetAudioVolume(
        &settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::Mute>());
    EXPECT_TRUE(
        settings.flags.Test<
            ::nn::settings::system::AudioVolumeFlag::HighPower>());
    EXPECT_EQ(settings.volume, volume);

    EXPECT_TRUE(
        ::nn::settings::system::GetHeadphoneVolumeUpdateFlag());
}
