﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nnt/nntest.h>

#include "testSettings_Utility.h"

class SettingsUsageErrorSuite : public testing::Test
{
protected:
    //!< 長すぎる設定名の長さ
    static const int NameSize =
        ::nn::settings::fwdbg::SettingsNameLengthMax + 2;

    //!< 長すぎる設定項目名の長さ
    static const int KeySize =
        ::nn::settings::fwdbg::SettingsItemKeyLengthMax + 2;

    //!< 長すぎる設定名
    static char s_Name[NameSize];

    //!< 長すぎる設定項目名
    static char s_Key[KeySize];

    static void SetUpTestCase() NN_NOEXCEPT
    {
        const int nameLength = NameSize - 1;
        s_Name[nameLength] = '\0';
        for (int i = 0; i < nameLength; ++i)
        {
            s_Name[i] = 'a';
        }

        const int keyLength = KeySize - 1;
        s_Key[keyLength] = '\0';
        for (int i = 0; i < keyLength; ++i)
        {
            s_Key[i] = 'b';
        }
    }

    virtual void SetUp() NN_OVERRIDE
    {
        ::nnt::settings::fwdbg::InitializeSettings();
    }
};

char SettingsUsageErrorSuite::s_Name[SettingsUsageErrorSuite::NameSize];

char SettingsUsageErrorSuite::s_Key[SettingsUsageErrorSuite::KeySize];

//!< 設定名が NULL の際に GetSettingsItemValueSize() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueSizeNameTest1)
{
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValueSize(nullptr, "foo"), "");
}

//!< 設定名が空文字の際に GetSettingsItemValueSize() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueSizeNameTest2)
{
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValueSize("", "foo"), "");
}

//!< 設定名が存在しない際に GetSettingsItemValueSize() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueSizeNameTest3)
{
    EXPECT_EQ(0, ::nn::settings::fwdbg::GetSettingsItemValueSize("bar", "foo"));
}

//!< 設定名が不正な際に GetSettingsItemValueSize() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueSizeNameTest4)
{
    const char* const name = "!";
    const char* const key = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValueSize(name, key), "");
}

//!< 設定名が長過ぎる際に GetSettingsItemValueSize() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueSizeNameTest5)
{
    const char* const key = "foo";
    ::nnt::settings::fwdbg::CreateSettings(s_Name);
    ::nnt::settings::fwdbg::CreateSettingsItem(s_Name, key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValueSize(s_Name, key), "");
}

//!< 設定項目名が NULL の際に GetSettingsItemValueSize() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueSizeKeyTest1)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValueSize(name, nullptr), "");
}

//!< 設定項目名が空文字の際に GetSettingsItemValueSize() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueSizeKeyTest2)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValueSize(name, ""), "");
}

//!< 設定項目名が存在しない際に GetSettingsItemValueSize() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueSizeKeyTest3)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    EXPECT_EQ(0, ::nn::settings::fwdbg::GetSettingsItemValueSize(name, "bar"));
}

//!< 設定項目名が不正な際に GetSettingsItemValueSize() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueSizeKeyTest4)
{
    const char* const name = "foo";
    const char* const key = "!";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValueSize(name, key), "");
}

//!< 設定項目名が長過ぎる際に GetSettingsItemValueSize() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueSizeKeyTest5)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, s_Key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValueSize(name, s_Key), "");
}

//!< 設定名が NULL の際に GetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueNameTest1)
{
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValue(&c, 1, nullptr, "foo"), "");
}

//!< 設定名が空文字の際に GetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueNameTest2)
{
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValue(&c, 1, "", "foo"), "");
}

//!< 設定名が存在しない際に GetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueNameTest3)
{
    char c;
    EXPECT_EQ(
        0, ::nn::settings::fwdbg::GetSettingsItemValue(&c, 1, "bar", "foo"));
}

//!< 設定名が不正な際に GetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueNameTest4)
{
    const char* const name = "!";
    const char* const key = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, "a", 2);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValue(&c, 1, name, key), "");
}

//!< 設定名が長過ぎる際に GetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueNameTest5)
{
    const char* const key = "foo";
    ::nnt::settings::fwdbg::CreateSettings(s_Name);
    ::nnt::settings::fwdbg::CreateSettingsItem(s_Name, key, "a", 2);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValue(&c, 1, s_Name, key), "");
}

//!< 設定項目名が NULL の際に GetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueKeyTest1)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValue(&c, 1, name, nullptr), "");
}

//!< 設定項目名が空文字の際に GetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueKeyTest2)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValue(&c, 1, name, ""), "");
}

//!< 設定項目名が存在しない際に GetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueKeyTest3)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    char c;
    EXPECT_EQ(
        0, ::nn::settings::fwdbg::GetSettingsItemValue(&c, 1, name, "bar"));
}

//!< 設定項目名が不正な際に GetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueKeyTest4)
{
    const char* const name = "foo";
    const char* const key = "!";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, "a", 2);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValue(&c, 1, name, key), "");
}

//!< 設定項目名が長過ぎる際に GetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueKeyTest5)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, s_Key, "a", 2);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValue(&c, 1, name, s_Key), "");
}

//!< バッファが NULL な際に GetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemValueBufferTest1)
{
    const char* const name = "foo";
    const char* const key = "bar";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemValue(nullptr, 0, name, key), "");
}

//!< 設定名が NULL の際に CreateSettingsItemKeyIterator() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, CreateSettingsItemKeyIteratorNameTest1)
{
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(nullptr), "");
}

//!< 設定名が空文字の際に CreateSettingsItemKeyIterator() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, CreateSettingsItemKeyIteratorNameTest2)
{
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(""), "");
}

//!< 設定名が存在しない際に CreateSettingsItemKeyIterator() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, CreateSettingsItemKeyIteratorNameTest3)
{
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::CreateSettingsItemKeyIterator("bar"), "");
}

//!< 設定名が不正な際に CreateSettingsItemKeyIterator() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, CreateSettingsItemKeyIteratorNameTest4)
{
    const char* const name = "!";
    const char* const key = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(name), "");
}

//!< 設定名が長過ぎる際に CreateSettingsItemKeyIterator() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, CreateSettingsItemKeyIteratorNameTest5)
{
    const char* const key = "foo";
    ::nnt::settings::fwdbg::CreateSettings(s_Name);
    ::nnt::settings::fwdbg::CreateSettingsItem(s_Name, key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(s_Name), "");
}

//!< ハンドルが NULL な際に DestroySettingsItemKeyIterator() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, DestroySettingsItemKeyIteratorTest1)
{
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::DestroySettingsItemKeyIterator(nullptr), "");
}

//!< ハンドルが不正な際に DestroySettingsItemKeyIterator() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, DestroySettingsItemKeyIteratorTest2)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, "bar", "a", 2);
    ::nn::settings::fwdbg::SettingsItemKeyIterator iter =
          ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(name);
    ::nn::settings::fwdbg::DestroySettingsItemKeyIterator(&iter);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::DestroySettingsItemKeyIterator(&iter), "");
}

//!< ハンドルが NULL な際に AdvanceSettingsItemKeyIterator() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, AdvanceSettingsItemKeyIteratorTest1)
{
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::AdvanceSettingsItemKeyIterator(nullptr), "");
}

//!< ハンドルが不正な際に AdvanceSettingsItemKeyIterator() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, AdvanceSettingsItemKeyIteratorTest2)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, "bar", "a", 2);
    ::nn::settings::fwdbg::SettingsItemKeyIterator iter =
          ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(name);
    ::nn::settings::fwdbg::DestroySettingsItemKeyIterator(&iter);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::AdvanceSettingsItemKeyIterator(&iter), "");
}

//!< ハンドルが不正な際に GetSettingsItemKeySize() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemKeySizeTest1)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, "bar", "a", 2);
    ::nn::settings::fwdbg::SettingsItemKeyIterator iter =
          ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(name);
    ::nn::settings::fwdbg::DestroySettingsItemKeyIterator(&iter);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemKeySize(iter), "");
}

//!< ハンドルが不正な際に GetSettingsItemKey() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, GetSettingsItemKeyTest1)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, "bar", "a", 2);
    ::nn::settings::fwdbg::SettingsItemKeyIterator iter =
          ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(name);
    ::nn::settings::fwdbg::DestroySettingsItemKeyIterator(&iter);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::GetSettingsItemKey(&c, 1, iter), "");
}

//!< 設定名が NULL の際に SetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, SetSettingsItemValueNameTest1)
{
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::SetSettingsItemValue(nullptr, "foo", &c, 1), "");
}

//!< 設定名が空文字の際に SetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, SetSettingsItemValueNameTest2)
{
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::SetSettingsItemValue("", "foo", &c, 1), "");
}

//!< 設定名が存在しない際に SetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, SetSettingsItemValueNameTest3)
{
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::SetSettingsItemValue("bar", "foo", &c, 1), "");
}

//!< 設定名が不正な際に SetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, SetSettingsItemValueNameTest4)
{
    const char* const name = "!";
    const char* const key = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, "a", 2);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::SetSettingsItemValue(name, key, &c, 1), "");
}

//!< 設定名が長過ぎる際に SetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, SetSettingsItemValueNameTest5)
{
    const char* const key = "foo";
    ::nnt::settings::fwdbg::CreateSettings(s_Name);
    ::nnt::settings::fwdbg::CreateSettingsItem(s_Name, key, "a", 2);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::SetSettingsItemValue(s_Name, key, &c, 1), "");
}

//!< 設定項目名が NULL の際に SetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, SetSettingsItemValueKeyTest1)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::SetSettingsItemValue(name, nullptr, &c, 1), "");
}

//!< 設定項目名が空文字の際に SetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, SetSettingsItemValueKeyTest2)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::SetSettingsItemValue(name, "", &c, 1), "");
}

//!< 設定項目名が存在しない際に SetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, SetSettingsItemValueKeyTest3)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::SetSettingsItemValue(name, "bar", &c, 1), "");
}

//!< 設定項目名が不正な際に SetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, SetSettingsItemValueKeyTest4)
{
    const char* const name = "foo";
    const char* const key = "!";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, "a", 2);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::SetSettingsItemValue(name, key, &c, 1), "");
}

//!< 設定項目名が長過ぎる際に SetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, SetSettingsItemValueKeyTest5)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, s_Key, "a", 2);
    char c;
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::SetSettingsItemValue(name, s_Key, &c, 1), "");
}

//!< バッファが NULL な際に SetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, SetSettingsItemValueBufferTest1)
{
    const char* const name = "foo";
    const char* const key = "bar";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::SetSettingsItemValue(name, key, nullptr, 0), "");
}

//!< 設定名が NULL の際に ResetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, ResetSettingsItemValueNameTest1)
{
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::ResetSettingsItemValue(nullptr, "foo"), "");
}

//!< 設定名が空文字の際に ResetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, ResetSettingsItemValueNameTest2)
{
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::ResetSettingsItemValue("", "foo"), "");
}

//!< 設定名が存在しない際に ResetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, ResetSettingsItemValueNameTest3)
{
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::ResetSettingsItemValue("bar", "foo"), "");
}

//!< 設定名が不正な際に ResetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, ResetSettingsItemValueNameTest4)
{
    const char* const name = "!";
    const char* const key = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::ResetSettingsItemValue(name, key), "");
}

//!< 設定名が長過ぎる際に ResetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, ResetSettingsItemValueNameTest5)
{
    const char* const key = "foo";
    ::nnt::settings::fwdbg::CreateSettings(s_Name);
    ::nnt::settings::fwdbg::CreateSettingsItem(s_Name, key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::ResetSettingsItemValue(s_Name, key), "");
}

//!< 設定項目名が NULL の際に ResetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, ResetSettingsItemValueKeyTest1)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::ResetSettingsItemValue(name, nullptr), "");
}

//!< 設定項目名が空文字の際に ResetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, ResetSettingsItemValueKeyTest2)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::ResetSettingsItemValue(name, ""), "");
}

//!< 設定項目名が存在しない際に ResetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, ResetSettingsItemValueKeyTest3)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::ResetSettingsItemValue(name, "bar"), "");
}

//!< 設定項目名が不正な際に ResetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, ResetSettingsItemValueKeyTest4)
{
    const char* const name = "foo";
    const char* const key = "!";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::ResetSettingsItemValue(name, key), "");
}

//!< 設定項目名が長過ぎる際に ResetSettingsItemValue() が正しく動作するか
TEST_F(SettingsUsageErrorSuite, ResetSettingsItemValueKeyTest5)
{
    const char* const name = "foo";
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, s_Key, "a", 2);
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::settings::fwdbg::ResetSettingsItemValue(name, s_Key), "");
}
