﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include <nn/os.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/dd.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <nn/pcv/pcv.h>
#include "SdmmcCommon.h"

// 注意：初期プロセスの FS の sMMU 設定を無効にした環境でしか動作しません。

namespace
{
    struct ClockResetControllerRegisters
    {
        volatile uint32_t   rstSource0;         // 0x0
        volatile uint32_t   rstDevicesL0;       // 0x4
        volatile uint32_t   rstDevicesH0;       // 0x8
        volatile uint32_t   rstDevicesU0;       // 0xc
        volatile uint32_t   clkOutEnbL0;        // 0x10
        volatile uint32_t   clkOutEnbH0;        // 0x14
        volatile uint32_t   clkOutEnbU0;        // 0x18
        volatile uint8_t    dummy0[0x134];
        volatile uint32_t   clkSourceSdmmc1;    // 0x150
        volatile uint32_t   clkSourceSdmmc2;    // 0x154
        volatile uint8_t    dummy1[0xC];
        volatile uint32_t   clkSourceSdmmc4;    // 0x164
        volatile uint8_t    dummy2[0x54];
        volatile uint32_t   clkSourceSdmmc3;    // 0x1bc
        volatile uint8_t    dummy3[0xD8];
        volatile uint32_t   clkOutEnbY0;        // 0x298
        volatile uint32_t   clkEnbYSet0;        // 0x29c
        volatile uint32_t   clkEnbYClr0;        // 0x2a0
        volatile uint8_t    dummy4[0x5C];
        volatile uint32_t   rstDevLSet0;        // 0x300
        volatile uint32_t   rstDevLClr0;        // 0x304
        volatile uint8_t    dummy5[0x8];
        volatile uint32_t   rstDevUSet0;        // 0x310
        volatile uint32_t   rstDevUClr0;        // 0x314
        volatile uint8_t    dummy6[0x8];
        volatile uint32_t   clkEnbLSet0;        // 0x320
        volatile uint32_t   clkEnbLClr0;        // 0x324
        volatile uint8_t    dummy7[0x8];
        volatile uint32_t   clkEnbUSet0;        // 0x330
        volatile uint32_t   clkEnbUClr0;        // 0x334
        volatile uint8_t    dummy8[0x26C];
        volatile uint32_t   pllc4Base0;         // 0x5a4
        volatile uint32_t   pllc4Misc0;         // 0x5a8
        volatile uint8_t    dummy9[0x38];
        volatile uint32_t   pllc4Out0;          // 0x5e4
        volatile uint8_t    dummy10[0xAC];
        volatile uint32_t   clkSourceSdmmcLegacyTm0;    // 0x694
    };

    const nn::dd::PhysicalAddress ClockResetControllerRegistersPhysicalAddress = 0x60006000ull;
    const size_t ClockResetControllerRegistersSize = 0x1000;

    ClockResetControllerRegisters* s_pRegisters = nullptr;

    // 0x4 レジスタ CLK_RST_CONTROLLER_RST_DEVICES_L_0 ビット定義
    const uint32_t REG_RST_DEVICES_L_0__SDMMC4 = 0x1U << 15;
    const uint32_t REG_RST_DEVICES_L_0__SDMMC1 = 0x1U << 14;
    const uint32_t REG_RST_DEVICES_L_0__SDMMC2 = 0x1U << 9;

    // 0xc レジスタ CLK_RST_CONTROLLER_RST_DEVICES_U_0 ビット定義
    const uint32_t REG_RST_DEVICES_U_0__SDMMC3 = 0x1U << 5;

    // 0x10 レジスタ CLK_RST_CONTROLLER_CLK_OUT_ENB_L_0 ビット定義
    const uint32_t REG_CLK_OUT_ENB_L_0__SDMMC4 = 0x1U << 15;
    const uint32_t REG_CLK_OUT_ENB_L_0__SDMMC1 = 0x1U << 14;
    const uint32_t REG_CLK_OUT_ENB_L_0__SDMMC2 = 0x1U << 9;

    // 0x18 レジスタ CLK_RST_CONTROLLER_CLK_OUT_ENB_U_0 ビット定義
    const uint32_t REG_CLK_OUT_ENB_U_0__SDMMC3 = 0x1U << 5;

    // 0x29c レジスタ CLK_RST_CONTROLLER_CLK_ENB_Y_SET_0 ビット定義
    const uint32_t REG_CLK_ENB_Y_SET_0__SDMMC_LEGACY_TM = 0x1U << 1;

    enum Module
    {
        Module_Sdmmc1,
        Module_Sdmmc2,
        Module_Sdmmc3,
        Module_Sdmmc4,

        Module_Num,
    };

    bool IsResetReleased(Module module) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

        switch (module)
        {
        case Module_Sdmmc1:
            if ((s_pRegisters->rstDevicesL0 & REG_RST_DEVICES_L_0__SDMMC1) != 0)
            {
                return false;
            }
            else
            {
                return true;
            }
        case Module_Sdmmc2:
            if ((s_pRegisters->rstDevicesL0 & REG_RST_DEVICES_L_0__SDMMC2) != 0)
            {
                return false;
            }
            else
            {
                return true;
            }
        case Module_Sdmmc3:
            if ((s_pRegisters->rstDevicesU0 & REG_RST_DEVICES_U_0__SDMMC3) != 0)
            {
                return false;
            }
            else
            {
                return true;
            }
        case Module_Sdmmc4:
            if ((s_pRegisters->rstDevicesL0 & REG_RST_DEVICES_L_0__SDMMC4) != 0)
            {
                return false;
            }
            else
            {
                return true;
            }
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    bool IsClockEnabled(Module module) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

        switch (module)
        {
        case Module_Sdmmc1:
            if ((s_pRegisters->clkOutEnbL0 & REG_CLK_OUT_ENB_L_0__SDMMC1) != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        case Module_Sdmmc2:
            if ((s_pRegisters->clkOutEnbL0 & REG_CLK_OUT_ENB_L_0__SDMMC2) != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        case Module_Sdmmc3:
            if ((s_pRegisters->clkOutEnbU0 & REG_CLK_OUT_ENB_U_0__SDMMC3) != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        case Module_Sdmmc4:
            if ((s_pRegisters->clkOutEnbL0 & REG_CLK_OUT_ENB_L_0__SDMMC4) != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // レジスタログ
    #define NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(basePointer, memberVariable) NN_LOG(#memberVariable "(0x%03x) 0x%08X\n",\
        reinterpret_cast<uintptr_t>(&(basePointer->memberVariable)) - reinterpret_cast<uintptr_t>(basePointer), (basePointer->memberVariable))

    void DumpRegs()
    {
        if (s_pRegisters == nullptr)
        {
            uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(ClockResetControllerRegistersPhysicalAddress, ClockResetControllerRegistersSize);
            NN_ABORT_UNLESS(registersAddress != 0);
            s_pRegisters = reinterpret_cast<ClockResetControllerRegisters*>(registersAddress);
        }

        NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(s_pRegisters, pllc4Base0);
        NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(s_pRegisters, clkSourceSdmmcLegacyTm0);
        NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(s_pRegisters, clkOutEnbY0);
        NN_LOG("    SDMMC_LEGACY_TM %u\n", s_pRegisters->clkOutEnbY0 & REG_CLK_ENB_Y_SET_0__SDMMC_LEGACY_TM);
        NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(s_pRegisters, rstDevicesL0);
        NN_LOG("    SDMMC1 %d\n", IsResetReleased(Module_Sdmmc1));
        NN_LOG("    SDMMC2 %d\n", IsResetReleased(Module_Sdmmc2));
        NN_LOG("    SDMMC4 %d\n", IsResetReleased(Module_Sdmmc4));
        NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(s_pRegisters, rstDevicesU0);
        NN_LOG("    SDMMC3 %d\n", IsResetReleased(Module_Sdmmc3));
        NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(s_pRegisters, clkOutEnbL0);
        NN_LOG("    SDMMC1 %d\n", IsClockEnabled(Module_Sdmmc1));
        NN_LOG("    SDMMC2 %d\n", IsClockEnabled(Module_Sdmmc2));
        NN_LOG("    SDMMC4 %d\n", IsClockEnabled(Module_Sdmmc4));
        NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(s_pRegisters, clkOutEnbU0);
        NN_LOG("    SDMMC3 %d\n", IsClockEnabled(Module_Sdmmc3));
        NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(s_pRegisters, clkSourceSdmmc1);
        NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(s_pRegisters, clkSourceSdmmc2);
        NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(s_pRegisters, clkSourceSdmmc3);
        NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(s_pRegisters, clkSourceSdmmc4);
        NN_LOG("\n");
    }
}

namespace
{
    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        const nn::dd::DeviceName MmcDeviceName = nn::dd::DeviceName::DeviceName_Sdmmc4a;
        #ifdef NN_BUILD_CONFIG_HARDWARE_JETSONTK1
            const nn::dd::DeviceName SdCardDeviceName = nn::dd::DeviceName::DeviceName_Sdmmc3a;
        #else   // NN_BUILD_CONFIG_HARDWARE_JETSONTK2, NN_BUILD_CONFIG_HARDWARE_NX
            const nn::dd::DeviceName SdCardDeviceName = nn::dd::DeviceName::DeviceName_Sdmmc1a;
        #endif
    #endif

    #define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
    #define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)

    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_MmcWorkBuffer[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(nn::sdmmc::MmcWorkBufferSize)];
    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_SdCardWorkBuffer[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(nn::sdmmc::SdCardWorkBufferSize)];

    const uint32_t TestSectorIndex = 1;
    const uint32_t TestSize = 4 * 1024 * 1024;
    const uint32_t TestNumSectors = TestSize / nn::sdmmc::SectorSize;
    const size_t DataBufferSize = nn::util::align_up(TestSize, nn::os::MemoryBlockUnitSize);
    void* g_pDataBuffer;
    bool g_IsMmcInitialized = false;
    bool g_IsSdCardInitialized = false;

    void SwitchToPcvClockResetControl()
    {
        NN_LOG("Before switch to pcv\n");
        DumpRegs();

        if (g_IsSdCardInitialized)
        {
            NN_LOG("nn::sdmmc::PutSdCardToSleep\n");
            nn::sdmmc::PutSdCardToSleep(nn::sdmmc::Port_SdCard0);
        }
        if (g_IsMmcInitialized)
        {
            NN_LOG("nn::sdmmc::PutMmcToSleep\n");
            nn::sdmmc::PutMmcToSleep(nn::sdmmc::Port_Mmc0);
        }

        NN_LOG("nn::sdmmc::SwitchToPcvClockResetControl\n");
        nn::sdmmc::SwitchToPcvClockResetControl();

        if (g_IsMmcInitialized)
        {
            NN_LOG("nn::sdmmc::AwakenMmc\n");
            nn::sdmmc::AwakenMmc(nn::sdmmc::Port_Mmc0);
        }
        if (g_IsSdCardInitialized)
        {
            NN_LOG("nn::sdmmc::AwakenSdCard\n");
            nn::sdmmc::AwakenSdCard(nn::sdmmc::Port_SdCard0);
        }

        NN_LOG("After switch to pcv\n");
        DumpRegs();
    }
}

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");
}

extern "C" void nnMain()
{
    // メモリヒープから malloc で使用するメモリ領域を確保し、設定する
    uintptr_t heapPtrForMalloc;
    const size_t HeapSizeForMalloc = 256 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&heapPtrForMalloc, HeapSizeForMalloc));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(heapPtrForMalloc), HeapSizeForMalloc);

    // pcv 使用後にレジスタ直たたきが繰り返せない WAR
    nn::pcv::Initialize();
    #ifdef NN_BUILD_CONFIG_HARDWARE_JETSONTK1
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetReset(nn::pcv::Module_Sdmmc3, false));
    #else   // NN_BUILD_CONFIG_HARDWARE_JETSONTK2, NN_BUILD_CONFIG_HARDWARE_NX
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetReset(nn::pcv::Module_Sdmmc1, false));
    #endif
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetReset(nn::pcv::Module_Sdmmc4, false));
    nn::pcv::Finalize();

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    nn::sdmmc::Initialize(nn::sdmmc::Port_Mmc0);
    g_IsMmcInitialized = true;

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceAddressSpaceType mmcDas;
        SetupDeviceAddressSpace(&mmcDas, MmcDeviceName);
        nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

        nn::dd::DeviceAddressSpaceMapInfo mmcInfo;
        nn::dd::DeviceVirtualAddress mmcWorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&mmcDas, &mmcInfo,
            reinterpret_cast<uintptr_t>(g_MmcWorkBuffer), sizeof(g_MmcWorkBuffer), deviceVirtualAddressOffset);
        deviceVirtualAddressOffset = mmcWorkBufferDeviceVirtualAddress + sizeof(g_MmcWorkBuffer);
        nn::sdmmc::RegisterDeviceVirtualAddress(nn::sdmmc::Port_Mmc0, reinterpret_cast<uintptr_t>(g_MmcWorkBuffer), sizeof(g_MmcWorkBuffer), mmcWorkBufferDeviceVirtualAddress);
    #endif
    nn::sdmmc::SetMmcWorkBuffer(nn::sdmmc::Port_Mmc0, g_MmcWorkBuffer, sizeof(g_MmcWorkBuffer));

    uintptr_t dataBufferAddress;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&dataBufferAddress, DataBufferSize));

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceVirtualAddress mmcDataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&mmcDas, dataBufferAddress, DataBufferSize, deviceVirtualAddressOffset);
        nn::sdmmc::RegisterDeviceVirtualAddress(nn::sdmmc::Port_Mmc0, dataBufferAddress, DataBufferSize, mmcDataBufferDeviceVirtualAddress);
    #endif

    g_pDataBuffer = reinterpret_cast<uint8_t*>(dataBufferAddress);

    NN_LOG("nn::sdmmc::Activate MMC\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::Activate(nn::sdmmc::Port_Mmc0));
    LogBusStatus(nn::sdmmc::Port_Mmc0);

    NN_LOG("nn::sdmmc::Read MMC\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, nn::sdmmc::Port_Mmc0, TestSectorIndex, TestNumSectors));

#if 0   // MMC アクセス後の制御切り替えテスト
    SwitchToPcvClockResetControl();
#endif
    LogBusStatus(nn::sdmmc::Port_Mmc0);

    NN_LOG("nn::sdmmc::Read MMC\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, nn::sdmmc::Port_Mmc0, TestSectorIndex, TestNumSectors));

    nn::sdmmc::Initialize(nn::sdmmc::Port_SdCard0);
    g_IsSdCardInitialized = true;

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceAddressSpaceType sdCardDas;
        SetupDeviceAddressSpace(&sdCardDas, SdCardDeviceName);
        deviceVirtualAddressOffset = 0;

        nn::dd::DeviceAddressSpaceMapInfo sdCardInfo;
        nn::dd::DeviceVirtualAddress sdCardWorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&sdCardDas, &sdCardInfo,
            reinterpret_cast<uintptr_t>(g_SdCardWorkBuffer), sizeof(g_SdCardWorkBuffer), deviceVirtualAddressOffset);
        deviceVirtualAddressOffset = sdCardWorkBufferDeviceVirtualAddress + sizeof(g_SdCardWorkBuffer);
        nn::sdmmc::RegisterDeviceVirtualAddress(nn::sdmmc::Port_SdCard0, reinterpret_cast<uintptr_t>(g_SdCardWorkBuffer), sizeof(g_SdCardWorkBuffer), sdCardWorkBufferDeviceVirtualAddress);
    #endif
    nn::sdmmc::SetSdCardWorkBuffer(nn::sdmmc::Port_SdCard0, g_SdCardWorkBuffer, sizeof(g_SdCardWorkBuffer));

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceVirtualAddress sdCardDataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&sdCardDas, dataBufferAddress, DataBufferSize, deviceVirtualAddressOffset);
        nn::sdmmc::RegisterDeviceVirtualAddress(nn::sdmmc::Port_SdCard0, dataBufferAddress, DataBufferSize, sdCardDataBufferDeviceVirtualAddress);
    #endif

    NN_LOG("nn::sdmmc::Activate SdCard\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::Activate(nn::sdmmc::Port_SdCard0));
    LogBusStatus(nn::sdmmc::Port_SdCard0);

    NN_LOG("nn::sdmmc::Read SdCard\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, nn::sdmmc::Port_SdCard0, TestSectorIndex, TestNumSectors));

#if 1   // SD カードアクセス後の制御切り替えテスト
    SwitchToPcvClockResetControl();
#endif
    LogBusStatus(nn::sdmmc::Port_Mmc0);
    LogBusStatus(nn::sdmmc::Port_SdCard0);

    NN_LOG("nn::sdmmc::Read MMC\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, nn::sdmmc::Port_Mmc0, TestSectorIndex, TestNumSectors));

    NN_LOG("nn::sdmmc::Read SdCard\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, nn::sdmmc::Port_SdCard0, TestSectorIndex, TestNumSectors));

    NN_LOG("nn::sdmmc::Deactivate SdCard\n");
    nn::sdmmc::Deactivate(nn::sdmmc::Port_SdCard0);
    NN_LOG("nn::sdmmc::Deactivate MMC\n");
    nn::sdmmc::Deactivate(nn::sdmmc::Port_Mmc0);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(nn::sdmmc::Port_SdCard0, dataBufferAddress, DataBufferSize, sdCardDataBufferDeviceVirtualAddress);
        nn::sdmmc::UnregisterDeviceVirtualAddress(nn::sdmmc::Port_Mmc0, dataBufferAddress, DataBufferSize, mmcDataBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&sdCardDas, dataBufferAddress, DataBufferSize, sdCardDataBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&mmcDas, dataBufferAddress, DataBufferSize, mmcDataBufferDeviceVirtualAddress);
    #endif
    nn::os::FreeMemoryBlock(dataBufferAddress, DataBufferSize);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(nn::sdmmc::Port_SdCard0, reinterpret_cast<uintptr_t>(g_SdCardWorkBuffer), sizeof(g_SdCardWorkBuffer), sdCardWorkBufferDeviceVirtualAddress);
        nn::sdmmc::UnregisterDeviceVirtualAddress(nn::sdmmc::Port_Mmc0, reinterpret_cast<uintptr_t>(g_MmcWorkBuffer), sizeof(g_MmcWorkBuffer), mmcWorkBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&sdCardInfo);
        UnmapDeviceAddressSpaceAligned(&mmcInfo);

        CleanDeviceAddressSpace(&sdCardDas, SdCardDeviceName);
        CleanDeviceAddressSpace(&mmcDas, MmcDeviceName);
    #endif

    nn::sdmmc::Finalize(nn::sdmmc::Port_SdCard0);
    g_IsSdCardInitialized = false;
    nn::sdmmc::Finalize(nn::sdmmc::Port_Mmc0);
    g_IsMmcInitialized = false;

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());

    nn::os::FreeMemoryBlock(heapPtrForMalloc, HeapSizeForMalloc);

    NN_LOG("Done\n");
}
