﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sdmmc/sdmmc_SdCard.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <mutex>

namespace nn { namespace fs {   // 疑似 fs

    nn::os::Mutex g_SdCardDetectionEventsMutex(true);
    const int NumSdCardDetectionEvents = 4;
    nn::os::SystemEventType g_SdCardDetectionEvents[NumSdCardDetectionEvents];
    nn::os::SystemEventType* g_pClientSdCardDetectionEvents[NumSdCardDetectionEvents];

    void SdCardDetectionEventCallback(void* pParameter)
    {
        NN_UNUSED(pParameter);

        std::lock_guard<nn::os::Mutex> lock(g_SdCardDetectionEventsMutex);

        for (int i = 0; i < NumSdCardDetectionEvents; i++)
        {
            if (g_pClientSdCardDetectionEvents[i] != nullptr)
            {
                NN_LOG("[pseudo::fs] Signal(handle:0x%08X)\n", nn::os::GetReadableHandleOfSystemEvent(&g_SdCardDetectionEvents[i]));
                nn::os::SignalSystemEvent(&g_SdCardDetectionEvents[i]);
            }
        }
    }

    nn::Result BindSdCardDetectionEvent(nn::os::SystemEventType* pClientSystemEvent) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(g_SdCardDetectionEventsMutex);

        for (int i = 0; i < NumSdCardDetectionEvents; i++)
        {
            if (g_pClientSdCardDetectionEvents[i] == nullptr)
            {
                nn::Result result = nn::os::CreateSystemEvent(&g_SdCardDetectionEvents[i], nn::os::EventClearMode_ManualClear, true);
                if (result.IsFailure())
                {
                    NN_LOG("[pseudo::fs] nn::os::CreateSystemEvent is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
                    return result;
                }
                nn::os::NativeHandle readableHandle = nn::os::GetReadableHandleOfSystemEvent(&g_SdCardDetectionEvents[i]);
                nn::os::AttachReadableHandleToSystemEvent(pClientSystemEvent, readableHandle, true, nn::os::EventClearMode_ManualClear);
                g_pClientSdCardDetectionEvents[i] = pClientSystemEvent;
                return nn::ResultSuccess();
            }
        }
        return nn::fs::ResultUnexpected();  // リソース不足的な Result が適切
    }

    nn::Result UnbindSdCardDetectionEvent(nn::os::SystemEventType* pClientSystemEvent) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(g_SdCardDetectionEventsMutex);

        for (int i = 0; i < NumSdCardDetectionEvents; i++)
        {
            if (g_pClientSdCardDetectionEvents[i] == pClientSystemEvent)
            {
                g_pClientSdCardDetectionEvents[i] = nullptr;
                (void)nn::os::DetachReadableHandleOfSystemEvent(pClientSystemEvent);
                nn::os::DestroySystemEvent(&g_SdCardDetectionEvents[i]);
                return nn::ResultSuccess();
            }
        }
        return nn::fs::ResultUnexpected();  // pClientSystemEvent が Bind されていない的な Result が適切
    }

    bool IsSdCardInserted() NN_NOEXCEPT
    {
        return nn::sdmmc::IsSdCardInserted(nn::sdmmc::Port_SdCard0);
    }

    void Initialize() NN_NOEXCEPT
    {
        // クロック制御は pcv 経由で行う
        nn::sdmmc::SwitchToPcvClockResetControl();

        nn::sdmmc::Initialize(nn::sdmmc::Port_SdCard0);
        nn::sdmmc::RegisterSdCardDetectionEventCallback(nn::sdmmc::Port_SdCard0, SdCardDetectionEventCallback, nullptr);
    }

    void Finalize() NN_NOEXCEPT
    {
        nn::sdmmc::UnregisterSdCardDetectionEventCallback(nn::sdmmc::Port_SdCard0);
        nn::sdmmc::Finalize(nn::sdmmc::Port_SdCard0);
    }

}}  // namespace nn { namespace fs {   // 疑似 fs

namespace   // クライアント側
{
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_ThreadStack[4096];

    void ClientThread(void* argument) NN_NOEXCEPT
    {
        NN_UNUSED(argument);

        nn::os::SystemEventType sdCardDetectionEvent;
        nn::fs::BindSdCardDetectionEvent(&sdCardDetectionEvent);

        bool isCurrentInserted = nn::fs::IsSdCardInserted();
        NN_LOG("[client] %s\n", (isCurrentInserted ? "Inserted" : "Removed"));

        for (int i = 0; i < 10; i++)    // 10 回 挿抜イベントをチェックしたら終わる
        {
            NN_LOG("[client] Wait(handle:0x%08X)\n", nn::os::GetReadableHandleOfSystemEvent(&sdCardDetectionEvent));
            nn::os::WaitSystemEvent(&sdCardDetectionEvent);
            nn::os::ClearSystemEvent(&sdCardDetectionEvent);

            // 次のイベントを漏らさない状態（ClearSystemEvent）にしてから、現在の挿抜状態に対する処理を行う

            bool isLastInserted = isCurrentInserted;
            isCurrentInserted = nn::fs::IsSdCardInserted();

            if ((!isLastInserted) && (!isCurrentInserted))
            {
                // 抜 -> ... -> 抜 の場合は何もしない
                NN_LOG("[client] Insert (LOST)\n");
                NN_LOG("[client] Remove (ignore)\n");
            }
            else if ((!isLastInserted) && isCurrentInserted)
            {
                // 抜 -> ... -> 挿 の場合
                NN_LOG("[client] Insert\n");
            }
            else if (isLastInserted && (!isCurrentInserted))
            {
                // 挿 -> ... -> 抜 の場合
                NN_LOG("[client] Remove\n");
            }
            else    // (isLastInserted && isCurrentInserted)
            {
                // 挿 -> ... -> 挿 の場合
                NN_LOG("[client] Remove (LOST)\n");
                NN_LOG("[client] Insert\n");
            }
        }

        nn::fs::UnbindSdCardDetectionEvent(&sdCardDetectionEvent);
    }

    void DoClientProcess()
    {
        nn::os::ThreadType thread;
        nn::os::CreateThread(&thread, ClientThread, nullptr, g_ThreadStack, sizeof(g_ThreadStack), 10);
        nn::os::StartThread(&thread);

        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);
    }
}

extern "C" void nnMain()
{
    // 本物の fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    nn::fs::Initialize();
    DoClientProcess();
    nn::fs::Finalize();

    // 本物の fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());
}
