﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       加速度センサーの処理に特化した API の宣言
 */

#pragma once

#include <nn/xcd/xcd_Input.h>

namespace nn { namespace hid { namespace detail {

/**
 * @brief       設定可能な加速度センサーの遊びの挙動方法
 */
enum AccelerationPlayMode
{
    AccelerationPlayMode_Loose,   //!< Loose mode
    AccelerationPlayMode_Tight,   //!< Tight mode (デフォルトはこちら)
};

//!< 加速度センサーの処理を担うクラス
class Accelerometer final
{
    NN_DISALLOW_COPY(Accelerometer);
    NN_DISALLOW_MOVE(Accelerometer);

private:
    nn::util::Float3            m_ZeroCount;        //!< ゼロ点カウント[count]
    nn::util::Float3            m_SensitivityCount; //!< 感度カウント[count]
    nn::util::Float3            m_Scale;            //!< 加速度値の変換パラメータ [G/count]
    nn::util::Float3            m_LastAcceleration; //!< 前回算出した加速度ベクトル[G]

    AccelerationPlayMode        m_PlayMode;         //!< 加速度センサーの遊びの挙動方法
    float                       m_PlayRadius;       //!< 加速度センサーの遊び半径
    float                       m_Sensitivity;      //!< 加速度センサーの感度

    float                       m_RevisePower;      //!< 加速度補正の重み。0 から 1 の間の値が設定されます。値が大きいほど急激に補正がかかります。
    float                       m_ReviseRange;      //!< 加速度補正の有効範囲。重力加速度を中心にこの範囲内の加速度センサの値を補正計算に使用します。
    bool                        m_ReviseFlag;       //!< 加速度補正機能の状態。True:ON, False:Off

    nn::xcd::AccelerometerFsr   m_Fsr;              //!< 加速度センサーのフルスケールレンジ

public:
    Accelerometer()  NN_NOEXCEPT;
    ~Accelerometer() NN_NOEXCEPT {};

    //!< 加速度を取得します。
    void GetState(nn::util::Float3* pOutState, const nn::xcd::SensorState& accelerationCount) NN_NOEXCEPT;

    //!< 無入力時の加速度カウントを取得します。
    void GetNoInputCount( nn::xcd::SensorState* pOutCount ) const NN_NOEXCEPT;

    //!< 姿勢を加速度値を用いて更新します。
    float UpdateDirection( nn::hid::DirectionState* pOutDirection, const nn::util::Float3& acceleration ) const NN_NOEXCEPT;

    //!< フルスケールレンジを設定/取得します。
    void SetSensorFsr(const nn::xcd::AccelerometerFsr& accFsr) NN_NOEXCEPT;
    void GetSensorFsr(nn::xcd::AccelerometerFsr* pOutAccFsr) NN_NOEXCEPT;

    //!< 加速度センサーのキャリブレーション値を設定します。
    void SetCalibrationValue( const nn::xcd::SensorState& accelerometerOrigin,
                              const nn::xcd::SensorState& accelerometerSensitivity ) NN_NOEXCEPT;

    //!< 加速度センサーの遊びの挙動方法を設定/取得します。
    void SetPlayMode( const AccelerationPlayMode& mode ) NN_NOEXCEPT;
    void GetPlayMode( AccelerationPlayMode* pOutMode ) const NN_NOEXCEPT;

    //!< 加速度センサーの遊び半径と感度を設定/取得します。
    void SetParameters( const float& playRadius, const float& sensitivity ) NN_NOEXCEPT;
    void GetParameters( float* pOutPlayRadius, float* pOutSensitivity ) const NN_NOEXCEPT;

    //!< 姿勢値の加速度補正パラメータを設定/取得します。
    void SetAccelerationReviseParameters( const float& revisePower, const float& reviseRange ) NN_NOEXCEPT;
    void GetAccelerationReviseParameters( float* pOutRevisePower, float* pOutReviseRange ) const NN_NOEXCEPT;
    void ResetAccelerationReviseParameters() NN_NOEXCEPT;

    //!< 姿勢値の加速度補正機能の状態を設定/取得します。
    void ControlAccelerationRevise( bool isEnable ) NN_NOEXCEPT;
    bool IsEnableAccelerationRevise() const NN_NOEXCEPT;

private:

    //!< 加速度センサーのスケーリングパラメータを設定します。
    void SetScaleValue() NN_NOEXCEPT;

    //!< カウント値を加速度値[G] に変換します。
    void ConvertToAcceleration( nn::util::Float3* pOut, const nn::xcd::SensorState& count ) const NN_NOEXCEPT;

    //!< 遊びの挙動方法に応じた加速度値[G]の補正を行います。
    void ReviseAccelerationByPlayMode( nn::util::Float3* pOut, const nn::util::Float3& acceleration ) const NN_NOEXCEPT;

    //!< 内部的に保持している前回サンプルの加速度カウント値を更新します。
    void UpdateLastAcceleration( const nn::util::Float3& acceleration ) NN_NOEXCEPT;

    //!<  AccelerationPlayMode に応じた加速度の補正をします。( __calcAcc 相当 )
    float CalcAcceleration(const float& currentAcc, const float& lastAcc ) const NN_NOEXCEPT;

};

}}} // namespace nn::hid::detail
