﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   lsm6ds3 のレジスタ周りの定義
 * @details 参考文献：
 *            - [Ref1] LSM6DS3iNEMO inertial module: 3D accelerometer and 3D gyroscope, Jul 2015
 *                     (DocID026899 Rev 6)
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BitPack.h>

#include "lsm6ds3_Types.h"

namespace nnt {
namespace sasbus {
namespace lsm6ds3 {
namespace detail {

const uint8_t WhoAmIValue = 0x69; //!< [Ref1] 9.11 WHO_AM_I register value

/**
 * @brief レジスタのビット定義
 */
struct Register
{
    // [Ref1] 9.11 WHO_AM_I  (0Fh)
    typedef nn::util::BitPack8::Field< 0,  8, int >  WhoAmI;

    // [Ref1] 9.12 CTRL1_XL  (10h)
    typedef nn::util::BitPack8::Field< 4,  4, int >  Ctrl1XlOdr;
    typedef nn::util::BitPack8::Field< 2,  2, int >  Ctrl1XlFs;
    typedef nn::util::BitPack8::Field< 0,  2, int >  Ctrl1XlBw;

    // [Ref1] 9.13 CTRL2_G   (11h)
    typedef nn::util::BitPack8::Field< 4,  4, int >  Ctrl2GOdr;
    typedef nn::util::BitPack8::Field< 1,  3, int >  Ctrl2GFs; // FS125 と併合

    // [Ref1] 9.14 CTRL3_C   (12h)
    typedef nn::util::BitPack8::Field< 6,  1, bool > Ctrl3CBdu;
    typedef nn::util::BitPack8::Field< 2,  1, bool > Ctrl3CIfInc;
    typedef nn::util::BitPack8::Field< 0,  1, bool > Ctrl3CSwReset;

    // [Ref1] 9.15 CTRL4_C   (13h)
    typedef nn::util::BitPack8::Field< 3,  1, bool > Ctrl4CDrdyMask;

    // [Ref1] 9.19 CTRL8_XL   (17h)
    typedef nn::util::BitPack8::Field< 7,  1, bool > Ctrl8XlLpf2XlEn;
    typedef nn::util::BitPack8::Field< 2,  1, bool > Ctrl8XlHpSlopeXlEn;

    // [Ref1] 9.20 CTRL9_XL  (18h)
    typedef nn::util::BitPack8::Field< 5,  1, bool > Ctrl9XlZen ;
    typedef nn::util::BitPack8::Field< 4,  1, bool > Ctrl9XlYen ;
    typedef nn::util::BitPack8::Field< 3,  1, bool > Ctrl9XlXen ;

    // [Ref1] 9.21 CTRL10_C  (19h)
    typedef nn::util::BitPack8::Field< 5,  1, bool > Ctrl10CZen ;
    typedef nn::util::BitPack8::Field< 4,  1, bool > Ctrl10CYen ;
    typedef nn::util::BitPack8::Field< 3,  1, bool > Ctrl10CXen ;
    typedef nn::util::BitPack8::Field< 2,  1, bool > Ctrl10CFuncEn ;

    // [Ref1] 9.28 OUTX_L_G  (22h)
    typedef nn::util::BitPack8::Field< 0,  8, int > OutXLGD  ;
};

/**
 * @brief レジスタアドレスの定義
 */
enum class Address : uint8_t
{
    WhoAmI  = 0x0F, // [Ref1] 9.11 WHO_AM_I  (0Fh)
    Ctrl1Xl = 0x10, // [Ref1] 9.12 CTRL1_XL  (10h)
    Ctrl2G  = 0x11, // [Ref1] 9.13 CTRL2_G   (11h)
    Ctrl3C  = 0x12, // [Ref1] 9.14 CTRL3_C   (12h)
    Ctrl4C  = 0x13, // [Ref1] 9.15 CTRL4_C   (13h)
    Ctrl8Xl = 0x17, // [Ref1] 9.19 CTRL8_XL  (17h)
    Ctrl9Xl = 0x18, // [Ref1] 9.20 CTRL9_XL  (18h)
    Ctrl10C = 0x19, // [Ref1] 9.21 CTRL10_C  (19h)
    OutXLG  = 0x22, // [Ref1] 9.28 OUTX_L_G  (22h)
};

/**
 * @brief 加速度センサーの ODR
 */
enum class AccOdrRegValue : uint8_t
{
    AccOdrRegValue_0Hz    = 0x00,
    AccOdrRegValue_13Hz   = 0x01,
    AccOdrRegValue_26Hz   = 0x02,
    AccOdrRegValue_52Hz   = 0x03,
    AccOdrRegValue_104Hz  = 0x04,
    AccOdrRegValue_208Hz  = 0x05,
    AccOdrRegValue_416Hz  = 0x06,
    AccOdrRegValue_833Hz  = 0x07,
    AccOdrRegValue_1660Hz = 0x08,
    AccOdrRegValue_3330Hz = 0x09,
    AccOdrRegValue_6660Hz = 0x0A,
};

/**
 * @brief ジャイロセンサーの ODR
 */
enum class GyroOdrRegValue : uint8_t
{
    GyroOdrRegValue_0Hz    = 0x00,
    GyroOdrRegValue_13Hz   = 0x01,
    GyroOdrRegValue_26Hz   = 0x02,
    GyroOdrRegValue_52Hz   = 0x03,
    GyroOdrRegValue_104Hz  = 0x04,
    GyroOdrRegValue_208Hz  = 0x05,
    GyroOdrRegValue_416Hz  = 0x06,
    GyroOdrRegValue_833Hz  = 0x07,
    GyroOdrRegValue_1660Hz = 0x08,
};

/**
 * @brief 加速度センサーの FSR
 */
enum class AccFsrRegValue : uint8_t
{
    AccFsrRegValue_2G     = 0x00,
    AccFsrRegValue_16G    = 0x01,
    AccFsrRegValue_4G     = 0x02,
    AccFsrRegValue_8G     = 0x03,
};

/**
 * @brief ジャイロセンサーの FSR
 */
enum class GyroFsrRegValue : uint8_t
{
    GyroFsrRegValue_125dps   = 0x01,
    GyroFsrRegValue_245dps   = 0x00,
    GyroFsrRegValue_500dps   = 0x02,
    GyroFsrRegValue_1000dps  = 0x04,
    GyroFsrRegValue_2000dps  = 0x06,
};

/**
 * @brief 加速度センサー向けのアンチエイリアシングフィルタの周波数帯域
 */
enum class AccAntiAliasingFilterBandwidthRegValue : uint8_t
{
    AccAntiAliasingFilterBandwidthRegValue_400Hz = 0x00,
    AccAntiAliasingFilterBandwidthRegValue_200Hz = 0x01,
    AccAntiAliasingFilterBandwidthRegValue_100Hz = 0x02,
    AccAntiAliasingFilterBandwidthRegValue_50Hz  = 0x03,
};

} // detail
} // lsm6ds3
} // sasbus
} // nnt
