﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/os.h>

#include "WorkerUtil.h"

// ------------------------------------------------------------------------------------------------
// WorkerUtil
// ------------------------------------------------------------------------------------------------
void WorkerUtil::ExecuteWorker(void* pWorker)
{
    WorkerUtil* pWorkerObj = reinterpret_cast<WorkerUtil*>(pWorker);
    nn::Result* pResult    = &(pWorkerObj->m_Result);

    *pResult = pWorkerObj->m_pFunc(pWorkerObj->m_pUserPtr);

    if (pWorkerObj->m_pDoneCallbackFunc)
    {
        pWorkerObj->m_pDoneCallbackFunc();
    }

    pWorkerObj->m_State = State_Done;
    pWorkerObj->m_WaitToBeDone.Signal();

}

WorkerUtil::WorkerUtil(
    WorkerUtil::Func pFunc,
    WorkerUtil::DoneCallbackFunc pDoneCallbackFunc,
    void* pUserPtr,
    int threadStaskSize) : m_WaitToBeDone(nn::os::EventClearMode_ManualClear),
                           m_ThreadStackSize(threadStaskSize),
                           m_pFunc(pFunc),
                           m_pDoneCallbackFunc((pDoneCallbackFunc != nullptr)?(pDoneCallbackFunc):(nullptr)),
                           m_State(State_Invalid)
{
    m_pThreadStack = new char[threadStaskSize + nn::os::StackRegionAlignment];
    m_pUserPtr     = pUserPtr;
    m_State        = State_Initialized;
}

WorkerUtil::~WorkerUtil()
{
    if (m_pThreadStack)
    {
        delete[] m_pThreadStack;
    }
}

WorkerUtil::Error WorkerUtil::Run(int threadPriority)
{
    Error ret = Error_Ok;

    do
    {
        if (m_pThreadStack == nullptr)
        {
            ret = Error_Fatal;
            break;
        }

        if(m_State == State_Running)
        {
            ret = Error_Inprogress;
            break;
        }

        char* pAlignedThreadStackHead = m_pThreadStack;
        pAlignedThreadStackHead = reinterpret_cast<char*>(
            reinterpret_cast<uint64_t>(pAlignedThreadStackHead + (nn::os::StackRegionAlignment - 1)) &
            reinterpret_cast<uint64_t>((uint8_t *)~(nn::os::StackRegionAlignment - 1))
        );

        nn::Result result = nn::os::CreateThread(
            &m_ThreadTid,
            WorkerUtil::ExecuteWorker,
            this,
            pAlignedThreadStackHead,
            m_ThreadStackSize,
            threadPriority);
        if (result.IsFailure())
        {
            ret = Error_Fatal;
            break;
        }

        m_State = State_Running;
        nn::os::StartThread(&m_ThreadTid);
    } while (NN_STATIC_CONDITION(false));

    return ret;
}

WorkerUtil::Error WorkerUtil::WaitToBeDone()
{
    if (m_State < State_Running)
    {
        return Error_NotRunning;
    }

    m_WaitToBeDone.Wait();

    nn::os::WaitThread(&m_ThreadTid);
    nn::os::DestroyThread(&m_ThreadTid);

    return Error_Ok;
}

nn::Result WorkerUtil::GetResult()
{
    return m_Result;
}
