﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <curl/curl.h>
#include <nn/ssl.h>

class SslTestCommonUtil
{
private:
    static const uint32_t TimeoutOnNicConfigurationInSec = 30;

public:
    nn::Result SetupNetwork();
    void FinalizeNetwork();
    int CreateTcpSocket(
        bool bEstablishConn,
        uint16_t portNumber,
        const char* pInHostName,
        uint32_t ipAddress);
    void CloseTcpSocket(int socketFd);

    static void DumpCurlResults(CURL* pInCurlHandle);
};

//-------------------------------------------------------------------------------------------------
//  class SimpleHttpsClient
//-------------------------------------------------------------------------------------------------
class SimpleHttpsClient
{
private:
    static const uint32_t ServerCertBufferSize     = 1024 * 32;  //!< The size of the buffer for server certificate
    static const uint32_t ReadBufferSize           = 1024 * 512; //!< The size of the buffer for received data
    static const uint32_t MaxAsyncDoHandshakeRetry = 30;         //!< The number of count to retry DoHandshake
    static const uint32_t MsecPollTimeout          = 5000;       //!< Poll timeout

    nn::ssl::Context     m_SslContext;         //!< SSL context
    nn::ssl::Connection  m_SslConnection;      //!< SSL connection context
    char                 m_HostName[256];      //!< The host name of the server
    bool                 m_IsNonBlocking;      //!< Blocking mode of I/O APIs
    uint16_t             m_PortNumber;         //!< Post number of the server to establish TCP connection
    int                  m_TcpSocket;          //!< TCP socket descriptor
    bool                 m_IsSocketImported;   //!< It becomes true when m_TcpSocket is passed to the SSL library
    char*                m_pReceivedData;      //!< Pointer to the buffer which received data is stored
    uint32_t             m_ReceivedTotalBytes; //!< The number of bytes received
    bool                 m_IsInitialized;      //!< It becomes true when this class is initialized
    nn::Result           m_LastResult;         //!< Result value of last operation
    nn::ssl::CertStoreId m_ServerCertStoreId;  //!< Certstore ID for server PKI
    nn::ssl::CertStoreId m_ClientCertStoreId;  //!< Certstore ID for client PKI
    uint32_t             m_ManualIpAddress;    //!< Used when resolver is not available

    int        SetupTcpSocket();
    nn::Result SetupSslContext(nn::ssl::Context::SslVersion sslVersion);
    nn::Result SetupSslConnection(nn::ssl::Connection::VerifyOption verifyOption);

public:
    SimpleHttpsClient(bool IsBlocking, const char* pInHostName, uint16_t portNum);
    ~SimpleHttpsClient();

    nn::Result GetLastResult();
    void SetIpAddress(uint32_t address);

    nn::ssl::Connection* GetSslConnection();

    bool Initialize(
        nn::ssl::Context::SslVersion SslVersion,
        nn::ssl::Connection::VerifyOption verifyOption);
    void Finalize();
    bool ImportServerPki(
        const char* pInCertData,
        uint32_t certDataSize,
        nn::ssl::CertificateFormat certFormat);
    bool ImportClientPki(
        const char* pInP12Data,
        const char* pInPwData,
        uint32_t  p12DataSize,
        uint32_t  pwDataSize);
    bool RegisterDeviceClientCert();

    bool PerformSslHandshake(bool g_IsPrintServerCert);
    bool SendHttpGetOverSsl();
    bool ReceiveAllHttpData();
    void PrintReceivedData();
    bool EnableSessionCache();
};

//-------------------------------------------------------------------------------------------------
//  class SimpleCurlHttpsClient
//-------------------------------------------------------------------------------------------------
class SimpleCurlHttpsClient
{
public:
    typedef size_t (*CurlSslCtxFunction)(CURL *curl, void *sslctx, void *parm);
    static const uint32_t HostNameLength = 256;
    static const uint32_t ProxyStrBuffLength = 256;
    static const uint32_t TransactionTimeoutSec = 10;
    enum InitMode
    {
        InitMode_Auto,
        InitMode_Manual
    };

private:
    char        m_HostName[HostNameLength];   //<! Host name
    const char* m_pServerCert;                //<! The pointer to the server cert to import
    uint32_t    m_ServerCertSize;             //<! The size of server cert buffer
    const char* m_pClientPki;                 //<! The pointer to the p12 data
    uint32_t    m_ClientPkiSize;              //<! The size of p12 buffer
    const char* m_pPassword;                  //<! The pointer to the password for p12 data
    uint32_t    m_PasswordSize;               //!< The size of password buffer
    bool        m_IsPerformPeerCaValidation;  //!< Perform server validation when true (true by default)
    bool        m_IsPeformNameValidation;     //!< Perform host name validation when true (true by default)
    InitMode    m_InitMode;                   //!< Initialize/Finalize curl inside this class
    bool        m_IsCurlVerbose;              //!< Enable CURLOPT_VERBOSE when true (false by default)

    nn::ssl::CertStoreId m_ServerCertStoreId; //!< CertStoreId for imported server PKI
    nn::ssl::CertStoreId m_ClientCertStoreId; //!< CertStoreId for imported client PKI

    CurlSslCtxFunction m_pCurlCtxCallback;

    struct ProxyInfo
    {
        bool     isConfigured;
        char     address[ProxyStrBuffLength];
        uint16_t port;
        char     userName[ProxyStrBuffLength];
        char     password[ProxyStrBuffLength];
    } m_ProxyInfo;

public:
    SimpleCurlHttpsClient(
        const char* pInHostName,
        const char* pInServerCert,
        const char* pInClientPki,
        const char* pInPassword,
        uint32_t serverCertSize,
        uint32_t clientPkiSize,
        uint32_t passwordSize,
        InitMode initMode);
    ~SimpleCurlHttpsClient();

    void ConfigureValidation(bool peerCaValidation, bool hostNameValidation);
    void SetCurlCtxFunction(CurlSslCtxFunction pCallback);
    void EnableCurlVerbose();

    bool InitializeManual();
    void FinalizeManual();
    bool Perform();
    bool IsImportServerPki();
    bool IsImportClientPki();

    const char* GetServerCert();
    const char* GetClientPki();
    const char* GetPassword();
    uint32_t    GetServerCertSize();
    uint32_t    GetClientPkiSize();
    uint32_t    GetPasswordSize();

    bool SetupProxy(
        const char* pInProxyAddress,
        const char* pInUserName,
        const char* pInPassword,
        uint16_t portNumber);

    void SetServerCertStoreId(nn::ssl::CertStoreId id);
    void SetClientCertStoreId(nn::ssl::CertStoreId id);
};
