﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/profiler.h>

#include <nnt/nntest.h>

#include <nv/nv_MemoryManagement.h>
#include <nvn/nvn_Cpp.h>
#include <nvn/nvn_CppFuncPtr.h>
#include <nvn/nvn_CppMethods.h>
#include <nvn/nvn_CppFuncPtrImpl.h>
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtr.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>

#include "../../Common/testProfiler_Common.h"

extern "C" nvn::GenericFuncPtrFunc NVNAPIENTRY nvnBootstrapLoader(const char *name);
PFNNVNGENERICFUNCPTRPROC NVNAPIENTRY nvnBootstrapLoader(const char *name);

namespace /*anonymous*/ {

    NN_ALIGNAS(4096) char gProfilerBuffer[nn::profiler::MinimumBufferSize];

    void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        // According to specifications of aligned_alloc(), we need to coordinate the size parameter to become the integral multiple of alignment.
        return aligned_alloc(alignment, nn::util::align_up(size, alignment));
    }
    void NvFreeFunction(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }
    void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }

    void* NvDevtoolsAllocateFunction(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        // According to specifications of aligned_alloc(), we need to coordinate the size parameter to become the integral multiple of alignment.
        return aligned_alloc(alignment, nn::util::align_up(size, alignment));
    }
    void NvDevtoolsFreeFunction(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }
    void* NvDevtoolsReallocateFunction(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }

    void InitializeCommonNvn()
    {
        /*
        * Set memory allocator for graphics subsystem.
        * This function must be called before using any graphics API's.
        */
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);

        /*
        * Set memory allocator for graphics developer tools and NVN debug layer.
        * This function must be called before using any graphics developer features.
        */
        nv::SetGraphicsDevtoolsAllocator(NvDevtoolsAllocateFunction, NvDevtoolsFreeFunction, NvDevtoolsReallocateFunction, NULL);

        /*
        * Donate memory for graphics driver to work in.
        * This function must be called before using any graphics API's.
        */
        size_t graphicsSystemMemorySize = 8 * 1024 * 1024;
        void* graphicsHeap = malloc(graphicsSystemMemorySize);
        nv::InitializeGraphics(graphicsHeap, graphicsSystemMemorySize);
    }

    void InitializeNvn(bool useCpp)
    {
        if (useCpp)
        {
            nvn::DeviceGetProcAddressFunc cppGetProc = (nvn::DeviceGetProcAddressFunc)((*nvnBootstrapLoader)("nvnDeviceGetProcAddress"));
            nvn::nvnLoadCPPProcs(nullptr, cppGetProc);
        }
        else
        {
            PFNNVNDEVICEGETPROCADDRESSPROC getProc = (PFNNVNDEVICEGETPROCADDRESSPROC)((*nvnBootstrapLoader)("nvnDeviceGetProcAddress"));
            nvnLoadCProcs(nullptr, getProc);
        }
    }

    void UninitializeNvn()
    {
        // Bit of a misnomer here. We are only clearing the bit that NXCP uses to find NVN functions.
        pfnc_nvnDeviceGetProcAddress = nullptr;
        nvn::pfncpp_nvnDeviceGetProcAddress = nullptr;
    }
} // anonymous


namespace nn { namespace profiler {

bool IsPCConnected();

}} // namespace nn::profiler



void WaitForProfileToBeTaken()
{
    nn::Result result;
    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    EXPECT_RESULT_SUCCESS(result);
    while (!nn::profiler::IsPCConnected())
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    while (nn::profiler::GetProfilerStatus() != nn::profiler::ProfilerStatus_Profiling)
    {
        nn::os::YieldThread();
    }
    while (nn::profiler::GetProfilerStatus() != nn::profiler::ProfilerStatus_Active)
    {
        nn::os::YieldThread();
    }

    while (nn::profiler::IsPCConnected())
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    result = nn::profiler::Finalize();
    EXPECT_RESULT_SUCCESS(result);
}



TEST(AutogenVerify, Nvn)
{
    NN_LOG("Check with unloaded NVN function pointers.\n");
    InitializeCommonNvn();
    WaitForProfileToBeTaken();

    NN_LOG("Check with loaded C-style NVN function pointers.\n");
    InitializeNvn(false);
    WaitForProfileToBeTaken();

    NN_LOG("Check again with unloaded NVN function pointers.\n");
    UninitializeNvn();
    WaitForProfileToBeTaken();

    NN_LOG("Check with load C++-style NVN function pointers.\n");
    InitializeNvn(true);
    WaitForProfileToBeTaken();
}
