﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/util/util_Endian.h>
#include <nn/util/util_FormatString.h>

#include <nn/prepo.h>
#include <nn/prepo/detail/prepo_ApiDetail.h>
#include <nn/prepo/detail/msgpack/prepo_MessagePack.h>

namespace
{
    const size_t BufferSize = nn::prepo::PlayReport::BufferSizeMax * 2;
    nn::Bit8 g_Buffer[BufferSize];
}

TEST(ValidKeyValueCount, Zero)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 0));

    EXPECT_TRUE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(ValidKeyValueCount, Max)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, nn::prepo::KeyValueCountMax));

    for (int i = 0; i < nn::prepo::KeyValueCountMax; i++)
    {
        const int keyLength = nn::prepo::KeyLengthMax;
        char key[keyLength + 1];
        ASSERT_EQ(keyLength, nn::util::SNPrintf(key, sizeof(key), "%0*d", keyLength, i));
        ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, key, keyLength));

        const int valueLength = nn::prepo::StringValueLengthMax;
        char value[valueLength + 1];
        ASSERT_EQ(valueLength, nn::util::SNPrintf(value, sizeof(value), "%0*d", valueLength, i));
        ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, value, valueLength));
    }

    EXPECT_TRUE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidKeyValueCount, Overflow)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, BufferSize};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, nn::prepo::KeyValueCountMax + 1));

    for (int i = 0; i < nn::prepo::KeyValueCountMax + 1; i++)
    {
        const int keyLength = nn::prepo::KeyLengthMax + 1;
        char key[keyLength + 1];
        ASSERT_EQ(keyLength, nn::util::SNPrintf(key, sizeof(key), "%0*d", keyLength, i));
        ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, key, keyLength));

        const int valueLength = nn::prepo::StringValueLengthMax + 1;
        char value[valueLength + 1];
        ASSERT_EQ(valueLength, nn::util::SNPrintf(value, sizeof(value), "%0*d", valueLength, i));
        ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, value, valueLength));
    }

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidKeyValueCount, TooFew)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteSignedIntegerAutoSize(&stream, 0));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key2", sizeof ("key2") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteSignedIntegerAutoSize(&stream, 0));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidKeyValueCount, TooMany)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 2));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteSignedIntegerAutoSize(&stream, 0));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(ValidKeyFormat, LengthMax)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    const int keyLength = nn::prepo::KeyLengthMax;
    char key[keyLength + 1];
    ASSERT_EQ(keyLength, nn::util::SNPrintf(key, sizeof(key), "%0*d", keyLength, 0));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, key, keyLength));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteSignedIntegerAutoSize(&stream, 0));

    EXPECT_TRUE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidKeyFormat, LengthTooLong)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    const int keyLength = nn::prepo::KeyLengthMax + 1;
    char key[keyLength + 1];
    ASSERT_EQ(keyLength, nn::util::SNPrintf(key, sizeof(key), "%0*d", keyLength, 0));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, key, keyLength));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteSignedIntegerAutoSize(&stream, 0));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidKeyFormat, EmptyString)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "", sizeof ("") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteSignedIntegerAutoSize(&stream, 0));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(KeyDuplication, SaveValue)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 2));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteSignedIntegerAutoSize(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteSignedIntegerAutoSize(&stream, 2));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(KeyDuplication, SameCase)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 2));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteSignedIntegerAutoSize(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "KEY1", sizeof ("KEY1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteSignedIntegerAutoSize(&stream, 2));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidKeyType, KeyTypeSignedInteger)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteSignedIntegerAutoSize(&stream, 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "value1", sizeof ("value1") - 1));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidKeyType, KeyTypeBinary)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteBinary(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "value1", sizeof ("value1") - 1));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidKeyType, KeyTypeBool)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteBool(&stream, true));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "value1", sizeof ("value1") - 1));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidKeyType, KeyTypeArray)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteArrayAutoSize(&stream, 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "array_value1", sizeof ("array_value1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "value1", sizeof ("value1") - 1));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidKeyType, KeyTypeMap)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMapAutoSize(&stream, 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "map_key1", sizeof ("map_key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "map_value1", sizeof ("map_value1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "value1", sizeof ("value1") - 1));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidKeyType, KeyTypeExtension)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    nn::Bit32 value;
    nn::util::StoreBigEndian(&value, static_cast<nn::Bit32>(1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteExtension(&stream, nn::prepo::detail::MsgpackExtensionType_Any64BitId, &value, sizeof(value)));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "value1", sizeof ("value1") - 1));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(ValidValueFormat, StringLengthMax)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));

    const int valueLength = nn::prepo::StringValueLengthMax;
    char value[valueLength + 1];
    ASSERT_EQ(valueLength, nn::util::SNPrintf(value, sizeof(value), "%0*d", valueLength, 0));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, value, valueLength));

    EXPECT_TRUE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidValueFormat, StringLengthTooLong)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));

    const int valueLength = nn::prepo::StringValueLengthMax + 1;
    char value[valueLength + 1];
    ASSERT_EQ(valueLength, nn::util::SNPrintf(value, sizeof(value), "%0*d", valueLength, 0));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, value, valueLength));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidValueType, ValueTypeBinary)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteBinary(&stream, "value1", sizeof ("value1") - 1));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidValueType, ValueTypeBool)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteBool(&stream, true));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidValueType, ValueTypeArray)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteArrayAutoSize(&stream, 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "array_value1", sizeof ("array_value1") - 1));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidValueType, ValueTypeMap)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMapAutoSize(&stream, 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "map_key1", sizeof ("map_key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "map_value1", sizeof ("map_value1") - 1));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidValueType, ValueTypeExtension)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    nn::Bit32 value;
    nn::util::StoreBigEndian(&value, static_cast<nn::Bit32>(1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteExtension(&stream, nn::prepo::detail::MsgpackExtensionType_Any64BitId, &value, sizeof(value)));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position));
}

TEST(InvalidSize, TooBig)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "value1", sizeof ("value1") - 1));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position + 1));
}

TEST(InvalidSize, TooSmall)
{
    nn::prepo::detail::msgpack::OutputStreamParam stream = {g_Buffer, nn::prepo::PlayReport::BufferSizeMax};

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteMap16(&stream, 1));

    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "key1", sizeof ("key1") - 1));
    ASSERT_TRUE(nn::prepo::detail::msgpack::WriteString(&stream, "value1", sizeof ("value1") - 1));

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(stream.buffer, stream.position - 1));
}
