﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os.h>

#include <nn/prepo.h>
#include <nn/prepo/prepo_ApiConfig.h>
#include <nn/prepo/prepo_ApiAdmin.h>
#include <nn/prepo/prepo_Result.h>
#include <nn/prepo/prepo_SystemPlayReport.h>

#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>

#define TEST_STATUS_STRING(str) "\033[36m[test] " str "\033[m"

namespace
{
    int g_UserCount;
    nn::account::Uid g_Users[nn::account::UserCountMax] = {};
}

TEST(UserAgreementCheckConfig, Initialize)
{
    nn::account::Initialize();
    NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&g_UserCount,
        g_Users, nn::account::UserCountMax));
    if (g_UserCount == 0)
    {
        NN_LOG(TEST_STATUS_STRING("This test needs one or more users.\n"));
        NN_ABORT_UNLESS(g_UserCount > 0);
    }
}

TEST(UserAgreementCheckConfig, Interface)
{
    bool isEnabled;

    NNT_EXPECT_RESULT_SUCCESS(nn::prepo::IsUserAgreementCheckEnabled(&isEnabled));
    NN_LOG(TEST_STATUS_STRING("user agreement check = %s\n"), isEnabled ? "enabled" : "disabled");

    NNT_EXPECT_RESULT_SUCCESS(nn::prepo::SetUserAgreementCheckEnabled(false));
    NNT_EXPECT_RESULT_SUCCESS(nn::prepo::IsUserAgreementCheckEnabled(&isEnabled));
    NN_LOG(TEST_STATUS_STRING("user agreement check = %s\n"), isEnabled ? "enabled" : "disabled");
    EXPECT_EQ(false, isEnabled);

    NNT_EXPECT_RESULT_SUCCESS(
        nn::prepo::SetUserAgreementCheckEnabled(true));
    NNT_EXPECT_RESULT_SUCCESS(nn::prepo::IsUserAgreementCheckEnabled(&isEnabled));
    NN_LOG(TEST_STATUS_STRING("user agreement check = %s\n"), isEnabled ? "enabled" : "disabled");
    EXPECT_EQ(true, isEnabled);
}

TEST(UserAgreementCheckConfig, CheckEnabled)
{
    NNT_ASSERT_RESULT_SUCCESS(
        nn::prepo::SetUserAgreementCheckEnabled(true));

    nn::prepo::PlayReport report;

    static char buffer[nn::prepo::PlayReport::BufferSizeMax];
    report.SetBuffer(buffer, sizeof (buffer));

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("test_check_enabled"));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("key", "value"));

    NNT_EXPECT_RESULT_SUCCESS(report.Save());

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("test_check_enabled_with_user"));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("key", "value"));

    NNT_EXPECT_RESULT_SUCCESS(report.Save(g_Users[0]));
}

TEST(UserAgreementCheckConfig, CheckDisabled)
{
    NNT_ASSERT_RESULT_SUCCESS(
        nn::prepo::SetUserAgreementCheckEnabled(false));

    nn::prepo::PlayReport report;

    static char buffer[nn::prepo::PlayReport::BufferSizeMax];
    report.SetBuffer(buffer, sizeof (buffer));

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("test_check_disabled"));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("key", "value"));

    NNT_EXPECT_RESULT_SUCCESS(report.Save());

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("test_check_disabled_with_user"));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("key", "value"));

    NNT_EXPECT_RESULT_SUCCESS(report.Save(g_Users[0]));
}
