﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/os.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_NetworkConnection.h>

#include <nn/prepo.h>
#include <nn/prepo/prepo_ApiAdmin.h>
#include <nn/prepo/prepo_ApiConfig.h>
#include <nn/prepo/prepo_Result.h>
#include <nn/prepo/prepo_SystemPlayReport.h>

#include "../Common/testPrepo_Common.h"

#define TEST_STATUS_STRING(str) "\033[36m[test] " str "\033[m"

namespace
{
    void SavePlayReport(const char* eventId)
    {
        nn::prepo::SystemPlayReport report(eventId);
        char buffer[1024];

        nn::ApplicationId appId = {0x0100000000003006};

        report.SetApplicationId(appId);
        report.SetBuffer(buffer, sizeof (buffer));

        NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));
        NNT_EXPECT_RESULT_SUCCESS(report.Add("position:y", 0.00));
        NNT_EXPECT_RESULT_SUCCESS(report.Add("position:z", -5.55));

        report.Save();
    }
}

TEST(TransmissionStatus, Initialize)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::ClearStorage());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::SetUserAgreementCheckEnabled(false));

    NN_LOG("Wait for completion of upload playreport by PDM.\n");
    EXPECT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(nn::prepo::TransmissionStatus_Idle,
        nn::TimeSpan::FromSeconds(10)));

    nn::nifm::Initialize();
}

TEST(TransmissionStatus, Basic)
{
    NN_LOG(TEST_STATUS_STRING("Checking whether the uploader is idling.\n"));
    EXPECT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(nn::prepo::TransmissionStatus_Idle,
        nn::TimeSpan::FromSeconds(10)));

    NN_LOG(TEST_STATUS_STRING("Save play report.\n"));
    SavePlayReport("transmission_status_basic");

    NN_LOG(TEST_STATUS_STRING("Checking whether the uploading will start.\n"));
    EXPECT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(nn::prepo::TransmissionStatus_Processing,
        nn::TimeSpan::FromSeconds(20)));

    NN_LOG(TEST_STATUS_STRING("Checking whether the uploading will complete.\n"));
    EXPECT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(nn::prepo::TransmissionStatus_Idle,
        nn::TimeSpan::FromSeconds(30)));
}

// アプリが強いネットワーク利用要求を出したときに、プレイレポートのアップロードが停止するか。
TEST(TransmissionStatus, YieldConnection)
{
    nn::nifm::NetworkConnection nc;

    NN_LOG(TEST_STATUS_STRING("Request network connection from application.\n"));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(
        nc.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));
    nc.SubmitRequestAndWait();
    ASSERT_TRUE(nc.IsAvailable());

    NN_LOG(TEST_STATUS_STRING("Checking whether the uploader is idling.\n"));
    EXPECT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(nn::prepo::TransmissionStatus_Idle,
        nn::TimeSpan::FromSeconds(10)));

    NN_LOG(TEST_STATUS_STRING("Save play report.\n"));
    SavePlayReport("transmission_status_yield_conne");

    NN_LOG(TEST_STATUS_STRING("Checking whether the uploading will NOT start.\n"));
    EXPECT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(nn::prepo::TransmissionStatus_Pending,
        nn::TimeSpan::FromSeconds(10)));

    NN_LOG(TEST_STATUS_STRING("Cancel network request from application.\n"));
    nc.CancelRequest();

    NN_LOG(TEST_STATUS_STRING("Save play report.\n"));
    SavePlayReport("transmission_status");

    NN_LOG(TEST_STATUS_STRING("Checking whether the uploading will start.\n"));
    EXPECT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(nn::prepo::TransmissionStatus_Processing,
        nn::TimeSpan::FromSeconds(20)));

    NN_LOG(TEST_STATUS_STRING("Checking whether the uploading will complete.\n"));
    EXPECT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(nn::prepo::TransmissionStatus_Idle,
        nn::TimeSpan::FromSeconds(30)));
}

TEST(TransmissionStatus, Finalize)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::SetUserAgreementCheckEnabled(true));
    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::ClearStorage());
}
