﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/os.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_NetworkConnection.h>

#include <nn/prepo.h>
#include <nn/prepo/prepo_ApiAdmin.h>
#include <nn/prepo/prepo_Result.h>
#include <nn/prepo/prepo_SystemPlayReport.h>

#define TEST_STATUS_STRING(str) "\033[36m[test] " str "\033[m"

namespace
{
    void SetupMaxSizeReport(nn::prepo::SystemPlayReport* pReport, int id)
    {
        char eventId[nn::prepo::EventIdLengthMax + 1] = {};
        nn::util::SNPrintf(eventId, sizeof (eventId), "storage_overflow_0000%010d", id);
        NNT_EXPECT_RESULT_SUCCESS(pReport->SetEventId(eventId));

        for (auto i = 0; i < nn::prepo::KeyValueCountMax; i++)
        {
            char key[nn::prepo::KeyLengthMax + 1] = {};
            char value[nn::prepo::StringValueLengthMax + 1] = {};

            nn::util::SNPrintf(key, sizeof (key), "key_00000000000000000000000000000000000000000000000000000000%03d", i);
            nn::util::SNPrintf(value, sizeof (value), "value_000000000000000000000000000000000000000000000000000000%03d", i);

            NNT_EXPECT_RESULT_SUCCESS(pReport->Add(key, value));
        }
        EXPECT_EQ(pReport->GetCount(), nn::prepo::KeyValueCountMax);
    }
}

TEST(StorageOverFlow, Initialize)
{
    nn::nifm::Initialize();
    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::ClearStorage());
}

TEST(StorageOverFlow, Main)
{
    nn::prepo::SystemPlayReport report;
    static char buffer[nn::prepo::PlayReport::BufferSizeMax];
    report.SetBuffer(buffer, sizeof (buffer));
    nn::ApplicationId applicationId = {0x0123456789abcdef};
    NNT_EXPECT_RESULT_SUCCESS(report.SetApplicationId(applicationId));

    // 大量アップロードを防ぐために、強いネットワーク利用要求を出して、
    // アップロードが開始されないようにする。
    nn::nifm::NetworkConnection nc;

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(
        nc.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));

    NN_LOG(TEST_STATUS_STRING("Request network connection from application.\n"));
    nc.SubmitRequestAndWait();

    auto successCount = 0;
    auto outOfResourceCount = 0;
    for (auto i = 0; i < 1000; i++)
    {
        SetupMaxSizeReport(&report, i);

        auto result = report.Save();
        if (result.IsSuccess())
        {
            successCount++;
        }
        else if (nn::prepo::ResultOutOfResource::Includes(result))
        {
            outOfResourceCount++;
        }
        ASSERT_TRUE(result.IsSuccess() || nn::prepo::ResultOutOfResource::Includes(result)); // MEMO: ResultOutOfResource は将来的に shim で握りつぶす。

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
    }
    NN_LOG("success=%d, outOfResource=%d\n", successCount, outOfResourceCount);

    // 大量アップロードを防ぐために、データ管理ストレージをクリアしてから、
    // ネットワーク利用要求をキャンセルする。
    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::ClearStorage());

    NN_LOG(TEST_STATUS_STRING("Cancel network request from application.\n"));
    nc.CancelRequest();
}
