﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Common.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/prepo/detail/service/core/prepo_FileSystem.h>
#include <nn/prepo/detail/service/core/prepo_StatisticsManager.h>

#include "../Common/testPrepo_Common.h"

TEST(Initialize, Main)
{
    nn::prepo::detail::service::core::FileSystem::EnableTestMode();

    nn::prepo::detail::service::core::FileSystem::MountAll();
}

TEST(Load, NoFile)
{
    auto& manager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();

    nn::fs::DeleteFile(manager.FilePath);

    ASSERT_FALSE(nnt::prepo::FileExists(manager.FilePath));

    NNT_ASSERT_RESULT_SUCCESS(manager.Load());

    nn::prepo::Statistics statistics;

    manager.GetStatistics(&statistics);

    for (int i = 0; i < nn::prepo::ReportCategory_Count; i++)
    {
        const auto category = static_cast<nn::prepo::ReportCategory>(i);

        EXPECT_EQ(0, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].uploaded.size);
        EXPECT_EQ(0, statistics.groups[category].lostByBufferShortage.count);
        EXPECT_EQ(0, statistics.groups[category].lostByBufferShortage.size);
        EXPECT_EQ(0, statistics.groups[category].lostByStorageShortage.count);
        EXPECT_EQ(0, statistics.groups[category].lostByStorageShortage.size);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.size);
    }
}

TEST(Load, FileExists)
{
    auto& manager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();

    nn::fs::DeleteFile(manager.FilePath);

    ASSERT_FALSE(nnt::prepo::FileExists(manager.FilePath));

    nn::prepo::Statistics savedStatistics;
    nn::os::GenerateRandomBytes(&savedStatistics, sizeof (savedStatistics));

    NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::WriteFile(manager.FilePath, savedStatistics));

    NNT_ASSERT_RESULT_SUCCESS(manager.Load());

    nn::prepo::Statistics loadedStatistics;

    manager.GetStatistics(&loadedStatistics);

    EXPECT_EQ(0, std::memcmp(&savedStatistics, &loadedStatistics, sizeof (nn::prepo::Statistics)));
}

TEST(Load, InvalidFileSize)
{
    auto& manager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();

    // 不正なサイズの統計情報のファイルを作成する
    {
        nn::fs::DeleteFile(manager.FilePath);

        ASSERT_FALSE(nnt::prepo::FileExists(manager.FilePath));

        struct {
            nn::prepo::Statistics::Record uploaded;
            nn::prepo::Statistics::Record lostByBufferShortage;
            nn::prepo::Statistics::Record lostByStorageShortage;
            nn::prepo::Statistics::Record lostByDisagree;
            nn::Bit8 reserved[64];
        } invalidStatistics;

        NN_STATIC_ASSERT(!(sizeof (invalidStatistics) == sizeof (nn::prepo::Statistics)));

        NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::WriteFile(manager.FilePath, invalidStatistics));
    }

    // 不正なサイズの統計情報のファイルが削除されて、統計情報が初期化される
    {
        ASSERT_TRUE(nnt::prepo::FileExists(manager.FilePath));

        NNT_ASSERT_RESULT_SUCCESS(manager.Load());

        ASSERT_FALSE(nnt::prepo::FileExists(manager.FilePath));

        nn::prepo::Statistics statistics;

        manager.GetStatistics(&statistics);

        for (int i = 0; i < nn::prepo::ReportCategory_Count; i++)
        {
            const auto category = static_cast<nn::prepo::ReportCategory>(i);

            EXPECT_EQ(0, statistics.groups[category].uploaded.count);
            EXPECT_EQ(0, statistics.groups[category].uploaded.size);
            EXPECT_EQ(0, statistics.groups[category].lostByBufferShortage.count);
            EXPECT_EQ(0, statistics.groups[category].lostByBufferShortage.size);
            EXPECT_EQ(0, statistics.groups[category].lostByStorageShortage.count);
            EXPECT_EQ(0, statistics.groups[category].lostByStorageShortage.size);
            EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
            EXPECT_EQ(0, statistics.groups[category].lostByDisagree.size);
        }
    }
}

TEST(Save, NoFile)
{
    auto& manager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();

    for (int i = 0; i < nn::prepo::ReportCategory_Count; i++)
    {
        const auto category = static_cast<nn::prepo::ReportCategory>(i);

        manager.AddUploaded(category, nnt::prepo::GetRandom<size_t>(), 1);
        manager.AddLostByBufferShortage(category, nnt::prepo::GetRandom<size_t>(), 1);
        manager.AddLostByStorageShortage(category, nnt::prepo::GetRandom<size_t>(), 1);
        manager.AddLostByDisagree(category, nnt::prepo::GetRandom<size_t>(), 1);
    }

    nn::prepo::Statistics statistics;

    manager.GetStatistics(&statistics);

    nn::fs::DeleteFile(manager.FilePath);

    ASSERT_FALSE(nnt::prepo::FileExists(manager.FilePath));

    NNT_ASSERT_RESULT_SUCCESS(manager.Save());

    ASSERT_TRUE(nnt::prepo::FileExists(manager.FilePath));

    int64_t fileSize;

    NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::GetFileSize(&fileSize, manager.FilePath));

    ASSERT_EQ(sizeof (nn::prepo::Statistics), static_cast<size_t>(fileSize));

    nn::prepo::Statistics savedStatistics;

    NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::ReadFile(&savedStatistics, manager.FilePath));

    ASSERT_EQ(0, std::memcmp(&statistics, &savedStatistics, sizeof(nn::prepo::Statistics)));
}

TEST(Save, FileExists)
{
    auto& manager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();

    nn::fs::DeleteFile(manager.FilePath);

    ASSERT_FALSE(nnt::prepo::FileExists(manager.FilePath));

    nn::prepo::Statistics existsingStatistics = {};

    NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::WriteFile(manager.FilePath, existsingStatistics));

    for (int i = 0; i < nn::prepo::ReportCategory_Count; i++)
    {
        const auto category = static_cast<nn::prepo::ReportCategory>(i);

        manager.AddUploaded(category, nnt::prepo::GetRandom<size_t>(), 1);
        manager.AddLostByBufferShortage(category, nnt::prepo::GetRandom<size_t>(), 1);
        manager.AddLostByStorageShortage(category, nnt::prepo::GetRandom<size_t>(), 1);
        manager.AddLostByDisagree(category, nnt::prepo::GetRandom<size_t>(), 1);
    }

    nn::prepo::Statistics statistics;

    manager.GetStatistics(&statistics);

    NNT_ASSERT_RESULT_SUCCESS(manager.Save());

    int64_t fileSize;

    NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::GetFileSize(&fileSize, manager.FilePath));

    ASSERT_EQ(sizeof (nn::prepo::Statistics), static_cast<size_t>(fileSize));

    nn::prepo::Statistics savedStatistics;

    NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::ReadFile(&savedStatistics, manager.FilePath));

    ASSERT_EQ(0, std::memcmp(&statistics, &savedStatistics, sizeof(nn::prepo::Statistics)));
}

TEST(Save, InvalidFileSize)
{
    auto& manager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();

    // 不正なサイズの統計情報のファイルを作成する
    {
        nn::fs::DeleteFile(manager.FilePath);

        ASSERT_FALSE(nnt::prepo::FileExists(manager.FilePath));

        struct {
            nn::prepo::Statistics::Record uploaded;
            nn::prepo::Statistics::Record lostByBufferShortage;
            nn::prepo::Statistics::Record lostByStorageShortage;
            nn::prepo::Statistics::Record lostByDisagree;
            nn::Bit8 reserved[64];
        } invalidStatistics;

        NN_STATIC_ASSERT(!(sizeof(invalidStatistics) == sizeof(nn::prepo::Statistics)));

        NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::WriteFile(manager.FilePath, invalidStatistics));
    }

    // 不正なサイズの統計情報のファイルが削除されて、適切なサイズの統計情報のファイルが作成される
    {
        int64_t fileSize;

        NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::GetFileSize(&fileSize, manager.FilePath));

        ASSERT_NE(sizeof (nn::prepo::Statistics), fileSize);

        for (int i = 0; i < nn::prepo::ReportCategory_Count; i++)
        {
            const auto category = static_cast<nn::prepo::ReportCategory>(i);

            manager.AddUploaded(category, nnt::prepo::GetRandom<size_t>(), 1);
            manager.AddLostByBufferShortage(category, nnt::prepo::GetRandom<size_t>(), 1);
            manager.AddLostByStorageShortage(category, nnt::prepo::GetRandom<size_t>(), 1);
            manager.AddLostByDisagree(category, nnt::prepo::GetRandom<size_t>(), 1);
        }

        nn::prepo::Statistics statistics;

        manager.GetStatistics(&statistics);

        NNT_ASSERT_RESULT_SUCCESS(manager.Save());

        NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::GetFileSize(&fileSize, manager.FilePath));

        ASSERT_EQ(sizeof (nn::prepo::Statistics), fileSize);

        nn::prepo::Statistics savedStatistics;

        NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::ReadFile(&savedStatistics, manager.FilePath));

        ASSERT_EQ(0, std::memcmp(&statistics, &savedStatistics, sizeof(nn::prepo::Statistics)));
    }
}

TEST(Clear, NoFile)
{
    auto& manager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();

    nn::fs::DeleteFile(manager.FilePath);

    ASSERT_FALSE(nnt::prepo::FileExists(manager.FilePath));

    for (int i = 0; i < nn::prepo::ReportCategory_Count; i++)
    {
        const auto category = static_cast<nn::prepo::ReportCategory>(i);

        manager.AddUploaded(category, nnt::prepo::GetRandom<size_t>(), 1);
        manager.AddLostByBufferShortage(category, nnt::prepo::GetRandom<size_t>(), 1);
        manager.AddLostByStorageShortage(category, nnt::prepo::GetRandom<size_t>(), 1);
        manager.AddLostByDisagree(category, nnt::prepo::GetRandom<size_t>(), 1);
    }

    NNT_ASSERT_RESULT_SUCCESS(manager.Clear());

    ASSERT_FALSE(nnt::prepo::FileExists(manager.FilePath));

    nn::prepo::Statistics statistics;

    manager.GetStatistics(&statistics);

    for (int i = 0; i < nn::prepo::ReportCategory_Count; i++)
    {
        const auto category = static_cast<nn::prepo::ReportCategory>(i);

        EXPECT_EQ(0, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].uploaded.size);
        EXPECT_EQ(0, statistics.groups[category].lostByBufferShortage.count);
        EXPECT_EQ(0, statistics.groups[category].lostByBufferShortage.size);
        EXPECT_EQ(0, statistics.groups[category].lostByStorageShortage.count);
        EXPECT_EQ(0, statistics.groups[category].lostByStorageShortage.size);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.size);
    }
}

TEST(Clear, FileExists)
{
    auto& manager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();

    nn::fs::DeleteFile(manager.FilePath);

    ASSERT_FALSE(nnt::prepo::FileExists(manager.FilePath));

    nn::prepo::Statistics savedStatistics;
    nn::os::GenerateRandomBytes(&savedStatistics, sizeof (savedStatistics));

    NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::WriteFile(manager.FilePath, savedStatistics));

    NNT_ASSERT_RESULT_SUCCESS(manager.Clear());

    ASSERT_FALSE(nnt::prepo::FileExists(manager.FilePath));

    nn::prepo::Statistics statistics;

    manager.GetStatistics(&statistics);

    for (int i = 0; i < nn::prepo::ReportCategory_Count; i++)
    {
        const auto category = static_cast<nn::prepo::ReportCategory>(i);

        EXPECT_EQ(0, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].uploaded.size);
        EXPECT_EQ(0, statistics.groups[category].lostByBufferShortage.count);
        EXPECT_EQ(0, statistics.groups[category].lostByBufferShortage.size);
        EXPECT_EQ(0, statistics.groups[category].lostByStorageShortage.count);
        EXPECT_EQ(0, statistics.groups[category].lostByStorageShortage.size);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.size);
    }
}
