﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Common.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/prepo/detail/prepo_PlayReportGenerator.h>
#include <nn/prepo/detail/service/core/prepo_FileSystem.h>
#include <nn/prepo/detail/service/core/prepo_ReportFileManager.h>
#include <nn/prepo/detail/service/core/prepo_SystemInfo.h>

#include "../Common/testPrepo_Common.h"

namespace
{
    nn::Result WriteReport(size_t* outSize, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        auto data = static_cast<nn::Bit8*>(buffer);
        auto dataSize = bufferSize;

        size_t position;

        nn::prepo::detail::PlayReportGenerator::Initialize(&position, data, dataSize);

        auto Add = [&](const char* key, double value) -> nn::Result
        {
            return nn::prepo::detail::PlayReportGenerator::AddKeyValue(
                &position, key, value, data, dataSize, position);
        };

        NN_RESULT_DO(Add("x", 1.23));
        NN_RESULT_DO(Add("y", 4.56));
        NN_RESULT_DO(Add("z", 7.89));

        const char eventId[] = "dummy";
        const nn::ApplicationId appId = { 0x1234567890ABCDEF };

        auto sysInfo = static_cast<nn::Bit8*>(buffer) + position;
        auto sysInfoSize = bufferSize - position;

        size_t actualSysInfoSize = 0;

        NN_RESULT_DO(nn::prepo::detail::service::core::SystemInfo::Collect(
            &actualSysInfoSize, sysInfo, sysInfoSize, appId, eventId));

        *outSize = position + actualSysInfoSize;

        NN_RESULT_SUCCESS;
    }

    nn::Result SetupReportBuffer(size_t* outSize, int* outCount, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        size_t position = 0;
        int count = 0;

        while (NN_STATIC_CONDITION(true))
        {
            size_t writeSize = 0;

            NN_RESULT_TRY(WriteReport(&writeSize, static_cast<nn::Bit8*>(buffer) + position, bufferSize - position))
                NN_RESULT_CATCH(nn::prepo::ResultOutOfResource)
                {
                    break;
                }
            NN_RESULT_END_TRY;

            position += writeSize;
            count++;
        }

        *outSize = position;
        *outCount = count;

        NN_RESULT_SUCCESS;
    }

    nn::prepo::detail::service::core::ReportFileManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::prepo::detail::service::core::ReportFileManager, s_Instance, ("prepo", true));
        return s_Instance;
    }
}

namespace
{
    nn::Bit8 g_ReportBuffer[nn::prepo::detail::ReportFileSizeMax];
    size_t g_ReportSize = 0;
    int g_ReportCount = 0;

    uint64_t g_IncrementalId = 0;

    char g_Path[nn::fs::EntryNameLengthMax + 1];
}

TEST(Initialize, Main)
{
    nn::time::Initialize();

    nn::prepo::detail::service::core::FileSystem::EnableTestMode();

    nn::prepo::detail::service::core::FileSystem::MountAll();

    nn::prepo::detail::service::core::SystemInfo::SetOperationMode(0);

    g_IncrementalId = nnt::prepo::GetRandom<uint64_t>();
    NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::WriteFile("prepo-sys:/id.bin", g_IncrementalId));

    NNT_ASSERT_RESULT_SUCCESS(SetupReportBuffer(&g_ReportSize, &g_ReportCount, g_ReportBuffer, sizeof(g_ReportBuffer)));
}

TEST(WriteFile, NoFile)
{
    nn::prepo::detail::service::core::FileSystem::ClearStorage("prepo");

    auto& manager = GetInstance();

    nn::prepo::detail::service::core::ReportFileManager::MakeReportFilePath(g_Path, sizeof(g_Path), "prepo", g_IncrementalId);
    NN_LOG("Target file: %s\n", g_Path);

    ASSERT_FALSE(nnt::prepo::FileExists(g_Path));

    nn::prepo::detail::service::ReportDataSummary summary;
    NNT_ASSERT_RESULT_SUCCESS(manager.WriteFile(&summary, g_ReportBuffer, g_ReportSize, g_ReportCount));

    ASSERT_TRUE(nnt::prepo::FileExists(g_Path));
}

TEST(WriteFile, FileExisting)
{
    nn::prepo::detail::service::core::FileSystem::ClearStorage("prepo");

    auto& manager = GetInstance();

    g_IncrementalId++;
    nn::prepo::detail::service::core::ReportFileManager::MakeReportFilePath(g_Path, sizeof(g_Path), "prepo", g_IncrementalId);
    NN_LOG("Target file: %s\n", g_Path);

    ASSERT_FALSE(nnt::prepo::FileExists(g_Path));

    nn::prepo::detail::service::ReportDataSummary summary;
    NNT_ASSERT_RESULT_SUCCESS(manager.WriteFile(&summary, g_ReportBuffer, g_ReportSize, g_ReportCount));

    ASSERT_TRUE(nnt::prepo::FileExists(g_Path));

    g_IncrementalId++;
    nn::prepo::detail::service::core::ReportFileManager::MakeReportFilePath(g_Path, sizeof(g_Path), "prepo", g_IncrementalId);
    NN_LOG("Target file: %s\n", g_Path);

    ASSERT_FALSE(nnt::prepo::FileExists(g_Path));

    NNT_ASSERT_RESULT_SUCCESS(manager.WriteFile(&summary, g_ReportBuffer, g_ReportSize, g_ReportCount));

    ASSERT_TRUE(nnt::prepo::FileExists(g_Path));
}

TEST(WriteFile, DeleteOldestFile)
{
    nn::prepo::detail::service::core::FileSystem::ClearStorage("prepo");

    auto& manager = GetInstance();

    const auto fileCount = static_cast<int>(nn::prepo::detail::service::core::FileSystem::GetTotalSpaceSize("prepo") / nn::prepo::detail::ReportFileSizeMax);

    g_IncrementalId++;
    nn::prepo::detail::service::core::ReportFileManager::MakeReportFilePath(g_Path, sizeof(g_Path), "prepo", g_IncrementalId);
    NN_LOG("Target file: %s\n", g_Path);

    ASSERT_FALSE(nnt::prepo::FileExists(g_Path));

    nn::prepo::detail::service::ReportDataSummary summary;
    NNT_ASSERT_RESULT_SUCCESS(manager.WriteFile(&summary, g_ReportBuffer, g_ReportSize, g_ReportCount));

    ASSERT_TRUE(nnt::prepo::FileExists(g_Path));

    ASSERT_EQ(summary.dataSize, 0);
    ASSERT_EQ(summary.restSize, 0);
    ASSERT_EQ(summary.restCount, 0);
    ASSERT_EQ(summary.deletedSize, 0);
    ASSERT_EQ(summary.deletedCount, 0);

    char firstFilePath[nn::fs::EntryNameLengthMax + 1] = {};
    ASSERT_GT(sizeof(firstFilePath), static_cast<size_t>(nn::util::Strlcpy(firstFilePath, g_Path, sizeof(firstFilePath))));

    for (int i = 0; i < fileCount - 2; i++)
    {
        g_IncrementalId++;
        nn::prepo::detail::service::core::ReportFileManager::MakeReportFilePath(g_Path, sizeof(g_Path), "prepo", g_IncrementalId);
        NN_LOG("Target file: %s\n", g_Path);

        ASSERT_FALSE(nnt::prepo::FileExists(g_Path));

        NNT_ASSERT_RESULT_SUCCESS(manager.WriteFile(&summary, g_ReportBuffer, g_ReportSize, g_ReportCount));

        ASSERT_TRUE(nnt::prepo::FileExists(g_Path));

        ASSERT_EQ(summary.dataSize, 0);
        ASSERT_EQ(summary.restSize, 0);
        ASSERT_EQ(summary.restCount, 0);
        ASSERT_EQ(summary.deletedSize, 0);
        ASSERT_EQ(summary.deletedCount, 0);
    }

    g_IncrementalId++;
    nn::prepo::detail::service::core::ReportFileManager::MakeReportFilePath(g_Path, sizeof(g_Path), "prepo", g_IncrementalId);
    NN_LOG("Target file: %s\n", g_Path);

    ASSERT_TRUE(nnt::prepo::FileExists(firstFilePath));
    ASSERT_FALSE(nnt::prepo::FileExists(g_Path));

    NNT_ASSERT_RESULT_SUCCESS(manager.WriteFile(&summary, g_ReportBuffer, g_ReportSize, g_ReportCount));

    ASSERT_FALSE(nnt::prepo::FileExists(firstFilePath));
    ASSERT_TRUE(nnt::prepo::FileExists(g_Path));

    ASSERT_EQ(summary.dataSize, 0);
    ASSERT_EQ(summary.restSize, 0);
    ASSERT_EQ(summary.restCount, 0);
    ASSERT_EQ(summary.deletedSize, g_ReportSize);
    ASSERT_EQ(summary.deletedCount, g_ReportCount);
}

TEST(Finalize, Main)
{
    nn::prepo::detail::service::core::FileSystem::ClearStorage("prepo");
}
