﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <utility>  // std::pair
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/gpio/gpio.h>
#include <nn/psc/psc_Types.h>
#include <nn/psc/psc_PmControlTypes.h>
#include <nn/cec/cec_Api.h>

// テスト対象
#include <nn/spsm/spsm_Api.h>
#include <nn/spsm/spsm_ShimInternal.h>
#include <nn/spsm/server/spsm_Server.h>
#include <nn/spsm/server/spsm_PowerStateMessageQueue.h>
#include <nn/spsm/observer/spsm_Observer.h>
#include <server/spsm_PowerStateMachine.h> // ライブラリ内部実装ヘッダ
#include <server/spsm_WakeReason.h> // ライブラリ内部実装ヘッダ
#include <server/spsm_Debug.h> // ライブラリ内部実装ヘッダ
#include <server/spsm_EventLog.h> // ライブラリ内部実装ヘッダ

#include "stubs/ApmStub.h"
#include "stubs/BgtcStub.h"
#include "stubs/GpioStub.h"
#include "stubs/IdleStub.h"
#include "stubs/PsmStub.h"
#include "stubs/TcStub.h"
#include "stubs/TcapStub.h"

// Windows 環境でテストプログラム内で spsm の各種スレッドを走らせる
namespace nn { namespace am {
    void StartSpsmPowerStateMachine() NN_NOEXCEPT;
    void StartSpsmEventHandler(nn::os::SystemEventType* pCecEventType) NN_NOEXCEPT;
    void StopSpsmEventHandler() NN_NOEXCEPT;
}}  // namespace nn::am

namespace
{
    using namespace ::nn::spsm;

    // Windows 環境でテストプログラム内で spsm の各種スレッドを走らせる
    class PowerStateInterfaceServerImpl
        : public nn::spsm::server::ServerInterfaceImpl
        , public nn::spsm::observer::ObserverInterfaceImpl
    {
    };
    nn::sf::UnmanagedServiceObject<nn::spsm::detail::IPowerStateInterface, PowerStateInterfaceServerImpl> g_PowerStateInterfaceServer;

    bool g_IsOneTimePowerStateManagerInitDone = false;
    nn::os::SystemEventType g_cecEventType;

    class PowerStateManagerInitializer
    {
    public:
        PowerStateManagerInitializer() NN_NOEXCEPT
        {
            // NN_LOG("PowerStateManagerInitializer\n");
            if ( !g_IsOneTimePowerStateManagerInitDone )
            {
                nn::spsm::InitializeWith(g_PowerStateInterfaceServer.GetShared());
                nn::am::StartSpsmPowerStateMachine();

                nn::cec::Initialize(&g_cecEventType);

                g_IsOneTimePowerStateManagerInitDone = true;
            }
            else
            {
                // ステートマシンを初期状態に（無理やり）戻すだけ
                server::PowerStateMachine::GetInstance().Initialize(server::InitializeMode_NoWait);
            }

            // こちらは毎回初期化
            nn::am::StartSpsmEventHandler(&g_cecEventType);
        }
        ~PowerStateManagerInitializer() NN_NOEXCEPT
        {
            // NN_LOG("~PowerStateManagerInitializer\n");
            nn::am::StopSpsmEventHandler();
        }
    };

    void DumpEventLog(const server::LogItemList& log)
    {
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
        for (const auto& item : log)
        {
            switch (item.type)
            {
                case server::LogType_PscControl:
                    NN_LOG("LogPscControl: pmState=%d, order=%s, flags=0x%x\n",
                        item.param.pscControl.pmState,
                        (item.param.pscControl.order == nn::psc::PmTransitionOrder_ToHigherPowerState) ? "ToHigher" : "ToLower",
                        item.param.pscControl.flags);
                    break;

                case server::LogType_PowerStateHandler:
                    NN_LOG("LogPowerStateHandler: state=%s, newDestState=%s, handlerType=%s, message=%s\n",
                        server::GetPowerStateNameString(item.param.powerStateHandler.state),
                        server::GetPowerStateNameString(item.param.powerStateHandler.newDestState),
                        server::GetPowerStateHandlerTypeNameString(item.param.powerStateHandler.handlerType),
                        server::GetPowerStateMessageNameString(item.param.powerStateHandler.message));
                    break;

                default:
                    NN_UNEXPECTED_DEFAULT;
            }
        }
#else
        NN_UNUSED(log);
#endif
    }

    // ウェイク後のシーケンス遷移の類型
    enum StateTransitionPatternOnWake
    {
        StateTransitionPatternOnWake_ResumeFullAwake,//!< FullAwake に戻ってくる
        StateTransitionPatternOnWake_PeriodicWake,   //!< SleepReady と MinimumAwake の間を行き来する
        StateTransitionPatternOnWake_PeriodicWakeForBackgroundTask,   //!< SleepReady と MinimumAwake と MinimumAwakeForBackgroundTask の間を行き来する
        StateTransitionPatternOnWake_LowBattery,     //!< FullAwake に行こうとするが「要充電画面」に落ちて SleepReady に戻る
        StateTransitionPatternOnWake_Charging,       //!< 「充電中画面」に落ちて SleepReady に戻る
        StateTransitionPatternOnWake_Shutdown,       //!< ShutdownReady に飛ぶ
    };

    void TestStateTransitionPatternOnWake(StateTransitionPatternOnWake pattern, std::function<void(void)> fullawakeRestoreMethod)
    {
        nn::os::SystemEvent* pAwakeEvent = nullptr;
        server::PowerStateMachine::GetInstance().EnterSleep(&pAwakeEvent);

        switch ( pattern )
        {
            case StateTransitionPatternOnWake_ResumeFullAwake:
            {
                // 戻ってこないと永久ブロックで失敗
                pAwakeEvent->Wait();
                EXPECT_EQ(PowerState_FullAwake, server::PowerStateMachine::GetInstance().GetCurrentState());
                break;
            }
            case StateTransitionPatternOnWake_PeriodicWake:
            case StateTransitionPatternOnWake_Charging:
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1)); // TORIAEZU: FullAwake は脱していることの期待

                // SleepReady と MinimumAwake の間を行き来していることをテスト
                const int CheckCount = 50;
                for ( int i = 0; i < CheckCount; ++i )
                {
                    PowerState curr = server::PowerStateMachine::GetInstance().GetCurrentState();
                    EXPECT_TRUE(curr == PowerState_MinimumAwake || curr == PowerState_SleepReady);
                    if ( !(curr == PowerState_MinimumAwake || curr == PowerState_SleepReady) )
                    {
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
                        NN_LOG("curr was %s\n", server::GetPowerStateNameString(curr));
#endif
                    }
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
                }

                fullawakeRestoreMethod();
                pAwakeEvent->Wait();
                EXPECT_EQ(PowerState_FullAwake, server::PowerStateMachine::GetInstance().GetCurrentState());
                break;
            }
            case StateTransitionPatternOnWake_PeriodicWakeForBackgroundTask:
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1)); // TORIAEZU: FullAwake は脱していることの期待

                // SleepReady と MinimumAwake と MinimumAwakeForBackgroundTask の間を行き来していることをテスト
                const int CheckCount = 50;
                for ( int i = 0; i < CheckCount; ++i )
                {
                    PowerState curr = server::PowerStateMachine::GetInstance().GetCurrentState();
                    EXPECT_TRUE(curr == PowerState_MinimumAwake || curr == PowerState_MinimumAwakeForBackgroundTask || curr == PowerState_SleepReady);
                    if ( !(curr == PowerState_MinimumAwake || curr == PowerState_MinimumAwakeForBackgroundTask || curr == PowerState_SleepReady) )
                    {
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
                        NN_LOG("curr was %s\n", server::GetPowerStateNameString(curr));
#endif
                    }
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
                }

                fullawakeRestoreMethod();
                pAwakeEvent->Wait();
                EXPECT_EQ(PowerState_FullAwake, server::PowerStateMachine::GetInstance().GetCurrentState());
                break;
            }
            case StateTransitionPatternOnWake_LowBattery:
            {
                pAwakeEvent->Wait();
                EXPECT_EQ(PowerState_MinimumAwakeForLowBatterySign, server::PowerStateMachine::GetInstance().GetCurrentState());
                fullawakeRestoreMethod();
                server::PowerStateMachine::GetInstance().EnterSleep(&pAwakeEvent);
                pAwakeEvent->Wait();
                EXPECT_EQ(PowerState_FullAwake, server::PowerStateMachine::GetInstance().GetCurrentState());
                break;
            }
            case StateTransitionPatternOnWake_Shutdown:
            {
                nn::TimeSpan elapsed = 0;
                PowerState curr;
                while ( (curr = server::PowerStateMachine::GetInstance().GetCurrentState()) != PowerState_ShutdownReady )
                {
                    const auto PollingInterval = nn::TimeSpan::FromMilliSeconds(50);
                    if ( elapsed > nn::TimeSpan::FromSeconds(3) )
                    {
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
                        NN_LOG("Timed out: Power state did not change to ShutdownReady. Now it is %s\n", server::GetPowerStateNameString(curr));
#endif
                        ADD_FAILURE();
                        break;
                    }
                    nn::os::SleepThread(PollingInterval);
                    elapsed += PollingInterval;
                }
                break;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

} // anonymous namespace

// ----------------------------------------------------------------------------
// Applet Sleep Wake Interface Test
// ----------------------------------------------------------------------------

// spsm::SleepSystemAndWaitAwake() を呼び出し、
// spsm のステートマシンが想定通りの遷移をすることをテスト
TEST( AppletSleepWakeInterfaceTest, CheckEventLog )
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    // bluetooth 割り込みで起きたように見せかける
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlag::BluetoothActivityDetect::Mask);

    server::ResetLog(); // 内部ログを空にする
    nn::spsm::SleepSystemAndWaitAwake();

    // スリープ⇒フルウェイクまでにステートマシン上で起こるべき出来事の一部始終
    const server::LogItemList expectedEventLog =
    {
        // スリープ要求の受信をした
        server::LogItem(PowerState_FullAwake, PowerState_SleepReady, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_SleepRequested),
        // FullAwake を出た
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に MinimumAwake 遷移要求をした
        server::LogItem(nn::psc::PmState_MinimumAwake, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),
        // MinimumAwake に入った
        server::LogItem(PowerState_MinimumAwake, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // MinimumAwake を出た
        server::LogItem(PowerState_MinimumAwake, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に SleepReady 遷移要求をした
        server::LogItem(nn::psc::PmState_SleepReady, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),
        // SleepReady に入った
        server::LogItem(PowerState_SleepReady, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に EssentialServicesSleepReady 遷移要求をした
        server::LogItem(nn::psc::PmState_EssentialServicesSleepReady, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),

        // ここでスリープ
        // 以下、起床後

        // PSC 経由でシステムモジュール群に EssentialServicesAwake 遷移要求をした
        server::LogItem(nn::psc::PmState_EssentialServicesAwake, nn::psc::PmTransitionOrder_ToHigherPowerState, nn::psc::MakeNoPmFlags()),
        // 以上が SleepReady の OnDestination のハンドリングで、これを出た。目的地を一旦 MinimumAwake に
        server::LogItem(PowerState_SleepReady, PowerState_MinimumAwake, server::PowerStateHandlerType_OnDestination, server::PowerStateMessage_None),
        // SleepReady を出た
        server::LogItem(PowerState_SleepReady, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に MinimumAwake 遷移要求をした
        server::LogItem(nn::psc::PmState_MinimumAwake, nn::psc::PmTransitionOrder_ToHigherPowerState, nn::psc::MakeNoPmFlags()),
        // MinimumAwake の Entry の中で、行先が FullAwake になった
        server::LogItem(PowerState_MinimumAwake, PowerState_FullAwake, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // MinimumAwake を出た
        server::LogItem(PowerState_MinimumAwake, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に FullAwake 遷移要求をした
        server::LogItem(nn::psc::PmState_FullAwake, nn::psc::PmTransitionOrder_ToHigherPowerState, nn::psc::MakeNoPmFlags()),
        // FullAwakeに入った
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // FullAwake の OnDestination を出た
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_OnDestination, server::PowerStateMessage_None),
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlagSet()); // 終わったら空に

    nn::spsm::Finalize();
}

// ----------------------------------------------------------------------------
// Put error state
// ----------------------------------------------------------------------------

TEST(PutErrorState, CheckEventLog)
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    nn::spsm::PutErrorState();

    // Signal がハンドリングされるまで待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    const server::LogItemList expectedEventLog =
    {
        // エラーステート遷移要求の受信をした
        server::LogItem(PowerState_FullAwake, PowerState_Error, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_PutErrorRequested),
        // FullAwake を出た
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // Error に入った
        server::LogItem(PowerState_Error, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        server::LogItem(PowerState_Error, PowerState_None, server::PowerStateHandlerType_OnDestination, server::PowerStateMessage_None),
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    nn::spsm::Finalize();
}

// ----------------------------------------------------------------------------
// IDLE Sleep Interface Test
// ----------------------------------------------------------------------------

// idle の AutoPowerDown イベントをシグナルし、
// spsm のステートマシンが想定通りの遷移をすることをテスト
TEST( IdleSleepWakeInterfaceTest, CheckEventLog )
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    nn::idle::SignalAutoPowerDownEvent();

    // Signal がハンドリングされるまで待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    const server::LogItemList expectedEventLog =
    {
        // スリープ要求の受信をした
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_EventAutoPowerDownTimerExpired),
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    nn::spsm::Finalize();
}

// ----------------------------------------------------------------------------
// TC Sleep Interface Test
// ----------------------------------------------------------------------------

// tc の EventTarget_AbnormalTemperature を呼び出し、
// spsm のステートマシンが想定通りの遷移をすることをテスト
TEST( TcSleepWakeInterfaceTest, CheckEventLog )
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    nn::tc::SignalThermalEvent(nn::tc::EventTarget_AbnormalTemperature);

    // Signal がハンドリングされるまで待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    const server::LogItemList expectedEventLog =
    {
        // スリープ要求の受信をした
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_EventSleepRequiredByHighTemperature),
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    nn::spsm::Finalize();
}

// ----------------------------------------------------------------------------
// TCAP Sleep Interface Test
// ----------------------------------------------------------------------------

// tcap の ContinuousHighSkinTemperature を呼び出し、
// spsm のステートマシンが想定通りの遷移をすることをテスト
TEST( TcapSleepWakeInterfaceTest, CheckEventLog )
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    nn::tcap::SignalContinuousHighSkinTemperatureEvent();

    // Signal がハンドリングされるまで待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    const server::LogItemList expectedEventLog =
    {
        // スリープ要求の受信をした
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_EventSleepRequiredByHighTemperature),
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    nn::spsm::Finalize();
}

// ----------------------------------------------------------------------------
// APM Sleep Interface Test
// ----------------------------------------------------------------------------

// apm の EventTarget_SleepRequiredByLowVoltage を呼び出し、
// spsm のステートマシンが想定通りの遷移をすることをテスト
TEST( ApmSleepWakeInterfaceTest, CheckEventLog )
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    nn::apm::SignalPerformanceEvent(nn::apm::EventTarget_SleepRequiredByLowVoltage);

    // Signal がハンドリングされるまで待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    const server::LogItemList expectedEventLog =
    {
        // スリープ要求の受信をした
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_EventSleepRequiredByLowBattery),
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    nn::spsm::Finalize();
}

// ----------------------------------------------------------------------------
// PSM Sleep Interface Test
// ----------------------------------------------------------------------------

// psm の StateChanged イベントをシグナルし、
// spsm のステートマシンが想定通りの遷移をすることをテスト
TEST( PsmSleepWakeInterfaceTest, CheckEventLog )
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_SleepRequired);
    NN_UTIL_SCOPE_EXIT{ nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_Good); };
    nnt::psm::SignalBatteryVoltageStateChangeEvent();

    // Signal がハンドリングされるまで待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    const server::LogItemList expectedEventLog =
    {
        // スリープ要求の受信をした
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_EventSleepRequiredByLowBattery),
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    nn::spsm::Finalize();
}

// ----------------------------------------------------------------------------
// PSM Shutdown Interface Test
// ----------------------------------------------------------------------------

// psm の StateChanged イベントをシグナルし、
// spsm のステートマシンが想定通りの遷移をすることをテスト
TEST( PsmShutdownInterfaceTest, CheckEventLog )
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_ShutdownRequired);
    NN_UTIL_SCOPE_EXIT{ nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_Good); };
    nnt::psm::SignalBatteryVoltageStateChangeEvent();

    // Signal がハンドリングされるまで待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    const server::LogItemList expectedEventLog =
    {
        // シャットダウンの受信をした
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_EventShutdownRequiredByVeryLowBattery),
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    nn::spsm::Finalize();
}

// ----------------------------------------------------------------------------
// Applet Shutdown Interface Test
// ----------------------------------------------------------------------------

namespace nn { namespace spsm {
    // Applet 向けの API 自体は no return なので、その中で使われている内部関数を呼び出してテストする
    void StartShutdown(bool) NN_NOEXCEPT;
}} // namespace nn::spsm

class AppletShutdownInterfaceTest : public ::testing::TestWithParam<bool>
{};

// spsm::RebootSystem() / spsm::ShutdownSystem() を呼び出し、
// spsm のステートマシンが想定通りの遷移をすることをテスト
TEST_P(AppletShutdownInterfaceTest, CheckEventLog)
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    auto reboot = GetParam();
    nn::spsm::StartShutdown(reboot);

    // TORIAEZU
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // シャットダウンまでにステートマシン上で起こるべき出来事の一部始終
    const server::LogItemList expectedEventLog =
    {
        // シャットダウン or リブート要求の受信をした
        server::LogItem(PowerState_FullAwake, PowerState_ShutdownReady, server::PowerStateHandlerType_OnMessage, (reboot) ? server::PowerStateMessage_RebootRequested : server::PowerStateMessage_ShutdownRequested),
        // FullAwake を出た
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に ShutdownReady 遷移要求をした
        server::LogItem(nn::psc::PmState_ShutdownReady, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),
        // ShutdownReady に入った
        server::LogItem(PowerState_ShutdownReady, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // ShutdownReady の OnDestination を出た (Windows 版のみ)
        server::LogItem(PowerState_ShutdownReady, PowerState_None, server::PowerStateHandlerType_OnDestination, server::PowerStateMessage_None),
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    nn::spsm::Finalize();
}

INSTANTIATE_TEST_CASE_P(
    AppletShutdownInterfaceTestInst,
    AppletShutdownInterfaceTest,
    ::testing::Values(false, true)
);

// ----------------------------------------------------------------------------
// Wake Low Battery Sequence Test
// ----------------------------------------------------------------------------

// 低電池残量時にスリープ復帰しようとした際の判定のテスト
TEST(WakeLowBatterySequenceTest, Basic)
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    // bluetooth 割り込みで起きたように見せかける
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlag::BluetoothActivityDetect::Mask);

    // ギリギリ足りない残量にする
    nnt::psm::SetRawBatteryChargePercentage(2.99f);

    auto fullawakeRestoreMethod = []()
    {
        // バッテリ残量を 3% ジャストにすると FullAwake へ
        nnt::psm::SetRawBatteryChargePercentage(3.0f);
    };
    TestStateTransitionPatternOnWake(StateTransitionPatternOnWake_LowBattery, fullawakeRestoreMethod);

    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlagSet()); // 終わったら空に

    nn::spsm::Finalize();
}

// ----------------------------------------------------------------------------
// Wake Very Low Battery Sequence Test
// ----------------------------------------------------------------------------

// スリープ維持不可能な電圧レベルで起床した際の判定のテスト
TEST(WakeVeryLowBatterySequenceTest, Basic)
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    // Gpio のフラグで残量 IC からの割り込みと伝える
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlagSet());
    nnt::gpio::SetWakeEventActiveFlag({ nn::gpio::GpioPadName_BattMgicIrq });

    // アウトな電圧をセットする
    nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_ShutdownRequired);
    NN_UTIL_SCOPE_EXIT{ nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_Good); };

    TestStateTransitionPatternOnWake(StateTransitionPatternOnWake_Shutdown, nullptr);

    nnt::gpio::SetWakeEventActiveFlag({}); // 終わったら空に
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlagSet()); // 終わったら空に

    nn::spsm::Finalize();
}

// ----------------------------------------------------------------------------
// Wake Event Test
// ----------------------------------------------------------------------------

class WakeEventTest : public ::testing::TestWithParam<std::pair<WakeReasonFlagSet, StateTransitionPatternOnWake>>
{};

TEST_P( WakeEventTest, CheckState )
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    server::SetEmulatedWakeReasonFlagSet(GetParam().first);

    auto fullawakeRestoreMethod = []()
    {
        // フラグを bluetooth 割り込みに変更すると FullAwake へ
        server::SetEmulatedWakeReasonFlagSet(WakeReasonFlag::BluetoothActivityDetect::Mask);
    };
    TestStateTransitionPatternOnWake(GetParam().second, fullawakeRestoreMethod);

    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlagSet()); // 終わったら空に

    nn::spsm::Finalize();
}

INSTANTIATE_TEST_CASE_P(
    WakeEventTestInst,
    WakeEventTest,
    ::testing::Values(

        //------------------------------
        // 単体要因ごとの行先

        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::PowerButtonPressed::Mask)),
            StateTransitionPatternOnWake_ResumeFullAwake
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::AcOk::Mask)),
            StateTransitionPatternOnWake_ResumeFullAwake // ドックインで起きた場合、製品仕様上にはスリープに戻るが、充電中画面表示は omm が行ってから戻す
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::RtcForBackgroundTask::Mask)),
            StateTransitionPatternOnWake_PeriodicWakeForBackgroundTask
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::RtcForFullWakeup::Mask)),
            StateTransitionPatternOnWake_ResumeFullAwake
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::BatteryChargerUpdateRequired::Mask)),
            StateTransitionPatternOnWake_PeriodicWake
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::BatteryLevelTooLow::Mask)),
            StateTransitionPatternOnWake_Shutdown
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::BluetoothActivityDetect::Mask)),
            StateTransitionPatternOnWake_ResumeFullAwake
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::WiFiActivityDetect::Mask)),
            StateTransitionPatternOnWake_PeriodicWakeForBackgroundTask
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::SdCardDetect::Mask)),
            StateTransitionPatternOnWake_ResumeFullAwake
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::GameCardDetect::Mask)),
            StateTransitionPatternOnWake_ResumeFullAwake
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::ControllerDetectL::Mask)),
            StateTransitionPatternOnWake_PeriodicWake
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::ControllerDetectR::Mask)),
            StateTransitionPatternOnWake_PeriodicWake
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::ControllerButtonL::Mask)),
            StateTransitionPatternOnWake_ResumeFullAwake
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::ControllerButtonR::Mask)),
            StateTransitionPatternOnWake_ResumeFullAwake
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::ControllerBatteryLevelChangeL::Mask)),
            StateTransitionPatternOnWake_PeriodicWake
            ),
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(WakeReasonFlag::ControllerBatteryLevelChangeR::Mask)),
            StateTransitionPatternOnWake_PeriodicWake
            ),

        //------------------------------
        // 複合要因での優先度判定

        // SleepReady に戻る要因の集合
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(
                WakeReasonFlag::BatteryChargerUpdateRequired::Mask |
                WakeReasonFlag::ControllerDetectL::Mask |
                WakeReasonFlag::ControllerDetectR::Mask |
                WakeReasonFlag::ControllerBatteryLevelChangeL::Mask |
                WakeReasonFlag::ControllerBatteryLevelChangeR::Mask
            )),
            StateTransitionPatternOnWake_PeriodicWake
            ),

        // MinimumAwakeForBackgroundTask に行く要因が一つだけ加わればそれが優先
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(
                WakeReasonFlag::WiFiActivityDetect::Mask | // これ
                WakeReasonFlag::BatteryChargerUpdateRequired::Mask |
                WakeReasonFlag::ControllerDetectL::Mask |
                WakeReasonFlag::ControllerDetectR::Mask |
                WakeReasonFlag::ControllerBatteryLevelChangeL::Mask |
                WakeReasonFlag::ControllerBatteryLevelChangeR::Mask
            )),
            StateTransitionPatternOnWake_PeriodicWakeForBackgroundTask
            ),

        // FullAwake に行く要因が一つだけ加わればそれが優先
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(
                WakeReasonFlag::PowerButtonPressed::Mask | // これ
                WakeReasonFlag::WiFiActivityDetect::Mask |
                WakeReasonFlag::BatteryChargerUpdateRequired::Mask |
                WakeReasonFlag::ControllerDetectL::Mask |
                WakeReasonFlag::ControllerDetectR::Mask |
                WakeReasonFlag::ControllerBatteryLevelChangeL::Mask |
                WakeReasonFlag::ControllerBatteryLevelChangeR::Mask
            )),
            StateTransitionPatternOnWake_ResumeFullAwake
            ),

        // BatteryLevelTooLow が立ってたら最優先でシャットダウン
        std::make_pair<WakeReasonFlagSet, StateTransitionPatternOnWake>(
            std::move(WakeReasonFlagSet(
                WakeReasonFlag::PowerButtonPressed::Mask |
                WakeReasonFlag::AcOk::Mask |
                WakeReasonFlag::RtcForBackgroundTask::Mask |
                WakeReasonFlag::RtcForFullWakeup::Mask |
                WakeReasonFlag::BatteryChargerUpdateRequired::Mask |
                WakeReasonFlag::BatteryLevelTooLow::Mask | // これ
                WakeReasonFlag::BluetoothActivityDetect::Mask |
                WakeReasonFlag::WiFiActivityDetect::Mask |
                WakeReasonFlag::SdCardDetect::Mask |
                WakeReasonFlag::GameCardDetect::Mask |
                WakeReasonFlag::ControllerDetectL::Mask |
                WakeReasonFlag::ControllerDetectR::Mask |
                WakeReasonFlag::ControllerButtonL::Mask |
                WakeReasonFlag::ControllerButtonR::Mask |
                WakeReasonFlag::ControllerBatteryLevelChangeL::Mask |
                WakeReasonFlag::ControllerBatteryLevelChangeR::Mask
            )),
            StateTransitionPatternOnWake_Shutdown
            )
        )
);

// ----------------------------------------------------------------------------
// MinimumAwake for Background Task Test
// ----------------------------------------------------------------------------

// バックグラウンドタスク実行の基本的な遷移のテスト
TEST(MinimumAwakeForBackgroundTaskTest, Basic)
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    // バックグラウンドタスクがある状態にする
    nnt::bgtc::SetTaskAvailable(true);
    NN_UTIL_SCOPE_EXIT{ nnt::bgtc::SetTaskAvailable(false); };

    // RTC Alarm 割り込みで起きたように見せかけつつスリープに落とす
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlag::RtcForBackgroundTask::Mask);
    nn::os::SystemEvent* pAwakeEvent = nullptr;
    server::PowerStateMachine::GetInstance().EnterSleep(&pAwakeEvent);

    // しばらく待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // FullAwake に行くウェイク要因に切り替え
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlag::BluetoothActivityDetect::Mask);

    // バックグラウンドタスクが終了すると SleepReady に戻ったあと FullAwake に起床
    nnt::bgtc::SetTaskAvailable(false);
    pAwakeEvent->Wait();

    // (SIGLO-76706) この時点では最後のログ（FullAwake の OnDestination を出た）が記録されていることは
    // ギリギリ保証されていないので、少しウェイトしてから比較する
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(200));

    // ここまでにステートマシン上で起こるべき出来事の一部始終
    const server::LogItemList expectedEventLog =
    {
        // スリープ要求の受信をした
        server::LogItem(PowerState_FullAwake, PowerState_SleepReady, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_SleepRequested),
        // FullAwake を出た
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に MinimumAwake 遷移要求をした
        server::LogItem(nn::psc::PmState_MinimumAwake, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),
        // MinimumAwake に入った
        server::LogItem(PowerState_MinimumAwake, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // MinimumAwake を出た
        server::LogItem(PowerState_MinimumAwake, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に SleepReady 遷移要求をした
        server::LogItem(nn::psc::PmState_SleepReady, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),
        // SleepReady に入った
        server::LogItem(PowerState_SleepReady, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に EssentialServicesSleepReady 遷移要求をした
        server::LogItem(nn::psc::PmState_EssentialServicesSleepReady, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),

        // ここでスリープ
        // 以下、起床後

        // PSC 経由でシステムモジュール群に EssentialServicesAwake 遷移要求をした
        server::LogItem(nn::psc::PmState_EssentialServicesAwake, nn::psc::PmTransitionOrder_ToHigherPowerState, nn::psc::MakeNoPmFlags()),
        // 以上が SleepReady の OnDestination のハンドリングで、これを出た。目的地を一旦 MinimumAwake に
        server::LogItem(PowerState_SleepReady, PowerState_MinimumAwake, server::PowerStateHandlerType_OnDestination, server::PowerStateMessage_None),
        // SleepReady を出た
        server::LogItem(PowerState_SleepReady, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に MinimumAwake遷移要求をした
        server::LogItem(nn::psc::PmState_MinimumAwake, nn::psc::PmTransitionOrder_ToHigherPowerState, nn::psc::MakeNoPmFlags()),
        // MinimumAwake の Entry の中で、行先が MinimumAwakeForBackgroundTask になった
        server::LogItem(PowerState_MinimumAwake, PowerState_MinimumAwakeForBackgroundTask, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // MinimumAwake を出た
        server::LogItem(PowerState_MinimumAwake, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // MinimumAwakeForBackgroundTask に入った
        server::LogItem(PowerState_MinimumAwakeForBackgroundTask, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // MinimumAwakeForBackgroundTask の OnDestination を出た
        server::LogItem(PowerState_MinimumAwakeForBackgroundTask, PowerState_None, server::PowerStateHandlerType_OnDestination, server::PowerStateMessage_None),

        // ここでバックグラウンドタスク実行
        // 以下、バックグラウンドタスク終了後

        // バックグラウンドタスク完了メッセージを受信し、目的地を SleepReady に
        server::LogItem(PowerState_MinimumAwakeForBackgroundTask, PowerState_SleepReady, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_EventBackgroundTaskDone),
        // MinimumAwakeForBackgroundTask を出た
        server::LogItem(PowerState_MinimumAwakeForBackgroundTask, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に SleepReady 遷移要求をした
        server::LogItem(nn::psc::PmState_SleepReady, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),
        // SleepReady に入った
        server::LogItem(PowerState_SleepReady, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に EssentialServicesSleepReady 遷移要求をした
        server::LogItem(nn::psc::PmState_EssentialServicesSleepReady, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),

        // ここでスリープ
        // 以下、起床後

        // PSC 経由でシステムモジュール群に EssentialServicesAwake 遷移要求をした
        server::LogItem(nn::psc::PmState_EssentialServicesAwake, nn::psc::PmTransitionOrder_ToHigherPowerState, nn::psc::MakeNoPmFlags()),
        // 以上が SleepReady の OnDestination のハンドリングで、これを出た。目的地を一旦 MinimumAwake に
        server::LogItem(PowerState_SleepReady, PowerState_MinimumAwake, server::PowerStateHandlerType_OnDestination, server::PowerStateMessage_None),
        // SleepReady を出た
        server::LogItem(PowerState_SleepReady, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に MinimumAwake 遷移要求をした
        server::LogItem(nn::psc::PmState_MinimumAwake, nn::psc::PmTransitionOrder_ToHigherPowerState, nn::psc::MakeNoPmFlags()),
        // MinimumAwake の Entry の中で、行先が FullAwake になった
        server::LogItem(PowerState_MinimumAwake, PowerState_FullAwake, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // MinimumAwake を出た
        server::LogItem(PowerState_MinimumAwake, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に FullAwake 遷移要求をした
        server::LogItem(nn::psc::PmState_FullAwake, nn::psc::PmTransitionOrder_ToHigherPowerState, nn::psc::MakeNoPmFlags()),
        // FullAwakeに入った
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // FullAwake の OnDestination を出た
        server::LogItem(PowerState_FullAwake, PowerState_None, server::PowerStateHandlerType_OnDestination, server::PowerStateMessage_None)
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlagSet()); // 終わったら空に

    nn::spsm::Finalize();
}

// バックグラウンドタスク実行中からスリープ復帰しようとしたが電池不足だった場合の判定のテスト
TEST(MinimumAwakeForBackgroundTaskTest, LowBatteryForFullAwake)
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    // BGNUP がある状態にする
    nnt::bgtc::SetTaskAvailable(true);
    NN_UTIL_SCOPE_EXIT{ nnt::bgtc::SetTaskAvailable(false); };

    // ギリギリ足りない残量にする
    nnt::psm::SetRawBatteryChargePercentage(2.99f);

    // RTC Alarm 割り込みで起きたように見せかけつつスリープに落とす
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlag::RtcForBackgroundTask::Mask);
    nn::os::SystemEvent* pAwakeEvent = nullptr;
    server::PowerStateMachine::GetInstance().EnterSleep(&pAwakeEvent);

    // しばらくすると MinimumAwakeForBackgroundTask になっている
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    EXPECT_EQ(PowerState_MinimumAwakeForBackgroundTask, server::PowerStateMachine::GetInstance().GetCurrentState());

    // 電源ボタン押し始めイベントを送ると LowBattery ステートへ
    server::PowerStateMachine::GetInstance().GetPowerStateMessageQueue()->Enqueue(AddTimeStamp(server::PowerStateMessage_EventPowerButtonStartedPressing));
    pAwakeEvent->Wait();
    EXPECT_EQ(PowerState_MinimumAwakeForLowBatterySign, server::PowerStateMachine::GetInstance().GetCurrentState());

    // もう一度 RTC Alarm で起床するようにしつつ寝かせ、 MinimumAwakeForBackgroundTask へ
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlag::RtcForBackgroundTask::Mask);
    server::PowerStateMachine::GetInstance().EnterSleep(&pAwakeEvent);
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    EXPECT_EQ(PowerState_MinimumAwakeForBackgroundTask, server::PowerStateMachine::GetInstance().GetCurrentState());

    // バッテリ残量を 3% ジャストに
    nnt::psm::SetRawBatteryChargePercentage(3.0f);

    // 電源ボタン押し始めイベントを送ると今度は FullAwake
    server::PowerStateMachine::GetInstance().GetPowerStateMessageQueue()->Enqueue(AddTimeStamp(server::PowerStateMessage_EventPowerButtonStartedPressing));
    pAwakeEvent->Wait();
    EXPECT_EQ(PowerState_FullAwake, server::PowerStateMachine::GetInstance().GetCurrentState());

    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlagSet()); // 終わったら空に

    nn::spsm::Finalize();
}

// バックグラウンドタスクを開始しようとしたがスリープ必須レベルの電池残量だった場合の判定のテスト
TEST(MinimumAwakeForBackgroundTaskTest, SleepRequiredBeforeStartingBackgroundTask)
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    server::ResetLog(); // 内部ログを空にする

    // BGNUP がある状態にする
    nnt::bgtc::SetTaskAvailable(true);
    NN_UTIL_SCOPE_EXIT{ nnt::bgtc::SetTaskAvailable(false); };

    // 要スリープな残量にする
    nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_SleepRequired);
    NN_UTIL_SCOPE_EXIT { nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_Good); };

    // RTC Alarm 割り込みで起きたように見せかけつつスリープに落とす
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlag::RtcForBackgroundTask::Mask);
    NN_UTIL_SCOPE_EXIT{ server::SetEmulatedWakeReasonFlagSet(WakeReasonFlagSet()); };
    nn::os::SystemEvent* pAwakeEvent = nullptr;
    server::PowerStateMachine::GetInstance().EnterSleep(&pAwakeEvent);

    // RTC のため一度は MinimumAwakeForBackgroundTask に行くが、すぐに SleepReady <-> MinimumAwake の往復に落ちているはず
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlagSet());
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    const int CheckCount = 50;
    for ( int i = 0; i < CheckCount; ++i )
    {
        PowerState curr = server::PowerStateMachine::GetInstance().GetCurrentState();
        EXPECT_TRUE(curr == PowerState_MinimumAwake || curr == PowerState_SleepReady);
        if ( !(curr == PowerState_MinimumAwake || curr == PowerState_SleepReady) )
        {
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
            NN_LOG("curr was %s\n", server::GetPowerStateNameString(curr));
#endif
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
    }

    // 電源ボタン押し始めイベントを送ると今度は FullAwake
    server::PowerStateMachine::GetInstance().GetPowerStateMessageQueue()->Enqueue(AddTimeStamp(server::PowerStateMessage_EventPowerButtonStartedPressing));
    pAwakeEvent->Wait();
    EXPECT_EQ(PowerState_FullAwake, server::PowerStateMachine::GetInstance().GetCurrentState());

    nn::spsm::Finalize();
}

// バックグラウンドタスク実行中に電池が切れた場合にスリープすることのテスト
TEST(MinimumAwakeForBackgroundTaskTest, SleepRequiredDuringBackgroundTask)
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    // バックグラウンドタスクがある状態にする
    nnt::bgtc::SetTaskAvailable(true);
    NN_UTIL_SCOPE_EXIT{ nnt::bgtc::SetTaskAvailable(false); };

    // RTC Alarm 割り込みで起きたように見せかけつつスリープに落とす
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlag::RtcForBackgroundTask::Mask);
    nn::os::SystemEvent* pAwakeEvent = nullptr;
    server::PowerStateMachine::GetInstance().EnterSleep(&pAwakeEvent);

    // しばらくすると MinimumAwakeForBackgroundTask になっている
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    EXPECT_EQ(PowerState_MinimumAwakeForBackgroundTask, server::PowerStateMachine::GetInstance().GetCurrentState());

    // ここまでの内部ログを空にする
    server::ResetLog();

    // もう一度 RTC Alarm で起床するようにしつつ電池切れイベントを送ると SleepReady -> MinimumAwakeForBackgroundTask
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlag::RtcForBackgroundTask::Mask);
    server::PowerStateMachine::GetInstance().GetPowerStateMessageQueue()->Enqueue(AddTimeStamp(server::PowerStateMessage_EventSleepRequiredByLowBattery));
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // ここまでにステートマシン上で起こるべき出来事の一部始終
    const server::LogItemList expectedEventLog =
    {
        // 電池切れメッセージを受信し、目的地を SleepReady に
        server::LogItem(PowerState_MinimumAwakeForBackgroundTask, PowerState_SleepReady, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_EventSleepRequiredByLowBattery),
        // MinimumAwakeForBackgroundTask を出た
        server::LogItem(PowerState_MinimumAwakeForBackgroundTask, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に SleepReady 遷移要求をした
        server::LogItem(nn::psc::PmState_SleepReady, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),
        // SleepReady に入った
        server::LogItem(PowerState_SleepReady, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に EssentialServicesSleepReady 遷移要求をした
        server::LogItem(nn::psc::PmState_EssentialServicesSleepReady, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),

        // ここでスリープ
        // 以下、起床後

        // PSC 経由でシステムモジュール群に EssentialServicesAwake 遷移要求をした
        server::LogItem(nn::psc::PmState_EssentialServicesAwake, nn::psc::PmTransitionOrder_ToHigherPowerState, nn::psc::MakeNoPmFlags()),
        // 以上が SleepReady の OnDestination のハンドリングで、これを出た。目的地を一旦 MinimumAwake に
        server::LogItem(PowerState_SleepReady, PowerState_MinimumAwake, server::PowerStateHandlerType_OnDestination, server::PowerStateMessage_None),
        // SleepReady を出た
        server::LogItem(PowerState_SleepReady, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に MinimumAwake 遷移要求をした
        server::LogItem(nn::psc::PmState_MinimumAwake, nn::psc::PmTransitionOrder_ToHigherPowerState, nn::psc::MakeNoPmFlags()),
        // MinimumAwake の Entry の中で、行先が MinimumAwakeForBackgroundTask になった
        server::LogItem(PowerState_MinimumAwake, PowerState_MinimumAwakeForBackgroundTask, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // MinimumAwake を出た
        server::LogItem(PowerState_MinimumAwake, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // MinimumAwakeForBackgroundTask に入った
        server::LogItem(PowerState_MinimumAwakeForBackgroundTask, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // MinimumAwakeForBackgroundTask の OnDestination を出た
        server::LogItem(PowerState_MinimumAwakeForBackgroundTask, PowerState_None, server::PowerStateHandlerType_OnDestination, server::PowerStateMessage_None),
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    // 適当なイベントで FullAwake へ
    server::PowerStateMachine::GetInstance().GetPowerStateMessageQueue()->Enqueue(AddTimeStamp(server::PowerStateMessage_EventSdCardStateChanged));
    pAwakeEvent->Wait();
    EXPECT_EQ(PowerState_FullAwake, server::PowerStateMachine::GetInstance().GetCurrentState());

    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlagSet()); // 終わったら空に

    nn::spsm::Finalize();
}

// バックグラウンドタスク実行中に完全に電池が切れた場合にシャットダウンされることのテスト
// （通常 SleepRequired が先に来るので実機では起こらないはず）
TEST(MinimumAwakeForBackgroundTaskTest, ShutdownRequiredDuringBackgroundTask)
{
    PowerStateManagerInitializer powerStateManagerInitializer;

    // テスト開始状態になるのを待つ
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    } while ( GetCurrentState() != PowerState_FullAwake );

    nn::spsm::Initialize();

    // バックグラウンドタスクがある状態にする
    nnt::bgtc::SetTaskAvailable(true);
    NN_UTIL_SCOPE_EXIT{ nnt::bgtc::SetTaskAvailable(false); };

    // RTC Alarm 割り込みで起きたように見せかけつつスリープに落とす
    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlag::RtcForBackgroundTask::Mask);
    nn::os::SystemEvent* pAwakeEvent = nullptr;
    server::PowerStateMachine::GetInstance().EnterSleep(&pAwakeEvent);

    // しばらくすると MinimumAwakeForBackgroundTask になっている
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    EXPECT_EQ(PowerState_MinimumAwakeForBackgroundTask, server::PowerStateMachine::GetInstance().GetCurrentState());

    // ここまでの内部ログを空にする
    server::ResetLog();

    // 電池切れイベントを送ると ShutdownReady ステートへ
    server::PowerStateMachine::GetInstance().GetPowerStateMessageQueue()->Enqueue(AddTimeStamp(server::PowerStateMessage_EventShutdownRequiredByVeryLowBattery));
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // ここまでにステートマシン上で起こるべき出来事の一部始終
    const server::LogItemList expectedEventLog =
    {
        // 電池消尽メッセージを受信し、目的地を ShutdownReady に
        server::LogItem(PowerState_MinimumAwakeForBackgroundTask, PowerState_ShutdownReady, server::PowerStateHandlerType_OnMessage, server::PowerStateMessage_EventShutdownRequiredByVeryLowBattery),
        // MinimumAwakeForBackgroundTask を出た
        server::LogItem(PowerState_MinimumAwakeForBackgroundTask, PowerState_None, server::PowerStateHandlerType_Exit, server::PowerStateMessage_None),
        // PSC 経由でシステムモジュール群に ShutdownReady 遷移要求をした
        server::LogItem(nn::psc::PmState_ShutdownReady, nn::psc::PmTransitionOrder_ToLowerPowerState, nn::psc::MakeNoPmFlags()),
        // ShutdownReady に入った
        server::LogItem(PowerState_ShutdownReady, PowerState_None, server::PowerStateHandlerType_Entry, server::PowerStateMessage_None),
        // ShutdownReady の OnDestination を出た
        server::LogItem(PowerState_ShutdownReady, PowerState_None, server::PowerStateHandlerType_OnDestination, server::PowerStateMessage_None)
    };
    const auto& actualEventLog = server::GetLog();
    EXPECT_EQ(expectedEventLog, actualEventLog);

    server::SetEmulatedWakeReasonFlagSet(WakeReasonFlagSet()); // 終わったら空に

    nn::spsm::Finalize();
}
