﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/os.h>

#include <nn/pl/pl_SharedFontApi.h>
#include <nn/pl/pl_SharedFontApiForSystem.h>
#include <nn/pl/srv/pl_SharedFontManagerServer.h>

using namespace std;
using namespace nn;

namespace
{
    const char* SharedFontTypeName[] = {
        "Standard",
        "ChineseSimple",
        "ChineseSimpleExtension",
        "ChineseTraditional",
        "Korean",
        "NintendoExtension",
        "NintendoExtension2",
    };

    // pl は復号済みのデータを返すので、サイズは .ttf のもの。
    size_t SharedFontSize[nn::pl::SharedFontTypeForSystem_Max] = {
        2091276,    // Standard (nintendo_udsg-r_std_003.ttf)
        7814744,    // ChineseSimple (nintendo_udsg-r_org_zh-cn_003.ttf)
        110008,     // ChineseSimpleExtension (nintendo_udsg-r_ext_zh-cn_003.ttf)
        3580260,    // ChineseTraditional (nintendo_udjxh-db_zh-tw_003.ttf)
        3782952,    // Korean (nintendo_udsg-r_ko_003.ttf)
        106112,     // NintendoExtension (nintendo_ext_003.ttf)
        180860,     // NintendoExtension2 (nintendo_ext2_003.ttf)
    };

    // 先頭のヘッダは同じ値が多いので適当なオフセットでデータを比較する。
    const int SampleOffset = 32;
    const int SampleDataSize = 8;
    nn::Bit8 SharedFontSampleData[nn::pl::SharedFontTypeForSystem_Max][SampleDataSize] = {
        { 0xed,0x03,0xe9,0x40,0x00,0x00,0x01,0x14, }, // Standard
        { 0x6C,0x59,0xDC,0x2F,0x00,0x00,0x01,0x14, }, // ChineseSimple
        { 0x5e,0xd3,0x61,0xdd,0x00,0x00,0x01,0x14, }, // ChineseSimpleExtension
        { 0x5e,0x2a,0x92,0xd1,0x00,0x36,0x0d,0xd8, }, // ChineseTraditional
        { 0x9e,0x1d,0xc3,0x28,0x00,0x00,0x01,0x14, }, // Korean
        { 0xF0,0xA7,0x0A,0xED,0x00,0x00,0x0B,0x60, }, // NintendoExtension
        { 0xa4,0x2c,0xa0,0x89,0x00,0x00,0x12,0x6c, }, // NintendoExtension2
    };

    void CheckFontData(pl::SharedFontType fontType, const void* addr, size_t size) NN_NOEXCEPT
    {
        ASSERT_EQ(SharedFontSize[fontType], size) << "fontType = " << SharedFontTypeName[fontType] << "\n";
        for( int i = 0; i < SampleDataSize; i++ )
        {
            ASSERT_EQ(SharedFontSampleData[fontType][i], static_cast<const Bit8*>(addr)[SampleOffset + i]) << SharedFontTypeName[fontType] << "[" << (SampleOffset + i) << "]\n";
        }
    }

    void CheckFontData(pl::SharedFontType expectedFontTypes[], pl::SharedFontType actualFontTypes[], const void* addrs[], size_t sizes[], int count) NN_NOEXCEPT
    {
        for( int i = 0; i < count; i++ )
        {
            ASSERT_EQ(expectedFontTypes[i], actualFontTypes[i]) << "[" << i << "]";
            CheckFontData(expectedFontTypes[i], addrs[i], sizes[i]);
        }
    }

    void CheckFontData(pl::SharedFontType expectedFontTypes[], pl::SharedFontData actualData[], int count) NN_NOEXCEPT
    {
        for( int i = 0; i < count; i++ )
        {
            ASSERT_EQ(expectedFontTypes[i], actualData[i].type) << "[" << i << "]";
            CheckFontData(expectedFontTypes[i], actualData[i].address, actualData[i].size);
        }
    }

    void CheckFontDataForSystem(pl::SharedFontTypeForSystem fontType, const void* addr, size_t size) NN_NOEXCEPT
    {
        ASSERT_EQ(SharedFontSize[fontType], size) << "fontType = " << SharedFontTypeName[fontType] << "\n";
        for( int i = 0; i < SampleDataSize; i++ )
        {
            NN_UNUSED(addr);
            ASSERT_EQ(SharedFontSampleData[fontType][i], static_cast<const Bit8*>(addr)[SampleOffset + i]);
        }
    }

    void CheckFontDataForSystem(pl::SharedFontTypeForSystem expectedFontTypes[], pl::SharedFontTypeForSystem actualFontTypes[], const void* addrs[], size_t sizes[], int count) NN_NOEXCEPT
    {
        for( int i = 0; i < count; i++ )
        {
            ASSERT_EQ(expectedFontTypes[i], actualFontTypes[i]) << "[" << i << "]";
            CheckFontDataForSystem(expectedFontTypes[i], addrs[i], sizes[i]);
        }
    }
}

TEST(SharedFontTest, BasicWithoutStrictDataCheck)
{
    for( int i = 0; i < nn::pl::SharedFontType_Max; i++ )
    {
        nn::pl::SharedFontType type = static_cast<nn::pl::SharedFontType>(i);
        nn::pl::SharedFontLoadState loadState;
        while( (loadState = nn::pl::GetSharedFontLoadState(type)) == nn::pl::SharedFontLoadState_Loading )
        {
            NN_LOG("Waiting for font[%d] to be loaded.\n", type);
            nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
        }
        NN_LOG("font[%d] was loaded.\n", type);

        size_t size = nn::pl::GetSharedFontSize(type);
        // 何かしら有効なデータが取得できていることのみを確認する。
        const nn::Bit8* addr = static_cast<const nn::Bit8*>(nn::pl::GetSharedFontAddress(type));
        ASSERT_TRUE(addr != nullptr);
        ASSERT_GT(size, 0u);
    }
}

TEST(SharedFontTest, GetSharedFontWithoutStrictDataCheck)
{
    int count;
    pl::SharedFontData data[pl::SharedFontType_Max];

    while( !pl::GetSharedFont(&count, data, NN_ARRAY_SIZE(data), settings::LanguageCode::Make(settings::Language_AmericanEnglish)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    // 何かしら有効なデータが取得できていることのみを確認する。
    ASSERT_GT(count, 0);
    for( int i = 0; i < std::min(count, static_cast<int>(NN_ARRAY_SIZE(data))); i++ )
    {
        ASSERT_TRUE(data[i].address != nullptr);
        ASSERT_GT(data[i].size, 0u);
    }
}

TEST(SharedFontTest, Basic)
{
    for( int i = 0; i < nn::pl::SharedFontType_Max; i++ )
    {
        nn::pl::SharedFontType type = static_cast<nn::pl::SharedFontType>(i);
        nn::pl::SharedFontLoadState loadState;
        while( (loadState = nn::pl::GetSharedFontLoadState(type)) == nn::pl::SharedFontLoadState_Loading )
        {
            NN_LOG("Waiting for font[%d] to be loaded.\n", type);
            nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
        }
        NN_LOG("font[%d] was loaded.\n", type);

        size_t size = nn::pl::GetSharedFontSize(type);
        const nn::Bit8* addr = static_cast<const nn::Bit8*>(nn::pl::GetSharedFontAddress(type));
        CheckFontData(type, addr, size);
    }
}

TEST(SharedFontTest, LoadRequest)
{
    // クラッシュしないことのみ確認
    // TODO: どうやってテストしたらよいのだろう？
    for( int i = 0; i < nn::pl::SharedFontType_Max; i++ )
    {
        nn::pl::RequestSharedFontLoad(static_cast<nn::pl::SharedFontType>(i));
    }
    SUCCEED();
}

TEST(SharedFontTest, GetInOrder)
{
    const int OutCountMax = 6;
    int count;
    pl::SharedFontType types[OutCountMax];
    const void* addresses[OutCountMax];
    size_t sizes[OutCountMax];

    // 中国語
    while( !pl::GetSharedFontInOrderOfPriority(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_Chinese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(6, count);

    pl::SharedFontType expectedTypesChinese[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_Korean,
        pl::SharedFontType_Standard,
    };
    CheckFontData(expectedTypesChinese, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesChinese) / sizeof(pl::SharedFontType)));

    // 中国語（簡体字）
    while( !pl::GetSharedFontInOrderOfPriority(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_SimplifiedChinese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(6, count);

    pl::SharedFontType expectedTypesSimplifiedChinese[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_Korean,
        pl::SharedFontType_Standard,
    };
    CheckFontData(expectedTypesSimplifiedChinese, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesSimplifiedChinese) / sizeof(pl::SharedFontType)));


    // 韓国語
    while( !pl::GetSharedFontInOrderOfPriority(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_Korean)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(6, count);

    pl::SharedFontType expectedTypesKorean[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_Korean,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_Standard,
    };
    CheckFontData(expectedTypesKorean, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesChinese) / sizeof(pl::SharedFontType)));

    // 台湾語
    while( !pl::GetSharedFontInOrderOfPriority(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_Taiwanese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(6, count);

    pl::SharedFontType expectedTypesTaiwanese[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_Korean,
        pl::SharedFontType_Standard,
    };
    CheckFontData(expectedTypesTaiwanese, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesTaiwanese) / sizeof(pl::SharedFontType)));

    // 中国語（繁体字）
    while( !pl::GetSharedFontInOrderOfPriority(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_TraditionalChinese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(6, count);

    pl::SharedFontType expectedTypesTraditionalChinese[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_Korean,
        pl::SharedFontType_Standard,
    };
    CheckFontData(expectedTypesTraditionalChinese, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesTraditionalChinese) / sizeof(pl::SharedFontType)));


    // 上記以外（日本語、英語など）
    while( !pl::GetSharedFontInOrderOfPriority(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_Japanese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(6, count);

    pl::SharedFontType expectedTypesOthers[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_Standard,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_Korean,
    };
    CheckFontData(expectedTypesOthers, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesOthers) / sizeof(pl::SharedFontType)));
}

TEST(SharedFontTest, GetInOrderForSystem)
{
    const int OutCountMax = 6;
    int count;
    pl::SharedFontTypeForSystem types[OutCountMax];
    const void* addresses[OutCountMax];
    size_t sizes[OutCountMax];

    // 中国語
    while( !pl::GetSharedFontInOrderOfPriorityForSystem(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_Chinese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }
    ASSERT_EQ(6, count);

    pl::SharedFontTypeForSystem expectedTypesChinese[] = {
        pl::SharedFontTypeForSystem_NintendoExtension2,
        pl::SharedFontTypeForSystem_ChineseSimpleExtension,
        pl::SharedFontTypeForSystem_ChineseSimple,
        pl::SharedFontTypeForSystem_ChineseTraditional,
        pl::SharedFontTypeForSystem_Korean,
        pl::SharedFontTypeForSystem_Standard,
    };
    CheckFontDataForSystem(expectedTypesChinese, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesChinese) / sizeof(pl::SharedFontType)));

    // 中国語（簡体字）
    while( !pl::GetSharedFontInOrderOfPriorityForSystem(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_SimplifiedChinese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }
    ASSERT_EQ(6, count);

    pl::SharedFontTypeForSystem expectedTypesSimplifiedChinese[] = {
        pl::SharedFontTypeForSystem_NintendoExtension2,
        pl::SharedFontTypeForSystem_ChineseSimpleExtension,
        pl::SharedFontTypeForSystem_ChineseSimple,
        pl::SharedFontTypeForSystem_ChineseTraditional,
        pl::SharedFontTypeForSystem_Korean,
        pl::SharedFontTypeForSystem_Standard,
    };
    CheckFontDataForSystem(expectedTypesSimplifiedChinese, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesSimplifiedChinese) / sizeof(pl::SharedFontType)));

    // 韓国語
    while( !pl::GetSharedFontInOrderOfPriorityForSystem(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_Korean)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }
    ASSERT_EQ(6, count);

    pl::SharedFontTypeForSystem expectedTypesKorean[] = {
        pl::SharedFontTypeForSystem_NintendoExtension2,
        pl::SharedFontTypeForSystem_Korean,
        pl::SharedFontTypeForSystem_ChineseSimpleExtension,
        pl::SharedFontTypeForSystem_ChineseSimple,
        pl::SharedFontTypeForSystem_ChineseTraditional,
        pl::SharedFontTypeForSystem_Standard,
    };
    CheckFontDataForSystem(expectedTypesKorean, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesChinese) / sizeof(pl::SharedFontType)));

    // 台湾語
    while( !pl::GetSharedFontInOrderOfPriorityForSystem(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_Taiwanese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }
    ASSERT_EQ(6, count);

    pl::SharedFontTypeForSystem expectedTypesTaiwanese[] = {
        pl::SharedFontTypeForSystem_NintendoExtension2,
        pl::SharedFontTypeForSystem_ChineseTraditional,
        pl::SharedFontTypeForSystem_ChineseSimpleExtension,
        pl::SharedFontTypeForSystem_ChineseSimple,
        pl::SharedFontTypeForSystem_Korean,
        pl::SharedFontTypeForSystem_Standard,
    };
    CheckFontDataForSystem(expectedTypesTaiwanese, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesTaiwanese) / sizeof(pl::SharedFontType)));

    // 中国語（繁体字）
    while( !pl::GetSharedFontInOrderOfPriorityForSystem(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_TraditionalChinese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }
    ASSERT_EQ(6, count);

    pl::SharedFontTypeForSystem expectedTypesTraditionalChinese[] = {
        pl::SharedFontTypeForSystem_NintendoExtension2,
        pl::SharedFontTypeForSystem_ChineseTraditional,
        pl::SharedFontTypeForSystem_ChineseSimpleExtension,
        pl::SharedFontTypeForSystem_ChineseSimple,
        pl::SharedFontTypeForSystem_Korean,
        pl::SharedFontTypeForSystem_Standard,
    };
    CheckFontDataForSystem(expectedTypesTraditionalChinese, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesTraditionalChinese) / sizeof(pl::SharedFontType)));

    // 上記以外（日本語、英語など）
    while( !pl::GetSharedFontInOrderOfPriorityForSystem(&count, types, addresses, sizes, OutCountMax, settings::LanguageCode::Make(settings::Language_Japanese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }
    ASSERT_EQ(6, count);

    pl::SharedFontTypeForSystem expectedTypesOthers[] = {
        pl::SharedFontTypeForSystem_NintendoExtension2,
        pl::SharedFontTypeForSystem_Standard,
        pl::SharedFontTypeForSystem_ChineseSimpleExtension,
        pl::SharedFontTypeForSystem_ChineseSimple,
        pl::SharedFontTypeForSystem_ChineseTraditional,
        pl::SharedFontTypeForSystem_Korean,
    };
    CheckFontDataForSystem(expectedTypesOthers, types, addresses, sizes, static_cast<int>(sizeof(expectedTypesOthers) / sizeof(pl::SharedFontType)));
}

TEST(SharedFontTest, GetSharedFont)
{
    int count;
    pl::SharedFontData data[pl::SharedFontType_Max];

    // 中国語
    while( !pl::GetSharedFont(&count, data, pl::SharedFontType_Max, settings::LanguageCode::Make(settings::Language_Chinese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(pl::SharedFontType_Max, count);

    pl::SharedFontType expectedTypesChinese[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_Korean,
        pl::SharedFontType_Standard,
    };
    CheckFontData(expectedTypesChinese, data, pl::SharedFontType_Max);

    // 中国語（簡体字）
    while( !pl::GetSharedFont(&count, data, pl::SharedFontType_Max, settings::LanguageCode::Make(settings::Language_SimplifiedChinese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(pl::SharedFontType_Max, count);

    pl::SharedFontType expectedTypesSimplifiedChinese[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_Korean,
        pl::SharedFontType_Standard,
    };
    CheckFontData(expectedTypesSimplifiedChinese, data, pl::SharedFontType_Max);

    // 韓国語
    while( !pl::GetSharedFont(&count, data, pl::SharedFontType_Max, settings::LanguageCode::Make(settings::Language_Korean)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(pl::SharedFontType_Max, count);

    pl::SharedFontType expectedTypesKorean[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_Korean,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_Standard,
    };
    CheckFontData(expectedTypesKorean, data, pl::SharedFontType_Max);

    // 台湾語
    while( !pl::GetSharedFont(&count, data, pl::SharedFontType_Max, settings::LanguageCode::Make(settings::Language_Taiwanese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(pl::SharedFontType_Max, count);

    pl::SharedFontType expectedTypesTaiwanese[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_Korean,
        pl::SharedFontType_Standard,
    };
    CheckFontData(expectedTypesTaiwanese, data, pl::SharedFontType_Max);

    // 中国語（繁体字）
    while( !pl::GetSharedFont(&count, data, pl::SharedFontType_Max, settings::LanguageCode::Make(settings::Language_TraditionalChinese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(pl::SharedFontType_Max, count);

    pl::SharedFontType expectedTypesTraditionalChinese[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_Korean,
        pl::SharedFontType_Standard,
    };
    CheckFontData(expectedTypesTraditionalChinese, data, pl::SharedFontType_Max);

    // 上記以外（日本語、英語など）
    while( !pl::GetSharedFont(&count, data, pl::SharedFontType_Max, settings::LanguageCode::Make(settings::Language_Japanese)) )
    {
        NN_LOG("Waiting for all fonts to be loaded.\n");
        nn::os::SleepThread(nn::TimeSpan(std::chrono::milliseconds(1000)));
    }

    ASSERT_EQ(6, count);

    pl::SharedFontType expectedTypesOthers[] = {
        pl::SharedFontType_NintendoExtension,
        pl::SharedFontType_Standard,
        pl::SharedFontType_ChineseSimpleExtension,
        pl::SharedFontType_ChineseSimple,
        pl::SharedFontType_ChineseTraditional,
        pl::SharedFontType_Korean,
    };
    CheckFontData(expectedTypesOthers, data, pl::SharedFontType_Max);
}

