﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/dd.h>

#include <nn/pinmux/driver/pinmux.h>

#include <nne/pinmux/pinmux.h>

#include <nnt/nntest.h>

#define ADDITIONAL_SLEEP_FOR_AVOIDING_TIMEOUT

namespace
{
const nn::TimeSpan SleepTime = nn::TimeSpan::FromMilliSeconds(5);

const nn::Bit32 SchmtMask   = 1 << 12;
const nn::Bit32 SchmtEnable = 1 << 12;
const nn::Bit32 SchmtDisable= 0 << 12;

const nn::Bit32 PuPdMask   = 1 << 2 | 1 << 3;
const nn::Bit32 PuPdNone   = 0 << 2 | 0 << 3;
const nn::Bit32 PuPdDown   = 1 << 2 | 0 << 3;

uintptr_t g_RegistersAddress;

void CheckRegister(nn::Bit32 expectValue, nn::Bit32 mask)
{
    auto reg = nne::pinmux::tegra::ReadRegPinmuxPad(nne::pinmux::tegra::tx1::PinmuxPadIndex_Sdmmc1Clk);
    EXPECT_EQ(expectValue, mask & reg);
    EXPECT_EQ(expectValue, mask & *reinterpret_cast<volatile uint32_t*>(g_RegistersAddress + 0x3000));
}


}


//-----------------------------------------------------------------------------
// Pinmux ライブラリの Sdmmc 向けのテスト
// pinmuxDriver を使用して SDMMC 向けのレジスタ設定をいじるため、動かす環境に要注意
// ちゃんとした機能の確認は波形を見る必要があるため、ここでは特定のレジスタの値のチェックだけ行う
TEST(PinmuxSdmmcTest, Input)
{
#if (defined ADDITIONAL_SLEEP_FOR_AVOIDING_TIMEOUT)
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000LL));
#endif
    g_RegistersAddress = nn::dd::QueryIoMappingAddress(0x70000000ull, 0x1000);

    nn::pinmux::driver::PinmuxSession sdmmcSession;

    nn::pinmux::driver::Initialize();

    nn::pinmux::driver::OpenSession(&sdmmcSession, nn::pinmux::AssignablePinGroupName_Sdmmc1);

    // SCHMT を Enable へ
    NN_LOG("SCHMT : Enable\n");
    nn::pinmux::driver::SetPinAssignment(&sdmmcSession, nn::pinmux::PinAssignment_Sdmmc1SchmtEnable);

    CheckRegister(SchmtEnable, SchmtMask);

    // SCHMT を Disable へ
    NN_LOG("SCHMT : Disable\n");
    nn::pinmux::driver::SetPinAssignment(&sdmmcSession, nn::pinmux::PinAssignment_Sdmmc1SchmtDisable);

    CheckRegister(SchmtDisable, SchmtMask);

    // 内部的な pulldown の チェック
    // pullup も pulldown もしない
    NN_LOG("PUPD : None\n");
    nn::pinmux::driver::SetPinAssignment(&sdmmcSession, nn::pinmux::PinAssignment_Sdmmc1OutputHigh);

    CheckRegister(PuPdNone, PuPdMask);

    // pulldown へ
    NN_LOG("PUPD : Pulldown\n");
    nn::pinmux::driver::SetPinAssignment(&sdmmcSession, nn::pinmux::PinAssignment_Sdmmc1ResetState);

    CheckRegister(PuPdDown, PuPdMask);

    nn::pinmux::driver::CloseSession(&sdmmcSession);

    nn::pinmux::driver::Finalize();
}
