﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/oe.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>

#include <nnt/gfx/testGfx_GraphicsFramework.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include <nn/fs/fs_MemoryManagement.h>

#include "../OeCommon/testOe_Init.h"
namespace{

// ドック挿抜の試行回数
const int MaxCount = 2;
extern "C" int NvOsDrvOpen(const char *pathname);

//------------------------------------------------------------------------------
// テスト中の状態をあらわします
//
struct TestState
{
    nn::gfx::util::DebugFontTextWriter* pWriter;
    int testCount;
    bool IsTestFinished;
    bool IsTestSucceeded;
    nn::os::SystemEventType* pSystemEvent;
    nn::oe::PerformanceMode performanceMode;
};
//-------------------------------------------------------------------------------
// テストの状態の初期化
//-------------------------------------------------------------------------------

void InitializeTestState(TestState *pTestState, nn::gfx::util::DebugFontTextWriter *writer)
{
    pTestState->pWriter = writer;
    pTestState->pSystemEvent = nn::oe::GetNotificationMessageEvent();
    pTestState->testCount = 0;
    pTestState->IsTestFinished = false;
    pTestState->IsTestSucceeded = false;
    pTestState->performanceMode = nn::oe::GetPerformanceMode();
}



//------------------------------------------------------------------------------
//  NX上で文字列を描画する関数
//------------------------------------------------------------------------------
void Print(nn::gfx::util::DebugFontTextWriter& writer, TestState *pData)
{
    const nn::util::Color4u8Type white = { { 255, 255, 255, 255 } };

    writer.SetScale(2.0f, 2.0f);

    writer.SetTextColor(white);
    writer.SetCursor(200, 250);
    auto pPerformanceModeStr = pData->performanceMode
        ? "PerformanceMode_Boost\n"
        : "PerformanceMode_Normal\n";
    auto pTestSuccessStr = pData->IsTestSucceeded
        ? "Success"
        : "Failed";
    writer.Print("PerformanceMode: %s\n", pPerformanceModeStr);
    writer.Print("Previous test result: %s\n", pTestSuccessStr);
    if (pData->testCount <= MaxCount)
    {
        writer.Print("Please dock or undock hardware.\n");
    }
    if ( 0 < pData->testCount && pData->testCount < MaxCount )
    {
        writer.Print("One more set!\n");
    }
    else if (pData->testCount >= MaxCount)
    {
        writer.Print("Finished!\n");
    }
}

//------------------------------------------------------------------------------
//  描画コマンド生成
//------------------------------------------------------------------------------


void MakeCommand(nnt::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pData)
{
    TestState* pTestState = reinterpret_cast<TestState*>(pData);

    // フォント表示
    Print(*pTestState->pWriter, pTestState);

    // コマンド生成
    pGraphicsFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

        // レンダーターゲット、ビューポートシザー設定
        nn::gfx::ColorTargetView* target = pGraphicsFramework->GetColorTargetView();
        rootCommandBuffer->ClearColor(target, 0.1f, 0.1f, 0.1f, 1.0f, nullptr);
        rootCommandBuffer->SetRenderTargets(1, &target, nullptr);
        rootCommandBuffer->SetViewportScissorState(pGraphicsFramework->GetViewportScissorState());

        // デバッグフォント用のコマンド生成
        pTestState->pWriter->Draw(rootCommandBuffer);
    }
    pGraphicsFramework->EndFrame(bufferIndex, true);
}

//----------------------
// テスト本体の処理
//----------------------
void Calculate(nnt::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    auto* pTestState = reinterpret_cast<TestState*>(pUserData);

    if (!TryWaitSystemEvent(pTestState->pSystemEvent))
    {
        return;
    }

    nn::oe::Message message = nn::oe::PopNotificationMessage();
    if ( message != nn::oe::MessagePerformanceModeChanged )
    {
        return;
    }
    pTestState->performanceMode = nn::oe::GetPerformanceMode();
    NN_LOG("Performance mode has been changed to %s.\n",
           (pTestState->performanceMode == nn::oe::PerformanceMode_Boost)
           ? "nn::oe::PerformanceMode_Boost"
           : "nn::oe::PerformanceMode_Normal");
    pTestState->IsTestSucceeded = RUN_ALL_TESTS() == 0;
    pTestState->testCount++;
    if ( pTestState->testCount == MaxCount )
    {
        NN_LOG("Finished!\n");
        pTestState->IsTestFinished = true;
    }
    else if ( pTestState->testCount != 0)
    {
        NN_LOG("One more set!\n");
    }
}

void MainLoop(nnt::gfx::GraphicsFramework* pGraphicsFramework, TestState *pTestState)
{
    int finishCount = 0;
    // 毎フレームのレンダリング
    while(true)
    {
        pGraphicsFramework->ProcessFrame();

        // 終了状態のときはしばらく待ってから終了
        if (pTestState->IsTestFinished)
        {
            finishCount++;
            if (finishCount > 60 * 10)
            {
                return;
            }
        }
    }
}

}

extern "C" void nninitStartup()
{
    nnt::oe::PreInitializeTestEnvironment();
}
void InitializeTestEnvironment() NN_NOEXCEPT
{
    // CI で実行するために malloc の定義が必要。
    static auto alloc = [](size_t size) -> void*
    {
        return std::malloc(size);
    };

    static auto dealloc = [](void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    };
    nn::fs::SetAllocator(alloc, dealloc);
}

//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    InitializeTestEnvironment();

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // フレームワーク初期化
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nnt::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);
    NvOsDrvOpen("/dev/nvhost-ctrl-gpu");

    const int BufferCount = 2;
    nnt::gfx::GraphicsFramework gfw;
    {
        nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
        fwInfo.SetDefault();
        fwInfo.SetDisplayWidth(1280);
        fwInfo.SetDisplayHeight(720);
        fwInfo.SetBufferCount(BufferCount);
        fwInfo.SetSwapChainBufferCount(BufferCount);
        gfw.Initialize(fwInfo);
    }


    // デバッグフォント初期化
    const bool userMemoryPoolEnable = false;    // true にすると、ユーザーのメモリプールを使用します
    nn::gfx::util::DebugFontTextWriterInfo info;
    {
        const int charCountMax = 1024;
        info.SetDefault();
        info.SetCharCountMax(charCountMax);
        info.SetBufferCount(BufferCount);
        info.SetUserMemoryPoolEnabled(userMemoryPoolEnable);
    }


    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(gfw.GetDevice(), info);
    nn::util::BytePtr debugFontHeap(new uint8_t[debugFontHeapSize]);

    size_t debugFontMemoryPoolSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(gfw.GetDevice(), info);

    nnt::gfx::GraphicsFramework::MemoryPoolType type = nnt::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
    nn::gfx::MemoryPool* pMemoryPool = gfw.GetMemoryPool(type);
    ptrdiff_t memoryPoolOffset = nn::gfx::util::MemoryPoolAllocator::InvalidOffset;
    nn::gfx::util::DebugFontTextWriter writer;
    if (NN_STATIC_CONDITION(userMemoryPoolEnable))
    {
        memoryPoolOffset = gfw.AllocatePoolMemory(type, debugFontMemoryPoolSize, 1);

        writer.Initialize(gfw.GetDevice(), info, debugFontHeap.Get(), debugFontHeapSize,
            pMemoryPool, memoryPoolOffset, debugFontMemoryPoolSize);
    }
    else
    {
        writer.Initialize(gfw.GetDevice(), info, debugFontHeap.Get(), debugFontHeapSize, nullptr, 0, 0);
    }

    int textureDescriptorIndex = gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    int samplerDescriptorIndex = gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);

    writer.SetDisplayWidth(gfw.GetDisplayWidth());
    writer.SetDisplayHeight(gfw.GetDisplayHeight());
    writer.SetTextureDescriptor(gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView), textureDescriptorIndex);
    writer.SetSamplerDescriptor(gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler), samplerDescriptorIndex);

    nn::oe::SetPerformanceModeChangedNotificationEnabled(true);

    TestState testState;
    InitializeTestState(&testState, &writer);
    testState.IsTestSucceeded = RUN_ALL_TESTS() == 0;

    gfw.SetMakeCommandCallback(MakeCommand, &testState);
    gfw.SetCalculateCallback(Calculate, &testState);

    MainLoop(&gfw, &testState);

    /////////////////////////////////////////////////
    // 終了処理
    gfw.QueueFinish();


    // デバッグフォント終了
    writer.Finalize();
    delete[] reinterpret_cast<uint8_t*>(debugFontHeap.Get());
    debugFontHeap.Reset(nullptr);

    if (NN_STATIC_CONDITION(userMemoryPoolEnable)) {
        gfw.FreePoolMemory(type, memoryPoolOffset);
    }
    gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, textureDescriptorIndex);
    gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, samplerDescriptorIndex);

    // フレームワーク終了
    gfw.Finalize();
}
