﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/oe.h>

#include <nn/fgm/fgm.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/init/init_Malloc.h>
#include <nn/pcv/pcv.h>

#include <nnt/nntest.h>

#include <nv/nv_MemoryManagement.h>

#include "testOe_Init.h"

namespace {

const int FirmwareMemoryMargin = 40 * 4096;
NN_ALIGNAS(nn::os::MemoryPageSize) char g_GraphicsMemory[16 * 4096 + FirmwareMemoryMargin];

extern "C" int NvOsDrvOpen(const char *pathname);

} // namespace

namespace nnt { namespace oe {

namespace {

// 値取得用のリクエスト。
nn::fgm::Request g_RequestCpu;
nn::fgm::Request g_RequestGpu;
nn::fgm::Request g_RequestEmc;

class OeEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp() override
    {
        nn::oe::Initialize();

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_RequestCpu.Initialize(nn::fgm::Module_Cpu, nn::fgm::Priority_Default));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_RequestGpu.Initialize(nn::fgm::Module_Gpu, nn::fgm::Priority_Default));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_RequestEmc.Initialize(nn::fgm::Module_Emc, nn::fgm::Priority_Default));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_RequestCpu.SetAndWait(nn::fgm::Setting_Min, nn::fgm::Setting_Max));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_RequestGpu.SetAndWait(nn::fgm::Setting_Min, nn::fgm::Setting_Max));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_RequestEmc.SetAndWait(nn::fgm::Setting_Min, nn::fgm::Setting_Max));

        nn::pcv::Initialize();
    }

    virtual void TearDown() override
    {
        nn::pcv::Finalize();

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_RequestCpu.Finalize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_RequestGpu.Finalize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_RequestEmc.Finalize());
    }
};

} // namespace

bool IsSettableToAllPerformanceMode(const ConfigInfo& configInfo) NN_NOEXCEPT
{
    // PerformanceMode_Normal のものはすべての性能モードに設定できる。
    return (configInfo.mode == nn::oe::PerformanceMode_Normal);
}

bool IsSettableToPerformanceModeNormal(const ConfigInfo& configInfo) NN_NOEXCEPT
{
    return (configInfo.mode == nn::oe::PerformanceMode_Normal);
}

bool IsSettableToPerformanceModeBoost(const ConfigInfo& configInfo) NN_NOEXCEPT
{
    return (configInfo.mode == nn::oe::PerformanceMode_Normal)
        || (configInfo.mode == nn::oe::PerformanceMode_Boost);
}

void CheckFrequencies(nn::fgm::Setting cpuSetting, nn::fgm::Setting gpuSetting, nn::fgm::Setting emcSetting) NN_NOEXCEPT
{
    nn::fgm::Setting setting = 0;

    EXPECT_TRUE(g_RequestCpu.Get(&setting).IsSuccess());
    NN_LOG("Current Cpu Setting Rate: %d\n", setting);
    EXPECT_EQ(cpuSetting, setting);

    setting = 0;

    EXPECT_TRUE(g_RequestGpu.Get(&setting).IsSuccess());
    NN_LOG("Current Gpu Setting Rate: %d\n", setting);
    EXPECT_EQ(gpuSetting, setting);

    setting = 0;

    EXPECT_TRUE(g_RequestEmc.Get(&setting).IsSuccess());
    NN_LOG("Current Emc Setting Rate: %d\n", setting);
    EXPECT_EQ(emcSetting, setting);

    // SIGLO-43419: FgmBasic テスト削除に伴い CPU/GPU/EMC についての FGM, PCV 間のクロックレートの同期の確認を本テストに移動。
    nn::pcv::ClockHz clockHz = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::GetClockRate(&clockHz, nn::pcv::Module_Cpu));
    EXPECT_EQ(static_cast<nn::pcv::ClockHz>(cpuSetting), clockHz);

    clockHz = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::GetClockRate(&clockHz, nn::pcv::Module_Gpu));
    EXPECT_EQ(static_cast<nn::pcv::ClockHz>(gpuSetting), clockHz);

    clockHz = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::GetClockRate(&clockHz, nn::pcv::Module_Emc));
    EXPECT_EQ(static_cast<nn::pcv::ClockHz>(emcSetting), clockHz);
}

void CheckFrequencies(const ConfigInfo& param) NN_NOEXCEPT
{
    CheckFrequencies(param.cpuSetting, param.gpuSetting, param.emcSetting);
}

void PreInitializeTestEnvironment() NN_NOEXCEPT
{
    // CI で実行するために malloc で確保できる十分なメモリ領域が必要。
    const size_t TotalHeapSize = 256 * 1024 * 1024;

    uintptr_t address = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(TotalHeapSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&address, TotalHeapSize / 2));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), TotalHeapSize / 2);

    ::testing::AddGlobalTestEnvironment(new OeEnvironment);
}

void InitializeTestEnvironment() NN_NOEXCEPT
{
    // nv::InitializeGraphics は 1 度しか呼べない。
    // --gtest_repeat オプションへの対応の為、nnMain に配置する。
    nv::InitializeGraphics(g_GraphicsMemory, sizeof(g_GraphicsMemory));
    NvOsDrvOpen("/dev/nvhost-ctrl-gpu");

    // CI で実行するために malloc の定義が必要。
    static auto alloc = [](size_t size) -> void* {
        return std::malloc(size);
    };

    static auto dealloc = [](void* p, size_t size) {
        NN_UNUSED(size);
        std::free(p);
    };

    nn::fs::SetAllocator(alloc, dealloc);
}

void SetConfigAndCheckFrequencies(const ConfigInfo& configInfo) NN_NOEXCEPT
{
    if ( IsSettableToAllPerformanceMode(configInfo) )
    {
        nn::oe::SetPerformanceConfiguration(configInfo.config);
        EXPECT_EQ(configInfo.config, nn::oe::GetPerformanceConfiguration(nn::oe::PerformanceMode_Normal));
        EXPECT_EQ(configInfo.config, nn::oe::GetPerformanceConfiguration(nn::oe::PerformanceMode_Boost));

        CheckFrequencies(configInfo);
    }
}

void SetConfigAndCheckFrequenciesOnNormal(const ConfigInfo& configInfo) NN_NOEXCEPT
{
    if ( IsSettableToPerformanceModeNormal(configInfo) )
    {
        nn::oe::SetPerformanceConfiguration(nn::oe::PerformanceMode_Normal, configInfo.config);
        EXPECT_EQ(configInfo.config, nn::oe::GetPerformanceConfiguration(nn::oe::PerformanceMode_Normal));

        // 現在の性能モードがノーマルモードならばそのままチェックする。
        if ( nn::oe::GetPerformanceMode() == nn::oe::PerformanceMode_Normal )
        {
            CheckFrequencies(configInfo);
        }
    }
}

void SetConfigAndCheckFrequenciesOnBoost(const ConfigInfo& configInfo) NN_NOEXCEPT
{
    if ( IsSettableToPerformanceModeBoost(configInfo) )
    {
        // ブーストモード側に性能を設定する。
        nn::oe::SetPerformanceConfiguration(nn::oe::PerformanceMode_Boost, configInfo.config);
        EXPECT_EQ(configInfo.config, nn::oe::GetPerformanceConfiguration(nn::oe::PerformanceMode_Boost));

        // 現在の性能モードがブーストモードならばそのままチェックする。
        if ( nn::oe::GetPerformanceMode() == nn::oe::PerformanceMode_Boost )
        {
            CheckFrequencies(configInfo);
        }
    }
}

void SetDeathConfigAndCheckFrequencies(const ConfigInfo& configInfo) NN_NOEXCEPT
{
    if ( !IsSettableToAllPerformanceMode(configInfo) )
    {
        EXPECT_DEATH_IF_SUPPORTED(nn::oe::SetPerformanceConfiguration(configInfo.config), "");
    }
}

void SetDeathConfigAndCheckFrequenciesOnNormal(const ConfigInfo& configInfo) NN_NOEXCEPT
{
    if ( !IsSettableToPerformanceModeNormal(configInfo) )
    {
        EXPECT_DEATH_IF_SUPPORTED(nn::oe::SetPerformanceConfiguration(nn::oe::PerformanceMode_Normal, configInfo.config), "");
    }
}

void SetDeathConfigAndCheckFrequenciesOnBoost(const ConfigInfo& configInfo) NN_NOEXCEPT
{
    if ( !IsSettableToPerformanceModeBoost(configInfo) )
    {
        EXPECT_DEATH_IF_SUPPORTED(nn::oe::SetPerformanceConfiguration(nn::oe::PerformanceMode_Boost, configInfo.config), "");
    }
}

}} // namespace nnt::oe
