﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/apm/apm.h>
#include <nn/apm/apm_System.h>
#include <nn/apm/server/apm.h>
#include <nn/psc.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>

#include <nnt/nnt_Argument.h>

#include <nnt/base/testBase_Exit.h>
#include <nnt/nntest.h>

#include "../ApmServerCommon/ApmServerTestInit.h"
#include "../ApmServerCommon/BpcStub.h"
#include "../ApmServerCommon/PsmStub.h"

#include "../../../../../Programs/Eris/Sources/Processes/ppc/ppc_ApmServer.h"

namespace nnt { namespace apm {

namespace {

struct ConfigInfo
{
    nn::apm::PerformanceMode mode;
    nn::apm::PerformanceConfiguration config;
    nn::fgm::Setting cpuSetting;
    nn::fgm::Setting gpuSetting;
    nn::fgm::Setting emcSetting;
};

enum Action
{
    Action_AcOkOn,
    Action_AcOkOff,

    Action_ChargerTypeUnconnected,
    Action_ChargerTypeLowPower,
    Action_ChargerTypeNotSupported,
    Action_ChargerTypeEnoughPower,

    Action_BatteryVoltageStateBoostPerformanceModeProhibited,
    Action_BatteryVoltageStateShutdownRequired,
    Action_BatteryVoltageStateSleepRequired,
    Action_BatteryVoltageStateGood,

    Action_RequestBoostMode,
    Action_RequestNormalMode,
};

struct StepInfo
{
    Action action;
    ConfigInfo expected;
};

struct TestInfo
{
    const StepInfo* pSteps;
    int size;
};

const ConfigInfo ConfigNormal     = { nn::apm::PerformanceMode_Normal, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz, 1020000000, 307200000, 1331200000 };

const ConfigInfo ConfigBoost      = { nn::apm::PerformanceMode_Boost,  nn::apm::PerformanceConfiguration_Cpu1020MhzGpu768Mhz,           1020000000, 768000000, 1600000000 };

// PerformanceMode, PerformanceConfiguration の値は参照されない想定
const ConfigInfo ConfigBackground = { nn::apm::PerformanceMode_Normal, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz, 1020000000, 307200000, 1331200000 };

const StepInfo StepsFullAwake01[] =
{
    { Action_AcOkOn,                  ConfigNormal },
    { Action_BatteryVoltageStateGood, ConfigNormal },
    { Action_ChargerTypeEnoughPower,  ConfigNormal },
    { Action_RequestBoostMode,        ConfigBoost  },
};

const TestInfo TestInfoList[] =
{
    { &(StepsFullAwake01[0]), sizeof(StepsFullAwake01) / sizeof(StepsFullAwake01[0]) },
};

class PerformanceModeTransitionTest : public ::testing::TestWithParam<nnt::apm::TestInfo>
{
    virtual void SetUp() NN_OVERRIDE
    {
        nn::ppc::InitializeApmServer(false);
        nn::ppc::StartApmServer();
        nn::apm::Initialize();
        nn::apm::InitializeForSystem();

        nnt::apm::server::SetUp();
    }

    virtual void TearDown() NN_OVERRIDE
    {
        nnt::apm::server::TearDown();

        nn::apm::FinalizeForSystem();
        nn::apm::Finalize();
        nn::ppc::RequestStopApmServer();
        nn::ppc::FinalizeApmServer();
    }
};

INSTANTIATE_TEST_CASE_P(TestInfos, PerformanceModeTransitionTest, ::testing::ValuesIn(TestInfoList));

} // namespace

// このテストでは APM ライブラリの各関数が各本体状態で期待通りに使用できることを確認する。
// また、APM サーバが PSM, FGM に対して適切な値を設定していることを確認する。（性能モード遷移時）
// ※ 強制ブーストモードは一旦確認対象外とする。

// FullAwake 状態で各関数が実行可能であることを確認する。
// 性能変更側で見れていない関数のみをコールするだけ。
TEST_F(PerformanceModeTransitionTest, FullAwakeTestFunctions)
{
    nn::psc::PmFlagSet flags;
    flags.Reset();

    nn::psc::PmModule* pPmModule;
    nn::ppc::GetPmModuleForTest(&pPmModule);
    pPmModule->Dispatch(nn::psc::PmState_FullAwake, flags);

    nn::apm::ThrottlingState state;

    nn::apm::GetThrottlingState(&state);
    nn::apm::GetLastThrottlingState(&state);
    nn::apm::ClearLastThrottlingState();
}

// MinimumAwake 状態で各関数が実行可能であることを確認する。
// 性能変更側で見れていない関数のみをコールするだけ。
TEST_F(PerformanceModeTransitionTest, MinimumAwakeTestFunctions)
{
    nn::psc::PmFlagSet flags;
    flags.Reset();

    nn::psc::PmModule* pPmModule;
    nn::ppc::GetPmModuleForTest(&pPmModule);
    pPmModule->Dispatch(nn::psc::PmState_MinimumAwake, flags);

    nn::apm::ThrottlingState state;

    nn::apm::GetThrottlingState(&state);
    nn::apm::GetLastThrottlingState(&state);
    nn::apm::ClearLastThrottlingState();
}

void HandleAction(Action action) NN_NOEXCEPT
{
    switch ( action )
    {
    case Action_AcOkOn:
        nnt::bpc::SetAcOk(true);
        break;
    case Action_AcOkOff:
        nnt::bpc::SetAcOk(false);
        break;
    case Action_ChargerTypeUnconnected:
        nnt::psm::SetChargerType(nn::psm::ChargerType_Unconnected);
        break;
    case Action_ChargerTypeLowPower:
        nnt::psm::SetChargerType(nn::psm::ChargerType_LowPower);
        break;
    case Action_ChargerTypeNotSupported:
        nnt::psm::SetChargerType(nn::psm::ChargerType_NotSupported);
        break;
    case Action_ChargerTypeEnoughPower:
        nnt::psm::SetChargerType(nn::psm::ChargerType_EnoughPower);
        break;
    case Action_BatteryVoltageStateBoostPerformanceModeProhibited:
        nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_BoostPerformanceModeProhibited);
        break;
    case Action_BatteryVoltageStateShutdownRequired:
        nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_ShutdownRequired);
        break;
    case Action_BatteryVoltageStateSleepRequired:
        nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_SleepRequired);
        break;
    case Action_BatteryVoltageStateGood:
        nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_Good);
        break;
    case Action_RequestNormalMode:
        nn::apm::RequestPerformanceMode(nn::apm::PerformanceMode_Normal);
        break;
    case Action_RequestBoostMode:
        nn::apm::RequestPerformanceMode(nn::apm::PerformanceMode_Boost);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void TestNormalBoostPermutation(bool backgroundMode) NN_NOEXCEPT
{
    std::vector<Action> actionPermutation;

    actionPermutation.push_back(Action_AcOkOn);
    actionPermutation.push_back(Action_ChargerTypeEnoughPower);
    actionPermutation.push_back(Action_BatteryVoltageStateGood);
    actionPermutation.push_back(Action_RequestBoostMode);

    /* 順列確認用
    do
    {
        for ( int index = 0; index < actionPermutation.size(); index ++ )
        {
            NN_LOG("%d,", actionPermutation[index]);
        }
        NN_LOG("\n");
    }
    while ( next_permutation(actionPermutation.begin(), actionPermutation.end()) );
    */

    do
    {
        nnt::bpc::SetAcOk(false);
        nnt::psm::SetChargerType(nn::psm::ChargerType_Unconnected);
        nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_BoostPerformanceModeProhibited);
        nn::apm::RequestPerformanceMode(nn::apm::PerformanceMode_Normal);

        ConfigInfo config = ConfigNormal;

        EXPECT_EQ(config.mode,   nn::apm::GetPerformanceMode());
        EXPECT_EQ(config.config, nn::apm::GetPerformanceConfiguration(nn::apm::GetPerformanceMode()));

        nn::os::SystemEventType performanceEvent;
        nn::apm::GetPerformanceEvent(&performanceEvent, nn::apm::EventTarget_PerformanceModeChanged, nn::os::EventClearMode_ManualClear);
        nn::os::ClearSystemEvent(&performanceEvent);

        if ( backgroundMode )
        {
            nnt::apm::server::CheckFrequencies(ConfigBackground.cpuSetting, ConfigBackground.gpuSetting, ConfigBackground.emcSetting);
        }
        else
        {
            nnt::apm::server::CheckFrequencies(config.cpuSetting, config.gpuSetting, config.emcSetting);
        }

        for ( int index = 0; index < actionPermutation.size(); index ++ )
        {
            ConfigInfo nextConfig = ConfigNormal;

            // SIGLO-82255: ACOK_R の割り込みハンドリングを除去したため最後の操作が AcOkOn の場合は Boost モードに遷移しないのが期待値
            if ( (index == (actionPermutation.size() - 1)) && (actionPermutation[index] != Action_AcOkOn) )
            {
                nextConfig = ConfigBoost;
            }

            HandleAction(actionPermutation[index]);

            if ( nextConfig.mode != config.mode )
            {
                EXPECT_EQ(true, nn::os::TimedWaitSystemEvent(&performanceEvent, nn::TimeSpan::FromMilliSeconds(10)));
                nn::os::ClearSystemEvent(&performanceEvent);
            }

            config = nextConfig;

            EXPECT_EQ(config.mode,   nn::apm::GetPerformanceMode());
            EXPECT_EQ(config.config, nn::apm::GetPerformanceConfiguration(nn::apm::GetPerformanceMode()));

            if ( backgroundMode )
            {
                nnt::apm::server::CheckFrequencies(ConfigBackground.cpuSetting, ConfigBackground.gpuSetting, ConfigBackground.emcSetting);
            }
            else
            {
                nnt::apm::server::CheckFrequencies(config.cpuSetting, config.gpuSetting, config.emcSetting);
            }
        }

        nn::os::DestroySystemEvent(&performanceEvent);
    }
    while ( next_permutation(actionPermutation.begin(), actionPermutation.end()) );
}

// ACOK, BatteryVoltage, PowerSupply, Request に応じて FGM の設定値をノーマルモード、ブーストモードのデフォルト値へ変更する事を確認する。
// Normal から Boost への遷移を要因を並べ替えて試行する。
TEST_F(PerformanceModeTransitionTest, FullAwakeTestNormalBoostPermutation)
{
    nn::psc::PmFlagSet flags;
    flags.Reset();

    nn::psc::PmModule* pPmModule;
    nn::ppc::GetPmModuleForTest(&pPmModule);
    pPmModule->Dispatch(nn::psc::PmState_FullAwake, flags);

    TestNormalBoostPermutation(false);
}

// ACOK, BatteryVoltage, PowerSupply, Request がどのような時でも MinimumAwake 状態では固定の性能を使用することを確認する。
// Normal から Boost への遷移を要因を並べ替えて試行する。
TEST_F(PerformanceModeTransitionTest, MinimumAwakeTestNormalBoostPermutation)
{
    nn::psc::PmFlagSet flags;
    flags.Reset();

    nn::psc::PmModule* pPmModule;
    nn::ppc::GetPmModuleForTest(&pPmModule);
    pPmModule->Dispatch(nn::psc::PmState_MinimumAwake, flags);

    TestNormalBoostPermutation(true);
}

void TestBoostNormalPermutation(bool backgroundMode) NN_NOEXCEPT
{
    std::vector<Action> actionPermutation;

    actionPermutation.push_back(Action_AcOkOff);
    actionPermutation.push_back(Action_ChargerTypeUnconnected);
    //actionPermutation.push_back(Action_ChargerTypeLowPower);
    //actionPermutation.push_back(Action_ChargerTypeNotSupported);
    //actionPermutation.push_back(Action_BatteryVoltageStateSleepRequired);
    //actionPermutation.push_back(Action_BatteryVoltageStateShutdownRequired);
    actionPermutation.push_back(Action_BatteryVoltageStateBoostPerformanceModeProhibited);
    actionPermutation.push_back(Action_RequestNormalMode);

    do
    {
        nnt::bpc::SetAcOk(true);
        nnt::psm::SetChargerType(nn::psm::ChargerType_EnoughPower);
        nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_Good);
        nn::apm::RequestPerformanceMode(nn::apm::PerformanceMode_Boost);

        ConfigInfo config = ConfigBoost;

        EXPECT_EQ(config.mode,   nn::apm::GetPerformanceMode());
        EXPECT_EQ(config.config, nn::apm::GetPerformanceConfiguration(nn::apm::GetPerformanceMode()));

        nn::os::SystemEventType performanceEvent;
        nn::apm::GetPerformanceEvent(&performanceEvent, nn::apm::EventTarget_PerformanceModeChanged, nn::os::EventClearMode_ManualClear);
        nn::os::ClearSystemEvent(&performanceEvent);

        if ( backgroundMode )
        {
            nnt::apm::server::CheckFrequencies(ConfigBackground.cpuSetting, ConfigBackground.gpuSetting, ConfigBackground.emcSetting);
        }
        else
        {
            nnt::apm::server::CheckFrequencies(config.cpuSetting, config.gpuSetting, config.emcSetting);
        }

        for ( int index = 0; index < actionPermutation.size(); index ++ )
        {
            ConfigInfo nextConfig = ConfigBoost;

            // soctherm の割り込みで十分なので AcOkOff では APM の状態は更新されない。
            if ( (index != 0) || (actionPermutation[index] != Action_AcOkOff) )
            {
                nextConfig = ConfigNormal;
            }

            HandleAction(actionPermutation[index]);

            if ( nextConfig.mode != config.mode )
            {
                EXPECT_EQ(true, nn::os::TimedWaitSystemEvent(&performanceEvent, nn::TimeSpan::FromMilliSeconds(10)));
                nn::os::ClearSystemEvent(&performanceEvent);
            }

            config = nextConfig;

            EXPECT_EQ(config.mode,   nn::apm::GetPerformanceMode());
            EXPECT_EQ(config.config, nn::apm::GetPerformanceConfiguration(nn::apm::GetPerformanceMode()));

            if ( backgroundMode )
            {
                nnt::apm::server::CheckFrequencies(ConfigBackground.cpuSetting, ConfigBackground.gpuSetting, ConfigBackground.emcSetting);
            }
            else
            {
                nnt::apm::server::CheckFrequencies(config.cpuSetting, config.gpuSetting, config.emcSetting);
            }
        }

        nn::os::DestroySystemEvent(&performanceEvent);
    }
    while ( next_permutation(actionPermutation.begin(), actionPermutation.end()) );
}

// ACOK, BatteryVoltage, PowerSupply, Request に応じて FGM の設定値をノーマルモード、ブーストモードのデフォルト値へ変更する事を確認する。
// Boost から Normal への遷移を要因を並べ替えて試行する。
TEST_F(PerformanceModeTransitionTest, FullAwakeTestBoostNormalPermutation)
{
    nn::psc::PmFlagSet flags;
    flags.Reset();

    nn::psc::PmModule* pPmModule;
    nn::ppc::GetPmModuleForTest(&pPmModule);
    pPmModule->Dispatch(nn::psc::PmState_FullAwake, flags);

    TestBoostNormalPermutation(false);
}

// ACOK, BatteryVoltage, PowerSupply, Request がどのような時でも MinimumAwake 状態では固定の性能を使用することを確認する。
// Boost から Normal への遷移を要因を並べ替えて試行する。
TEST_F(PerformanceModeTransitionTest, MinimumAwakeTestBoostNormalPermutation)
{
    nn::psc::PmFlagSet flags;
    flags.Reset();

    nn::psc::PmModule* pPmModule;
    nn::ppc::GetPmModuleForTest(&pPmModule);
    pPmModule->Dispatch(nn::psc::PmState_MinimumAwake, flags);

    TestBoostNormalPermutation(true);
}

// ACOK, BatteryVoltage, PowerSupply, Request に応じて FGM の設定値をノーマルモード、ブーストモードのデフォルト値へ変更する事を確認する。
TEST_P(PerformanceModeTransitionTest, FullAwakeTest)
{
    nn::psc::PmFlagSet flags;
    flags.Reset();

    nn::psc::PmModule* pPmModule;
    nn::ppc::GetPmModuleForTest(&pPmModule);
    pPmModule->Dispatch(nn::psc::PmState_FullAwake, flags);

    nnt::bpc::SetAcOk(false);
    nnt::psm::SetChargerType(nn::psm::ChargerType_Unconnected);
    nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_BoostPerformanceModeProhibited);
    nn::apm::RequestPerformanceMode(nn::apm::PerformanceMode_Normal);

    const StepInfo* pSteps = GetParam().pSteps;

    for ( int index = 0; index < GetParam().size; index ++ )
    {
        HandleAction((pSteps + index)->action);

        EXPECT_EQ((pSteps + index)->expected.mode,   nn::apm::GetPerformanceMode());
        EXPECT_EQ((pSteps + index)->expected.config, nn::apm::GetPerformanceConfiguration(nn::apm::GetPerformanceMode()));

        nnt::apm::server::CheckFrequencies((pSteps + index)->expected.cpuSetting, (pSteps + index)->expected.gpuSetting, (pSteps + index)->expected.emcSetting);
    }
}

// ACOK, BatteryVoltage, PowerSupply, Request の条件がどのような時でも MinimumAwake 状態では固定の性能を使用することを確認する。
TEST_P(PerformanceModeTransitionTest, MinimumAwakeTest)
{
    nn::psc::PmFlagSet flags;
    flags.Reset();

    nn::psc::PmModule* pPmModule;
    nn::ppc::GetPmModuleForTest(&pPmModule);
    pPmModule->Dispatch(nn::psc::PmState_MinimumAwake, flags);

    nnt::bpc::SetAcOk(false);
    nnt::psm::SetChargerType(nn::psm::ChargerType_Unconnected);
    nnt::psm::SetBatteryVoltageState(nn::psm::BatteryVoltageState_BoostPerformanceModeProhibited);
    nn::apm::RequestPerformanceMode(nn::apm::PerformanceMode_Normal);

    const StepInfo* pSteps = GetParam().pSteps;

    for ( int index = 0; index < GetParam().size; index ++ )
    {
        HandleAction((pSteps + index)->action);

        EXPECT_EQ((pSteps + index)->expected.mode,   nn::apm::GetPerformanceMode());
        EXPECT_EQ((pSteps + index)->expected.config, nn::apm::GetPerformanceConfiguration(nn::apm::GetPerformanceMode()));

        nnt::apm::server::CheckFrequencies(ConfigBackground.cpuSetting, ConfigBackground.gpuSetting, ConfigBackground.emcSetting);
    }
}

}} // namespace nnt::apm
