﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/apm/apm.h>
#include <nn/apm/apm_System.h>
#include <nn/gpio/gpio.h>
#include <nn/os.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>

namespace nnt { namespace apm {

namespace {

class ApmEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp() override
    {
        nn::apm::Initialize();
        nn::apm::InitializeForSystem();
    }

    virtual void TearDown() override
    {
        nn::apm::Finalize();
        nn::apm::FinalizeForSystem();
    }
};

} // namespace

TEST(GetThrottlingStateTest, CheckThrottling)
{
    // 今は余裕を見て 1000 ms としておく。
    const int64_t ThrottlingDurationLimitNanoSeconds = 1000000000LL;
    nn::apm::ThrottlingState throttlingState;

    nn::apm::GetLastThrottlingState(&throttlingState);

    NN_LOG("Throttling duration is %d micro seconds.\n", throttlingState.durationNanoSeconds / 1000);
    EXPECT_GT(ThrottlingDurationLimitNanoSeconds, throttlingState.durationNanoSeconds);
    EXPECT_EQ(0x40,  throttlingState.throttlingRateCpuDividend);
    EXPECT_EQ(0x100, throttlingState.throttlingRateCpuDivisor);
    EXPECT_EQ(1,     throttlingState.throttlingRateGpuDividend);
    EXPECT_EQ(4,     throttlingState.throttlingRateGpuDivisor);

    EXPECT_EQ(0x00000001, throttlingState.throttleStatus & 0x00001001);
    EXPECT_EQ(0x0003fff1, throttlingState.cpuPSkipStatus);
    EXPECT_EQ(0x00300001, throttlingState.gpuPSkipStatus);

    EXPECT_TRUE(throttlingState.enabled);
    EXPECT_TRUE(throttlingState.invoked);

    nn::apm::ClearLastThrottlingState();
}

}} // namespace nnt::apm

extern "C" void nninitStartup()
{
    ::testing::AddGlobalTestEnvironment(new nnt::apm::ApmEnvironment);
}
