﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>

#include <nn/dd/dd_IoMappingAddress.h>

#include <nne/soctherm/soctherm.h>

#include "soctherm_Spec.h"
#include "SocThermAccessor.h"

namespace nnt { namespace soctherm {

namespace {

const uint32_t NneSocthermRegBase = 0x700E2000;
const uint32_t NneSocthermRegRange = 0x00001000;

const uintptr_t NneSocthermVirtualBaseAddress =
    nn::dd::QueryIoMappingAddress(
    NneSocthermRegBase, NneSocthermRegRange);

const uint32_t EdpOcAlarmConfigRegOffset[nne::soctherm::OcAlarm_Size] =
{
    nne::soctherm::detail::RegisterMap::Edp::Oc::Alarm::Oc1::Cfg::Offset,
    nne::soctherm::detail::RegisterMap::Edp::Oc::Alarm::Oc2::Cfg::Offset,
    nne::soctherm::detail::RegisterMap::Edp::Oc::Alarm::Oc3::Cfg::Offset,
    nne::soctherm::detail::RegisterMap::Edp::Oc::Alarm::Oc4::Cfg::Offset,
    nne::soctherm::detail::RegisterMap::Edp::Oc::Alarm::Oc5::Cfg::Offset,
};

uintptr_t GetNneSocthermVirtualAddress(int32_t offset) NN_NOEXCEPT
{
    return NneSocthermVirtualBaseAddress + offset;
}

uint32_t ReadReg(uint32_t offset) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(offset <= NneSocthermRegRange - 4,
        "0x%08x is not a valid Soctherm register space.\n",
        NneSocthermRegBase + offset);
    uintptr_t virtualAddress = GetNneSocthermVirtualAddress(offset);
    return *reinterpret_cast<uint32_t*>(virtualAddress);
}

void WriteReg(uint32_t offset, uint32_t value) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(offset <= NneSocthermRegRange - 4,
        "0x%08x is not a valid Soctherm register space.\n",
        NneSocthermRegBase + offset);
    uintptr_t virtualAddress = GetNneSocthermVirtualAddress(offset);
    *reinterpret_cast<uint32_t*>(virtualAddress) = value;
    *reinterpret_cast<volatile uint32_t*>(virtualAddress);
}

void UpdateReg(uint32_t offset, uint32_t value, uint32_t mask) NN_NOEXCEPT
{
    uint32_t regVal = ReadReg(offset);
    regVal &= ~mask;
    regVal |= (value & mask);
    WriteReg(offset, regVal);
}

} // namespace

void SetOcAlarmPolarity(nne::soctherm::OcAlarm alarm, bool polarity) NN_NOEXCEPT
{
    if ( polarity )
    {
        UpdateReg(EdpOcAlarmConfigRegOffset[alarm],
            nne::soctherm::detail::RegisterMap::Edp::Oc::Alarm::Oc1::Cfg::AlarmPolarity_AssertHigh,
            nne::soctherm::detail::RegisterMap::Edp::Oc::Alarm::Oc1::Cfg::Mask_AlarmPolarity);
    }
    else
    {
        UpdateReg(EdpOcAlarmConfigRegOffset[alarm],
            nne::soctherm::detail::RegisterMap::Edp::Oc::Alarm::Oc1::Cfg::AlarmPolarity_AssertLow,
            nne::soctherm::detail::RegisterMap::Edp::Oc::Alarm::Oc1::Cfg::Mask_AlarmPolarity);
    }
}

}} // namespace nnt::soctherm
