﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/nn_Common.h>
#include <nnt/nntest.h>

#include "../Common/IsValidPowerDomain.h"

class SetMinVClockRateTest : public ::testing::TestWithParam<nn::pcv::Module>{};

const nn::pcv::Module clockRateModules[] =
    {
        // Insert safe module to test with
        nn::pcv::Module_Gpu,
    };

INSTANTIATE_TEST_CASE_P(SetClockRateModules,
    SetMinVClockRateTest,
    ::testing::ValuesIn(clockRateModules));

// SetMinVClockRate has not been implemented yet
TEST_P(SetMinVClockRateTest, SupportedModule)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;
    nn::pcv::Module module = GetParam();

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    ASSERT_TRUE(queryResult.IsSuccess());
    ASSERT_EQ(nn::pcv::ClockRatesListType_Discrete, listType);

    // Get intial clock rate
    nn::pcv::ClockHz initialClockRate = 0;
    nn::Result getClockRateResult = nn::pcv::GetClockRate(&initialClockRate, module);
    EXPECT_TRUE(getClockRateResult.IsSuccess());

    nn::Result result;

    // Check Voltage optimizes with SetClockRate
    result = nn::pcv::SetClockRate(module, rates[numRates - 1]);
    EXPECT_TRUE(result.IsSuccess());

    nn::pcv::PowerDomain gpuPowerDomain = nnt::pcv::GetProperGpuPowerDomain();

    nn::pcv::MicroVolt highVoltage = 0;
    result = nn::pcv::GetVoltageValue(&highVoltage, gpuPowerDomain);
    EXPECT_TRUE(result.IsSuccess());

    result = nn::pcv::SetClockRate(module, rates[0]);
    EXPECT_TRUE(result.IsSuccess());

    nn::pcv::MicroVolt lowVoltage = 0;
    result = nn::pcv::GetVoltageValue(&lowVoltage, gpuPowerDomain);
    EXPECT_TRUE(result.IsSuccess());

    ASSERT_GT(highVoltage, lowVoltage);

    // Check SetMinVClockRate behavior
    // Set back to maximum voltage via SetClockRate
    result = nn::pcv::SetClockRate(module, rates[numRates - 1]);
    EXPECT_TRUE(result.IsSuccess());

    result = nn::pcv::SetMinVClockRate(module, rates[numRates - 1]);
    EXPECT_TRUE(result.IsSuccess());

    nn::pcv::MicroVolt minMicroVolt = 0;
    result = nn::pcv::GetVoltageValue(&minMicroVolt, gpuPowerDomain);
    EXPECT_TRUE(result.IsSuccess());

    result = nn::pcv::SetClockRate(module, rates[0]);
    EXPECT_TRUE(result.IsSuccess());

    nn::pcv::MicroVolt microVolt = 0;
    result = nn::pcv::GetVoltageValue(&microVolt, gpuPowerDomain);
    EXPECT_TRUE(result.IsSuccess());

    ASSERT_GE(microVolt, minMicroVolt);

    // Restore initial clock rate
    result = nn::pcv::SetMinVClockRate(module, rates[0]);
    EXPECT_TRUE(result.IsSuccess());
    result = nn::pcv::SetClockRate(module, initialClockRate);
    ASSERT_TRUE(result.IsSuccess());
}

TEST_F(SetMinVClockRateTest, InvalidModule)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::Result result;
    result = nn::pcv::SetMinVClockRate(nn::pcv::Module_NumModule, 50 * 1000 * 1000);
    EXPECT_FALSE(result.IsSuccess());
    NN_LOG("Module: %3d\n", result.GetModule());
    NN_LOG("Description: %3d\n", result.GetDescription());
}
