﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include "../Common/ModuleNames.h"
#include "../Common/ModuleInfo.h"

using namespace nnt::pcv;

namespace {
    const nnt::pcv::ModuleInfo clockEnabledModules[] =
    {
        { nn::pcv::Module_AudioDsp,         Discrete,         Ceiling },
        { nn::pcv::Module_Spi1,             NonDiscrete,      Floor },
        { nn::pcv::Module_Cec,              Discrete,         SetNotSupported },
    };

    class SetClockEnabledTest : public ::testing::TestWithParam<nnt::pcv::ModuleInfo>{};

    INSTANTIATE_TEST_CASE_P(clockEnabledModules, SetClockEnabledTest, testing::ValuesIn(clockEnabledModules));
}

TEST_P(SetClockEnabledTest, OnOff)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());
    nn::pcv::ModuleState modState = {0};

    nn::pcv::Module module = GetParam().name;

    // Intial State
    nn::pcv::ModuleState initialState = {0};
    nn::Result getInitStateResult = nn::pcv::GetState(&initialState, module);
    EXPECT_TRUE(getInitStateResult.IsSuccess());
    NN_LOG("Module Initial State: %s\n", GetModuleName(module));
    NN_LOG("\tReset        : %s\n", initialState.resetAsserted ? "TRUE" : "FALSE");
    NN_LOG("\tPower enabled: %s\n", initialState.powerEnabled ? "TRUE" : "FALSE");
    NN_LOG("\tClock enabled: %s\n", initialState.clockEnabled ? "TRUE" : "FALSE");
    NN_LOG("\tClock frequency: %.2f MHz\n\n", initialState.clockFrequency / (1000.0 * 1000.0));

    if (initialState.clockEnabled)
    {
        // Turn clock off
        nn::Result disableClockResult = nn::pcv::SetClockEnabled(module, false);
        EXPECT_TRUE(disableClockResult.IsSuccess());

        // Validate clock is off
        nn::Result validateClockResult = nn::pcv::GetState(&modState, module);
        EXPECT_TRUE(validateClockResult.IsSuccess());
        EXPECT_FALSE(modState.clockEnabled);

        NN_LOG("Module OFF: %s\n", GetModuleName(module));
        NN_LOG("\tReset        : %s\n", modState.resetAsserted ? "TRUE" : "FALSE");
        NN_LOG("\tPower enabled: %s\n", modState.powerEnabled ? "TRUE" : "FALSE");
        NN_LOG("\tClock enabled: %s\n", modState.clockEnabled ? "TRUE" : "FALSE");
        NN_LOG("\tClock frequency: %.2f MHz\n\n", modState.clockFrequency / (1000.0 * 1000.0));
    }
    else
    {
        // Turn clock on
        nn::Result enableClockResult = nn::pcv::SetClockEnabled(module, true);
        EXPECT_TRUE(enableClockResult.IsSuccess());

        // Vaildate clock is on
        nn::Result validateClockResult = nn::pcv::GetState(&modState, module);
        EXPECT_TRUE(validateClockResult.IsSuccess());
        EXPECT_TRUE(modState.clockEnabled);

        NN_LOG("Module ON: %s\n", GetModuleName(module));
        NN_LOG("\tReset        : %s\n", modState.resetAsserted ? "TRUE" : "FALSE");
        NN_LOG("\tPower enabled: %s\n", modState.powerEnabled ? "TRUE" : "FALSE");
        NN_LOG("\tClock enabled: %s\n", modState.clockEnabled ? "TRUE" : "FALSE");
        NN_LOG("\tClock frequency: %.2f MHz\n\n", modState.clockFrequency / (1000.0 * 1000.0));
    }

    // Return to initial state
    nn::Result restoreInitResult = nn::pcv::SetClockEnabled(module, initialState.clockEnabled);
    EXPECT_TRUE(restoreInitResult.IsSuccess());

    // Validate initial state
    restoreInitResult = nn::pcv::GetState(&modState, module);
    EXPECT_TRUE(restoreInitResult.IsSuccess());
    EXPECT_EQ(initialState.resetAsserted, modState.resetAsserted);
    EXPECT_EQ(initialState.clockEnabled, modState.clockEnabled);
    EXPECT_EQ(initialState.powerEnabled, modState.powerEnabled);
    EXPECT_EQ(initialState.clockFrequency, modState.clockFrequency);

    NN_LOG("Module ON: %s\n", GetModuleName(module));
    NN_LOG("\tReset        : %s\n", modState.resetAsserted ? "TRUE" : "FALSE");
    NN_LOG("\tPower enabled: %s\n", modState.powerEnabled ? "TRUE" : "FALSE");
    NN_LOG("\tClock enabled: %s\n", modState.clockEnabled ? "TRUE" : "FALSE");
    NN_LOG("\tClock frequency: %.2f MHz\n\n", modState.clockFrequency / (1000.0 * 1000.0));

}

TEST_F(SetClockEnabledTest, InvalidModule)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());

    nn::Result invalidModuleResult = nn::pcv::SetClockEnabled(nn::pcv::Module_NumModule, true);
    ASSERT_FALSE(invalidModuleResult.IsSuccess());
}

//Test Enable -> Enable -> Disable
